/*
 * Decompiled with CFR 0.152.
 */
package com.datastax.driver.core;

import com.datastax.driver.core.Assertions;
import com.datastax.driver.core.CodecRegistry;
import com.datastax.driver.core.DataType;
import com.datastax.driver.core.LocalDate;
import com.datastax.driver.core.ProtocolVersion;
import com.datastax.driver.core.TestUtils;
import com.datastax.driver.core.TupleType;
import com.datastax.driver.core.TupleValue;
import com.datastax.driver.core.TypeCodec;
import com.datastax.driver.core.UDTValue;
import com.datastax.driver.core.UserType;
import com.datastax.driver.core.exceptions.InvalidTypeException;
import com.datastax.driver.core.utils.Bytes;
import com.google.common.collect.Lists;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import org.assertj.core.api.AbstractCharSequenceAssert;
import org.assertj.core.api.AbstractObjectAssert;
import org.testng.Assert;
import org.testng.annotations.Test;

public class DataTypeTest {
    CodecRegistry codecRegistry = new CodecRegistry();
    ProtocolVersion protocolVersion = ProtocolVersion.NEWEST_SUPPORTED;

    static boolean exclude(DataType t) {
        return t.getName() == DataType.Name.COUNTER || t.getName() == DataType.Name.DURATION;
    }

    private static TestValue[] primitiveTestValues(DataType dt) {
        switch (dt.getName()) {
            case ASCII: 
            case TEXT: 
            case VARCHAR: {
                return new TestValue[]{new TestValue("foo", "'foo'", "'foo'"), new TestValue("fo'o", "'fo''o'", "'fo''o'"), new TestValue(null, null, "NULL"), new TestValue(null, "null", "NULL"), new TestValue(null, "NULL", "NULL")};
            }
            case BIGINT: {
                return new TestValue[]{new TestValue(42L, "42", "42"), new TestValue(91294377723L, "91294377723", "91294377723"), new TestValue(-133L, "-133", "-133"), new TestValue(null, null, "NULL"), new TestValue(null, "null", "NULL"), new TestValue(null, "NULL", "NULL")};
            }
            case TIMESTAMP: {
                return new TestValue[]{new TestValue(new Date(42L), "42", "42"), new TestValue(new Date(91294377723L), "91294377723", "91294377723"), new TestValue(new Date(-133L), "-133", "-133"), new TestValue(new Date(784041330999L), "'1994-11-05T14:15:30.999+0100'", "784041330999"), new TestValue(null, null, "NULL"), new TestValue(null, "null", "NULL"), new TestValue(null, "NULL", "NULL")};
            }
            case DATE: {
                return new TestValue[]{new TestValue(LocalDate.fromDaysSinceEpoch((int)16071), "'2014-01-01'", "'2014-01-01'"), new TestValue(LocalDate.fromDaysSinceEpoch((int)0), "'1970-01-01'", "'1970-01-01'"), new TestValue(LocalDate.fromDaysSinceEpoch((int)0), "'2147483648'", "'1970-01-01'"), new TestValue(LocalDate.fromDaysSinceEpoch((int)Integer.MIN_VALUE), "0", "'-5877641-06-23'"), new TestValue(null, null, "NULL"), new TestValue(null, "null", "NULL"), new TestValue(null, "NULL", "NULL")};
            }
            case TIME: {
                return new TestValue[]{new TestValue(54012123450000L, "'54012123450000'", "'15:00:12.123450000'"), new TestValue(0L, "'0'", "'00:00:00.000000000'"), new TestValue(54012012345000L, "'15:00:12.012345000'", "'15:00:12.012345000'"), new TestValue(null, null, "NULL"), new TestValue(null, "null", "NULL"), new TestValue(null, "NULL", "NULL")};
            }
            case BLOB: {
                return new TestValue[]{new TestValue(Bytes.fromHexString((String)"0x2450"), "0x2450", "0x2450"), new TestValue(ByteBuffer.allocate(0), "0x", "0x"), new TestValue(null, null, "NULL"), new TestValue(null, "null", "NULL"), new TestValue(null, "NULL", "NULL")};
            }
            case BOOLEAN: {
                return new TestValue[]{new TestValue(true, "true", "true"), new TestValue(false, "false", "false"), new TestValue(null, null, "NULL"), new TestValue(null, "null", "NULL"), new TestValue(null, "NULL", "NULL")};
            }
            case DECIMAL: {
                return new TestValue[]{new TestValue(new BigDecimal("1.23E+8"), "1.23E+8", "1.23E+8"), new TestValue(null, null, "NULL"), new TestValue(null, "null", "NULL"), new TestValue(null, "NULL", "NULL")};
            }
            case DOUBLE: {
                return new TestValue[]{new TestValue(2.39324324, "2.39324324", "2.39324324"), new TestValue(-12.0, "-12.0", "-12.0"), new TestValue(null, null, "NULL"), new TestValue(null, "null", "NULL"), new TestValue(null, "NULL", "NULL")};
            }
            case FLOAT: {
                return new TestValue[]{new TestValue(Float.valueOf(2.39f), "2.39", "2.39"), new TestValue(Float.valueOf(-12.0f), "-12.0", "-12.0"), new TestValue(null, null, "NULL"), new TestValue(null, "null", "NULL"), new TestValue(null, "NULL", "NULL")};
            }
            case INET: {
                try {
                    return new TestValue[]{new TestValue(InetAddress.getByName("128.2.12.3"), "'128.2.12.3'", "'128.2.12.3'"), new TestValue(null, null, "NULL"), new TestValue(null, "null", "NULL"), new TestValue(null, "NULL", "NULL")};
                }
                catch (UnknownHostException e) {
                    throw new RuntimeException();
                }
            }
            case TINYINT: {
                return new TestValue[]{new TestValue((byte)-4, "-4", "-4"), new TestValue((byte)44, "44", "44"), new TestValue(null, null, "NULL"), new TestValue(null, "null", "NULL"), new TestValue(null, "NULL", "NULL")};
            }
            case SMALLINT: {
                return new TestValue[]{new TestValue((short)-3, "-3", "-3"), new TestValue((short)43, "43", "43"), new TestValue(null, null, "NULL"), new TestValue(null, "null", "NULL"), new TestValue(null, "NULL", "NULL")};
            }
            case INT: {
                return new TestValue[]{new TestValue(-2, "-2", "-2"), new TestValue(42, "42", "42"), new TestValue(null, null, "NULL"), new TestValue(null, "null", "NULL"), new TestValue(null, "NULL", "NULL")};
            }
            case TIMEUUID: {
                return new TestValue[]{new TestValue(UUID.fromString("FE2B4360-28C6-11E2-81C1-0800200C9A66"), "fe2b4360-28c6-11e2-81c1-0800200c9a66", "fe2b4360-28c6-11e2-81c1-0800200c9a66"), new TestValue(null, null, "NULL"), new TestValue(null, "null", "NULL"), new TestValue(null, "NULL", "NULL")};
            }
            case UUID: {
                return new TestValue[]{new TestValue(UUID.fromString("FE2B4360-28C6-11E2-81C1-0800200C9A66"), "fe2b4360-28c6-11e2-81c1-0800200c9a66", "fe2b4360-28c6-11e2-81c1-0800200c9a66"), new TestValue(UUID.fromString("067e6162-3b6f-4ae2-a171-2470b63dff00"), "067e6162-3b6f-4ae2-a171-2470b63dff00", "067e6162-3b6f-4ae2-a171-2470b63dff00"), new TestValue(null, null, "NULL"), new TestValue(null, "null", "NULL"), new TestValue(null, "NULL", "NULL")};
            }
            case VARINT: {
                return new TestValue[]{new TestValue(new BigInteger("12387290982347987032483422342432"), "12387290982347987032483422342432", "12387290982347987032483422342432"), new TestValue(null, null, "NULL"), new TestValue(null, "null", "NULL"), new TestValue(null, "NULL", "NULL")};
            }
        }
        throw new RuntimeException("Missing handling of " + dt);
    }

    @Test(groups={"unit"})
    public void parseNativeTest() {
        for (DataType dt : DataType.allPrimitiveTypes()) {
            if (DataTypeTest.exclude(dt)) continue;
            for (TestValue value : DataTypeTest.primitiveTestValues(dt)) {
                ((AbstractObjectAssert)Assertions.assertThat((Object)this.codecRegistry.codecFor(dt).parse(value.cqlInputString)).as("Parsing input %s to a %s", new Object[]{value.cqlInputString, dt})).isEqualTo(value.javaObject);
            }
        }
    }

    @Test(groups={"unit"})
    public void formatNativeTest() {
        for (DataType dt : DataType.allPrimitiveTypes()) {
            if (DataTypeTest.exclude(dt)) continue;
            for (TestValue value : DataTypeTest.primitiveTestValues(dt)) {
                ((AbstractCharSequenceAssert)Assertions.assertThat((String)this.codecRegistry.codecFor(dt).format(value.javaObject)).as("Formatting a %s expecting %s", new Object[]{dt, value.cqlOutputString})).isEqualTo((Object)value.cqlOutputString);
            }
        }
    }

    @Test(groups={"unit"})
    public void parseFormatListTest() {
        String toParse = "['Foo','Bar','Foo''bar']";
        List<String> toFormat = Arrays.asList("Foo", "Bar", "Foo'bar");
        DataType.CollectionType dt = DataType.list((DataType)DataType.text());
        Assert.assertEquals((Object)this.codecRegistry.codecFor((DataType)dt).parse(toParse), toFormat);
        Assert.assertEquals((String)this.codecRegistry.codecFor((DataType)dt).format(toFormat), (String)toParse);
    }

    @Test(groups={"unit"})
    public void parseFormatSetTest() {
        String toParse = "{'Foo','Bar','Foo''bar'}";
        LinkedHashSet<String> toFormat = new LinkedHashSet<String>(){
            {
                this.add("Foo");
                this.add("Bar");
                this.add("Foo'bar");
            }
        };
        DataType.CollectionType dt = DataType.set((DataType)DataType.text());
        Assert.assertEquals((Object)this.codecRegistry.codecFor((DataType)dt).parse(toParse), (Object)toFormat);
        Assert.assertEquals((String)this.codecRegistry.codecFor((DataType)dt).format((Object)toFormat), (String)toParse);
    }

    @Test(groups={"unit"})
    public void parseFormatMapTest() {
        String toParse = "{'Foo':3,'Bar':42,'Foo''bar':-24}";
        LinkedHashMap<String, Integer> toFormat = new LinkedHashMap<String, Integer>(){
            {
                this.put("Foo", 3);
                this.put("Bar", 42);
                this.put("Foo'bar", -24);
            }
        };
        DataType.CollectionType dt = DataType.map((DataType)DataType.text(), (DataType)DataType.cint());
        Assert.assertEquals((Object)this.codecRegistry.codecFor((DataType)dt).parse(toParse), (Object)toFormat);
        Assert.assertEquals((String)this.codecRegistry.codecFor((DataType)dt).format((Object)toFormat), (String)toParse);
    }

    @Test(groups={"unit"})
    public void parseFormatUDTTest() {
        String toParse = "{t:'fo''o',i:3,\"L\":['a','b'],s:{3:{a:0x01}}}";
        final UserType udt1 = new UserType("ks", "t", false, Arrays.asList(new UserType.Field("a", DataType.blob())), this.protocolVersion, this.codecRegistry);
        UserType udt2 = new UserType("ks", "t", false, Arrays.asList(new UserType.Field("t", DataType.text()), new UserType.Field("i", DataType.cint()), new UserType.Field("L", (DataType)DataType.list((DataType)DataType.text())), new UserType.Field("s", (DataType)DataType.map((DataType)DataType.cint(), (DataType)udt1))), this.protocolVersion, this.codecRegistry);
        UDTValue toFormat = udt2.newValue();
        toFormat.setString("t", "fo'o");
        toFormat.setInt("i", 3);
        toFormat.setList("\"L\"", Arrays.asList("a", "b"));
        toFormat.setMap("s", (Map)new HashMap<Integer, UDTValue>(){
            {
                this.put(3, udt1.newValue().setBytes("a", ByteBuffer.wrap(new byte[]{1})));
            }
        });
        Assert.assertEquals((Object)this.codecRegistry.codecFor((DataType)udt2).parse(toParse), (Object)toFormat);
        Assert.assertEquals((String)this.codecRegistry.codecFor((DataType)udt2).format((Object)toFormat), (String)toParse);
    }

    @Test(groups={"unit"})
    public void parseFormatTupleTest() {
        String toParse = "(1,'foo',1.0)";
        TupleType t = new TupleType((List)Lists.newArrayList((Object[])new DataType[]{DataType.cint(), DataType.text(), DataType.cfloat()}), this.protocolVersion, this.codecRegistry);
        TupleValue toFormat = t.newValue(new Object[]{1, "foo", Float.valueOf(1.0f)});
        Assert.assertEquals((Object)this.codecRegistry.codecFor((DataType)t).parse(toParse), (Object)toFormat);
        Assert.assertEquals((String)this.codecRegistry.codecFor((DataType)t).format((Object)toFormat), (String)toParse);
    }

    @Test(groups={"unit"})
    public void serializeDeserializeTest() {
        for (ProtocolVersion v : ProtocolVersion.values()) {
            this.serializeDeserializeTest(v);
        }
    }

    public void serializeDeserializeTest(ProtocolVersion version) {
        for (DataType dt : DataType.allPrimitiveTypes()) {
            if (DataTypeTest.exclude(dt)) continue;
            Object value = TestUtils.getFixedValue(dt);
            TypeCodec codec = this.codecRegistry.codecFor(dt);
            Assert.assertEquals((Object)codec.deserialize(codec.serialize(value, version), version), (Object)value);
        }
        TypeCodec codec = this.codecRegistry.codecFor(DataType.bigint());
        try {
            ByteBuffer badValue = ByteBuffer.allocate(4);
            codec.deserialize(badValue, version);
            Assert.fail((String)"This should not have worked");
        }
        catch (InvalidTypeException invalidTypeException) {
            // empty catch block
        }
    }

    @Test(groups={"unit"})
    public void serializeDeserializeCollectionsTest() {
        for (ProtocolVersion v : ProtocolVersion.values()) {
            this.serializeDeserializeCollectionsTest(v);
        }
    }

    public void serializeDeserializeCollectionsTest(ProtocolVersion version) {
        List<String> l = Arrays.asList("foo", "bar");
        DataType.CollectionType dt = DataType.list((DataType)DataType.text());
        TypeCodec codec = this.codecRegistry.codecFor((DataType)dt);
        Assert.assertEquals((Collection)((Collection)codec.deserialize(codec.serialize(l, version), version)), l);
        try {
            DataType.CollectionType listOfBigint = DataType.list((DataType)DataType.bigint());
            codec = this.codecRegistry.codecFor((DataType)listOfBigint);
            codec.serialize(l, version);
            Assert.fail((String)"This should not have worked");
        }
        catch (InvalidTypeException invalidTypeException) {
            // empty catch block
        }
    }

    static class TestValue {
        final Object javaObject;
        final String cqlInputString;
        final String cqlOutputString;

        TestValue(Object javaObject, String cqlInputString, String cqlOutputString) {
            this.javaObject = javaObject;
            this.cqlInputString = cqlInputString;
            this.cqlOutputString = cqlOutputString;
        }
    }
}

