/*
 * Decompiled with CFR 0.152.
 */
package com.datastax.driver.core;

import com.datastax.driver.core.BoundStatement;
import com.datastax.driver.core.CCMTestsSupport;
import com.datastax.driver.core.CodecRegistry;
import com.datastax.driver.core.DataType;
import com.datastax.driver.core.Duration;
import com.datastax.driver.core.GettableByIndexData;
import com.datastax.driver.core.Host;
import com.datastax.driver.core.PreparedStatement;
import com.datastax.driver.core.PrimitiveTypeSamples;
import com.datastax.driver.core.ProtocolVersion;
import com.datastax.driver.core.Row;
import com.datastax.driver.core.SimpleStatement;
import com.datastax.driver.core.Statement;
import com.datastax.driver.core.TestUtils;
import com.datastax.driver.core.TypeCodec;
import com.datastax.driver.core.VersionNumber;
import com.datastax.driver.core.utils.CassandraVersion;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import java.lang.constant.Constable;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.concurrent.atomic.AtomicInteger;
import org.assertj.core.api.AbstractComparableAssert;
import org.assertj.core.api.AbstractObjectAssert;
import org.assertj.core.api.Assertions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.testng.annotations.Test;

public class DataTypeIntegrationTest
extends CCMTestsSupport {
    private static final Logger logger = LoggerFactory.getLogger(DataTypeIntegrationTest.class);
    private Map<DataType, Object> samples;
    private List<TestTable> tables;
    private VersionNumber cassandraVersion;

    @Override
    public void onTestContextInitialized() {
        ProtocolVersion protocolVersion = this.ccm().getProtocolVersion();
        this.samples = PrimitiveTypeSamples.samples(protocolVersion);
        this.tables = this.allTables();
        Host host = (Host)this.cluster().getMetadata().getAllHosts().iterator().next();
        this.cassandraVersion = host.getCassandraVersion().nextStable();
        ArrayList statements = Lists.newArrayList();
        for (TestTable table : this.tables) {
            if (this.cassandraVersion.compareTo(table.minCassandraVersion) < 0) {
                logger.debug("Skipping table because it uses a feature not supported by Cassandra {}: {}", (Object)this.cassandraVersion, (Object)table.createStatement);
                continue;
            }
            statements.add(table.createStatement);
        }
        this.execute(statements);
    }

    @Test(groups={"long"})
    public void should_insert_and_retrieve_data_with_legacy_statements() {
        this.should_insert_and_retrieve_data(StatementType.RAW_STRING);
    }

    @Test(groups={"long"})
    public void should_insert_and_retrieve_data_with_prepared_statements() {
        this.should_insert_and_retrieve_data(StatementType.PREPARED);
    }

    @Test(groups={"long"})
    @CassandraVersion(value="2.0", description="Uses parameterized simple statements, which are only available with protocol v2")
    public void should_insert_and_retrieve_data_with_parameterized_simple_statements() {
        this.should_insert_and_retrieve_data(StatementType.SIMPLE_WITH_PARAM);
    }

    protected void should_insert_and_retrieve_data(StatementType statementType) {
        ProtocolVersion protocolVersion = this.cluster().getConfiguration().getProtocolOptions().getProtocolVersion();
        CodecRegistry codecRegistry = this.cluster().getConfiguration().getCodecRegistry();
        for (TestTable table : this.tables) {
            if (this.cassandraVersion.compareTo(table.minCassandraVersion) < 0) continue;
            TypeCodec codec = codecRegistry.codecFor(table.testColumnType);
            switch (statementType) {
                case RAW_STRING: {
                    String formatValue = codec.format(table.sampleValue);
                    Assertions.assertThat((String)formatValue).isNotNull();
                    String query = table.insertStatement.replace("?", formatValue);
                    this.session().execute(query);
                    break;
                }
                case SIMPLE_WITH_PARAM: {
                    SimpleStatement statement = new SimpleStatement(table.insertStatement, new Object[]{table.sampleValue});
                    this.checkGetValuesReturnsSerializedValue(protocolVersion, statement, table);
                    this.session().execute((Statement)statement);
                    break;
                }
                case PREPARED: {
                    PreparedStatement ps = this.session().prepare(table.insertStatement);
                    BoundStatement bs = ps.bind(new Object[]{table.sampleValue});
                    this.checkGetterReturnsValue(bs, table);
                    this.session().execute((Statement)bs);
                }
            }
            Row row = this.session().execute(table.selectStatement).one();
            Object queriedValue = codec.deserialize(row.getBytesUnsafe("v"), protocolVersion);
            ((AbstractObjectAssert)Assertions.assertThat((Object)queriedValue).as("Test failure on %s statement with table:%n%s;%ninsert statement:%n%s;%n", new Object[]{statementType, table.createStatement, table.insertStatement})).isEqualTo(table.expectedValue);
            ((AbstractObjectAssert)Assertions.assertThat((Object)this.getValue((GettableByIndexData)row, table.testColumnType)).as("Test failure on %s statement with table:%n%s;%ninsert statement:%n%s;%n", new Object[]{statementType, table.createStatement, table.insertStatement})).isEqualTo(table.expectedPrimitiveValue);
            this.session().execute(table.truncateStatement);
        }
    }

    private void checkGetterReturnsValue(BoundStatement bs, TestTable table) {
        Object getterResult = this.getValue((GettableByIndexData)bs, table.testColumnType);
        ((AbstractObjectAssert)Assertions.assertThat((Object)getterResult).as("Expected values to match for " + table.testColumnType, new Object[0])).isEqualTo(table.expectedPrimitiveValue);
        ((AbstractObjectAssert)Assertions.assertThat((Object)bs.getObject(0)).as("Expected values to match for " + table.testColumnType, new Object[0])).isEqualTo(table.sampleValue);
        ((AbstractObjectAssert)Assertions.assertThat((Object)bs.getObject("v")).as("Expected values to match for " + table.testColumnType, new Object[0])).isEqualTo(table.sampleValue);
    }

    public void checkGetValuesReturnsSerializedValue(ProtocolVersion protocolVersion, SimpleStatement statement, TestTable table) {
        CodecRegistry codecRegistry = this.cluster().getConfiguration().getCodecRegistry();
        ByteBuffer[] values = statement.getValues(protocolVersion, codecRegistry);
        Assertions.assertThat((int)values.length).isEqualTo(1);
        ((AbstractComparableAssert)Assertions.assertThat((Comparable)values[0]).as("Value not serialized as expected for " + table.sampleValue, new Object[0])).isEqualTo((Object)codecRegistry.codecFor(table.testColumnType).serialize(table.sampleValue, protocolVersion));
    }

    private List<TestTable> allTables() {
        ArrayList tables = Lists.newArrayList();
        tables.addAll(this.tablesWithPrimitives());
        tables.addAll(this.tablesWithPrimitivesNull());
        tables.addAll(this.tablesWithCollectionsOfPrimitives());
        tables.addAll(this.tablesWithMapsOfPrimitives());
        tables.addAll(this.tablesWithNestedCollections());
        tables.addAll(this.tablesWithRandomlyGeneratedNestedCollections());
        return ImmutableList.copyOf((Collection)tables);
    }

    private List<TestTable> tablesWithPrimitives() {
        ArrayList tables = Lists.newArrayList();
        for (Map.Entry<DataType, Object> entry : this.samples.entrySet()) {
            tables.add(new TestTable(entry.getKey(), entry.getValue(), "1.2.0"));
        }
        return tables;
    }

    private List<TestTable> tablesWithPrimitivesNull() {
        ArrayList tables = Lists.newArrayList();
        block9: for (DataType dataType : TestUtils.allPrimitiveTypes(this.ccm().getProtocolVersion())) {
            Constable expectedPrimitiveValue = null;
            switch (dataType.getName()) {
                case BIGINT: 
                case TIME: {
                    expectedPrimitiveValue = 0L;
                    break;
                }
                case DOUBLE: {
                    expectedPrimitiveValue = 0.0;
                    break;
                }
                case FLOAT: {
                    expectedPrimitiveValue = Float.valueOf(0.0f);
                    break;
                }
                case INT: {
                    expectedPrimitiveValue = 0;
                    break;
                }
                case SMALLINT: {
                    expectedPrimitiveValue = (short)0;
                    break;
                }
                case TINYINT: {
                    expectedPrimitiveValue = (byte)0;
                    break;
                }
                case BOOLEAN: {
                    expectedPrimitiveValue = Boolean.valueOf(false);
                    break;
                }
                default: {
                    continue block9;
                }
            }
            tables.add(new TestTable(dataType, null, null, expectedPrimitiveValue, "1.2.0"));
        }
        return tables;
    }

    private List<TestTable> tablesWithCollectionsOfPrimitives() {
        ArrayList tables = Lists.newArrayList();
        for (Map.Entry<DataType, Object> entry : this.samples.entrySet()) {
            DataType elementType = entry.getKey();
            Object elementSample = entry.getValue();
            tables.add(new TestTable((DataType)DataType.list((DataType)elementType), Lists.newArrayList((Object[])new Object[]{elementSample, elementSample}), "1.2.0"));
            if (elementType == DataType.duration()) continue;
            tables.add(new TestTable((DataType)DataType.set((DataType)elementType), Sets.newHashSet((Object[])new Object[]{elementSample}), "1.2.0"));
        }
        return tables;
    }

    private List<TestTable> tablesWithMapsOfPrimitives() {
        ArrayList tables = Lists.newArrayList();
        for (Map.Entry<DataType, Object> keyEntry : this.samples.entrySet()) {
            DataType keyType = keyEntry.getKey();
            if (keyType == DataType.duration()) continue;
            Object keySample = keyEntry.getValue();
            for (Map.Entry<DataType, Object> valueEntry : this.samples.entrySet()) {
                DataType valueType = valueEntry.getKey();
                Object valueSample = valueEntry.getValue();
                tables.add(new TestTable((DataType)DataType.map((DataType)keyType, (DataType)valueType), ImmutableMap.builder().put(keySample, valueSample).build(), "1.2.0"));
            }
        }
        return tables;
    }

    private Collection<? extends TestTable> tablesWithNestedCollections() {
        ArrayList tables = Lists.newArrayList();
        ImmutableMap childCollectionSamples = ImmutableMap.builder().put((Object)DataType.frozenList((DataType)DataType.cint()), (Object)Lists.newArrayList((Object[])new Integer[]{1, 1})).put((Object)DataType.frozenSet((DataType)DataType.cint()), (Object)Sets.newHashSet((Object[])new Integer[]{1, 2})).put((Object)DataType.frozenMap((DataType)DataType.cint(), (DataType)DataType.cint()), (Object)ImmutableMap.builder().put((Object)1, (Object)2).put((Object)3, (Object)4).build()).build();
        for (Map.Entry entry : childCollectionSamples.entrySet()) {
            DataType elementType = (DataType)entry.getKey();
            Object elementSample = entry.getValue();
            tables.add(new TestTable((DataType)DataType.list((DataType)elementType), Lists.newArrayList((Object[])new Object[]{elementSample, elementSample}), "2.1.3"));
            tables.add(new TestTable((DataType)DataType.set((DataType)elementType), Sets.newHashSet((Object[])new Object[]{elementSample}), "2.1.3"));
            for (Map.Entry valueEntry : childCollectionSamples.entrySet()) {
                DataType valueType = (DataType)valueEntry.getKey();
                Object valueSample = valueEntry.getValue();
                tables.add(new TestTable((DataType)DataType.map((DataType)elementType, (DataType)valueType), ImmutableMap.builder().put(elementSample, valueSample).build(), "2.1.3"));
            }
        }
        return tables;
    }

    private Collection<? extends TestTable> tablesWithRandomlyGeneratedNestedCollections() {
        ArrayList tables = Lists.newArrayList();
        DataType nestedListType = this.buildNestedType(DataType.Name.LIST, 5);
        DataType nestedSetType = this.buildNestedType(DataType.Name.SET, 5);
        DataType nestedMapType = this.buildNestedType(DataType.Name.MAP, 5);
        tables.add(new TestTable(nestedListType, this.nestedObject(nestedListType), "2.1.3"));
        tables.add(new TestTable(nestedSetType, this.nestedObject(nestedSetType), "2.1.3"));
        tables.add(new TestTable(nestedMapType, this.nestedObject(nestedMapType), "2.1.3"));
        return tables;
    }

    public Object nestedObject(DataType type) {
        boolean isAtBottom;
        int typeIdx = type.getTypeArguments().size() > 1 ? 1 : 0;
        DataType argument = (DataType)type.getTypeArguments().get(typeIdx);
        boolean bl = isAtBottom = !(argument instanceof DataType.CollectionType);
        if (isAtBottom) {
            switch (type.getName()) {
                case LIST: {
                    return Lists.newArrayList((Object[])new Integer[]{1, 2, 3});
                }
                case SET: {
                    return Sets.newHashSet((Object[])new Integer[]{1, 2, 3});
                }
                case MAP: {
                    HashMap map = Maps.newHashMap();
                    map.put(1, 2);
                    map.put(3, 4);
                    map.put(5, 6);
                    return map;
                }
            }
        } else {
            switch (type.getName()) {
                case LIST: {
                    ArrayList l = Lists.newArrayListWithExpectedSize((int)2);
                    for (int i = 0; i < 5; ++i) {
                        l.add(this.nestedObject(argument));
                    }
                    return l;
                }
                case SET: {
                    HashSet s = Sets.newHashSet();
                    for (int i = 0; i < 5; ++i) {
                        s.add(this.nestedObject(argument));
                    }
                    return s;
                }
                case MAP: {
                    HashMap map = Maps.newHashMap();
                    for (int i = 0; i < 5; ++i) {
                        map.put(i, this.nestedObject(argument));
                    }
                    return map;
                }
            }
        }
        return null;
    }

    public DataType buildNestedType(DataType.Name baseType, int depth) {
        Random r = new Random();
        DataType.CollectionType t = null;
        for (int i = 1; i <= depth; ++i) {
            int chooser = r.nextInt(3);
            if (t == null) {
                if (chooser == 0) {
                    t = DataType.frozenList((DataType)DataType.cint());
                    continue;
                }
                if (chooser == 1) {
                    t = DataType.frozenSet((DataType)DataType.cint());
                    continue;
                }
                t = DataType.frozenMap((DataType)DataType.cint(), (DataType)DataType.cint());
                continue;
            }
            if (i == depth) {
                switch (baseType) {
                    case LIST: {
                        return DataType.list((DataType)t);
                    }
                    case SET: {
                        return DataType.set((DataType)t);
                    }
                    case MAP: {
                        return DataType.map((DataType)DataType.cint(), (DataType)t);
                    }
                }
                continue;
            }
            t = chooser == 0 ? DataType.frozenList((DataType)t) : (chooser == 1 ? DataType.frozenSet((DataType)t) : DataType.frozenMap((DataType)DataType.cint(), (DataType)t));
        }
        return null;
    }

    private Object getValue(GettableByIndexData data, DataType dataType) {
        CodecRegistry codecRegistry = this.cluster().getConfiguration().getCodecRegistry();
        switch (dataType.getName()) {
            case ASCII: {
                return data.getString(0);
            }
            case BIGINT: {
                return data.getLong(0);
            }
            case BLOB: {
                return data.getBytes(0);
            }
            case BOOLEAN: {
                return data.getBool(0);
            }
            case DECIMAL: {
                return data.getDecimal(0);
            }
            case DOUBLE: {
                return data.getDouble(0);
            }
            case FLOAT: {
                return Float.valueOf(data.getFloat(0));
            }
            case INET: {
                return data.getInet(0);
            }
            case TINYINT: {
                return data.getByte(0);
            }
            case SMALLINT: {
                return data.getShort(0);
            }
            case INT: {
                return data.getInt(0);
            }
            case TEXT: 
            case VARCHAR: {
                return data.getString(0);
            }
            case TIMESTAMP: {
                return data.getTimestamp(0);
            }
            case DATE: {
                return data.getDate(0);
            }
            case TIME: {
                return data.getTime(0);
            }
            case UUID: 
            case TIMEUUID: {
                return data.getUUID(0);
            }
            case VARINT: {
                return data.getVarint(0);
            }
            case LIST: {
                return data.getList(0, codecRegistry.codecFor((DataType)dataType.getTypeArguments().get(0)).getJavaType());
            }
            case SET: {
                return data.getSet(0, codecRegistry.codecFor((DataType)dataType.getTypeArguments().get(0)).getJavaType());
            }
            case MAP: {
                return data.getMap(0, codecRegistry.codecFor((DataType)dataType.getTypeArguments().get(0)).getJavaType(), codecRegistry.codecFor((DataType)dataType.getTypeArguments().get(1)).getJavaType());
            }
            case DURATION: {
                return data.get(0, Duration.class);
            }
        }
        Assertions.fail((String)("Unexpected type in bound statement test: " + dataType));
        return null;
    }

    static class TestTable {
        private static final AtomicInteger counter = new AtomicInteger();
        private String tableName = "date_type_test" + counter.incrementAndGet();
        final DataType testColumnType;
        final Object sampleValue;
        final Object expectedValue;
        final Object expectedPrimitiveValue;
        final String createStatement;
        final String insertStatement = String.format("INSERT INTO %s (k, v) VALUES (1, ?)", this.tableName);
        final String selectStatement = String.format("SELECT v FROM %s WHERE k = 1", this.tableName);
        final String truncateStatement = String.format("TRUNCATE %s", this.tableName);
        final VersionNumber minCassandraVersion;

        TestTable(DataType testColumnType, Object sampleValue, String minCassandraVersion) {
            this(testColumnType, sampleValue, sampleValue, minCassandraVersion);
        }

        TestTable(DataType testColumnType, Object sampleValue, Object expectedValue, String minCassandraVersion) {
            this(testColumnType, sampleValue, expectedValue, expectedValue, minCassandraVersion);
        }

        TestTable(DataType testColumnType, Object sampleValue, Object expectedValue, Object expectedPrimitiveValue, String minCassandraVersion) {
            this.testColumnType = testColumnType;
            this.sampleValue = sampleValue;
            this.expectedValue = expectedValue;
            this.expectedPrimitiveValue = expectedPrimitiveValue;
            this.minCassandraVersion = VersionNumber.parse((String)minCassandraVersion);
            this.createStatement = String.format("CREATE TABLE %s (k int PRIMARY KEY, v %s)", this.tableName, testColumnType);
        }
    }

    static enum StatementType {
        RAW_STRING,
        SIMPLE_WITH_PARAM,
        PREPARED;

    }
}

