/*
 * Decompiled with CFR 0.152.
 */
package com.datastax.driver.core;

import com.codahale.metrics.Gauge;
import com.datastax.driver.core.Assertions;
import com.datastax.driver.core.Cluster;
import com.datastax.driver.core.ConditionChecker;
import com.datastax.driver.core.Connection;
import com.datastax.driver.core.GuavaCompatibility;
import com.datastax.driver.core.Host;
import com.datastax.driver.core.HostConnectionPool;
import com.datastax.driver.core.HostDistance;
import com.datastax.driver.core.Message;
import com.datastax.driver.core.PoolingOptions;
import com.datastax.driver.core.PreparedStatement;
import com.datastax.driver.core.ProtocolVersion;
import com.datastax.driver.core.ResultSetFuture;
import com.datastax.driver.core.ScassandraTestBase;
import com.datastax.driver.core.Session;
import com.datastax.driver.core.SessionManager;
import com.datastax.driver.core.SocketOptions;
import com.datastax.driver.core.Statement;
import com.datastax.driver.core.TestUtils;
import com.datastax.driver.core.exceptions.BusyConnectionException;
import com.datastax.driver.core.exceptions.BusyPoolException;
import com.datastax.driver.core.exceptions.ConnectionException;
import com.datastax.driver.core.exceptions.DriverException;
import com.datastax.driver.core.exceptions.NoHostAvailableException;
import com.datastax.driver.core.policies.ConstantReconnectionPolicy;
import com.datastax.driver.core.policies.ReconnectionPolicy;
import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.base.Throwables;
import com.google.common.collect.Lists;
import com.google.common.util.concurrent.ForwardingListeningExecutorService;
import com.google.common.util.concurrent.FutureCallback;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListeningExecutorService;
import com.google.common.util.concurrent.Uninterruptibles;
import java.net.InetSocketAddress;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import org.assertj.core.api.AbstractObjectAssert;
import org.assertj.core.api.AbstractThrowableAssert;
import org.mockito.Mockito;
import org.mockito.verification.VerificationMode;
import org.scassandra.cql.CqlType;
import org.scassandra.cql.PrimitiveType;
import org.scassandra.http.client.ClosedConnectionReport;
import org.scassandra.http.client.PrimingRequest;
import org.scassandra.http.client.Result;
import org.testng.Assert;
import org.testng.annotations.BeforeClass;
import org.testng.annotations.Test;

public class HostConnectionPoolTest
extends ScassandraTestBase.PerClassCluster {
    static final Integer NEW_CONNECTION_THRESHOLD = (Integer)((Map)PoolingOptions.DEFAULTS.get(ProtocolVersion.V1)).get("newConnectionThresholdLocal");

    @BeforeClass(groups={"short", "long"})
    public void reinitializeCluster() {
        this.cluster.close();
    }

    private void assertPoolSize(HostConnectionPool pool, final int expectedSize) {
        ConditionChecker.check().before(5L, TimeUnit.SECONDS).that(pool, new Predicate<HostConnectionPool>(){

            public boolean apply(HostConnectionPool input) {
                return input.connections.size() == expectedSize;
            }
        }).becomesTrue();
    }

    private void assertBorrowedConnections(Iterable<MockRequest> requests, List<Connection> expectedConnections) {
        for (MockRequest request : requests) {
            Assertions.assertThat(expectedConnections).contains((Object[])new Connection[]{request.getConnection()});
        }
    }

    private void assertBorrowedConnection(Iterable<MockRequest> requests, Connection expectedConnection) {
        this.assertBorrowedConnections(requests, Collections.singletonList(expectedConnection));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test(groups={"short"})
    public void fixed_size_pool_should_fill_its_core_connections_and_queue_and_then_reject() {
        Cluster cluster = this.createClusterBuilder().build();
        ArrayList allRequests = Lists.newArrayList();
        try {
            HostConnectionPool pool = this.createPool(cluster, 2, 2);
            int maxQueueSize = 256;
            Assertions.assertThat((int)pool.connections.size()).isEqualTo(2);
            ArrayList coreConnections = Lists.newArrayList((Iterable)pool.connections);
            List requests = MockRequest.sendMany(256, pool);
            this.assertBorrowedConnections(requests, coreConnections);
            allRequests.addAll(requests);
            allRequests.addAll(MockRequest.sendMany(maxQueueSize, pool, maxQueueSize));
            MockRequest failedBorrow = MockRequest.send(pool, maxQueueSize);
            try {
                failedBorrow.getConnection();
                Assert.fail((String)"Expected a BusyPoolException");
            }
            catch (BusyPoolException e) {
                Assertions.assertThat((Throwable)e).hasMessageContaining("reached its max size");
            }
        }
        finally {
            MockRequest.completeAll(allRequests);
            cluster.close();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test(groups={"short"})
    public void should_reject_if_enqueued_and_timeout_reached() {
        Cluster cluster = this.createClusterBuilder().build();
        ArrayList allRequests = Lists.newArrayList();
        try {
            HostConnectionPool pool = this.createPool(cluster, 1, 1);
            List requests = MockRequest.sendMany(128, pool);
            allRequests.addAll(requests);
            MockRequest failedBorrow = MockRequest.send(pool, 100, 128);
            try {
                failedBorrow.getConnection();
                Assert.fail((String)"Expected a BusyPoolException");
            }
            catch (BusyPoolException e) {
                Assertions.assertThat((Throwable)e).hasMessageContaining("timed out");
            }
        }
        finally {
            MockRequest.completeAll(allRequests);
            cluster.close();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test(groups={"short"})
    public void should_not_hang_when_executing_sync_queries() {
        this.primingClient.prime(PrimingRequest.queryBuilder().withQuery("server_error query").withThen(PrimingRequest.then().withResult(Result.server_error)).build());
        Cluster cluster = this.createClusterBuilder().withReconnectionPolicy((ReconnectionPolicy)new ConstantReconnectionPolicy(10000L)).build();
        cluster.getConfiguration().getPoolingOptions().setPoolTimeoutMillis(500);
        try {
            Session session = cluster.connect();
            try {
                session.execute("server_error query");
                Assert.fail((String)"Exception expected");
            }
            catch (Exception exception) {
                // empty catch block
            }
            try {
                session.execute("this should not block indefinitely");
            }
            catch (NoHostAvailableException nhae) {
                Collection errors = nhae.getErrors().values();
                Assertions.assertThat(errors).hasSize(1);
                Throwable e = (Throwable)errors.iterator().next();
                Assertions.assertThat((Throwable)e).isInstanceOf(BusyPoolException.class);
                Assertions.assertThat((Throwable)e).hasMessageContaining("timed out");
            }
        }
        finally {
            cluster.close();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test(groups={"short"})
    public void requests_with_enqueued_borrow_requests_should_be_failed_when_pool_closes() {
        Cluster cluster = this.createClusterBuilder().build();
        List requests = Lists.newArrayList();
        try {
            HostConnectionPool pool = this.createPool(cluster, 2, 2);
            int maxQueueSize = 256;
            Assertions.assertThat((int)pool.connections.size()).isEqualTo(2);
            ArrayList coreConnections = Lists.newArrayList((Iterable)pool.connections);
            requests = MockRequest.sendMany(256, pool);
            this.assertBorrowedConnections(requests, coreConnections);
            List queuedRequests = MockRequest.sendMany(maxQueueSize, pool, maxQueueSize);
            pool.closeAsync();
            for (MockRequest queuedRequest : queuedRequests) {
                Assertions.assertThat((boolean)queuedRequest.connectionFuture.isDone()).isTrue();
                try {
                    queuedRequest.getConnection();
                    Assert.fail((String)"Expected a ConnectionException");
                }
                catch (ConnectionException connectionException) {}
            }
        }
        finally {
            MockRequest.completeAll(requests);
            cluster.close();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test(groups={"short"})
    public void should_adjust_connection_keyspace_on_dequeue_if_pool_state_is_different() throws TimeoutException, ExecutionException {
        Cluster cluster = this.createClusterBuilder().build();
        List requests = Lists.newArrayList();
        try {
            HostConnectionPool pool = this.createPool(cluster, 1, 1);
            int maxQueueSize = 256;
            Assertions.assertThat((int)pool.connections.size()).isEqualTo(1);
            ArrayList coreConnections = Lists.newArrayList((Iterable)pool.connections);
            requests = MockRequest.sendMany(128, pool);
            this.assertBorrowedConnections(requests, coreConnections);
            List queuedRequests = MockRequest.sendMany(maxQueueSize, pool, maxQueueSize);
            for (MockRequest request : requests) {
                Uninterruptibles.getUninterruptibly(request.connectionFuture, (long)5L, (TimeUnit)TimeUnit.SECONDS);
            }
            this.primingClient.prime(PrimingRequest.queryBuilder().withQuery("USE \"newkeyspace\"").withThen(PrimingRequest.then().withFixedDelay(Long.valueOf(2000L))));
            pool.manager.poolsState.setKeyspace("newkeyspace");
            MockRequest.completeAll(requests);
            int count = 0;
            for (MockRequest queuedRequest : queuedRequests) {
                try {
                    Uninterruptibles.getUninterruptibly(queuedRequest.connectionFuture, (long)10L, (TimeUnit)TimeUnit.SECONDS);
                    ++count;
                }
                catch (ExecutionException e) {
                    ((AbstractThrowableAssert)Assertions.assertThat((Throwable)e.getCause()).isInstanceOf(BusyPoolException.class)).hasMessageContaining("timed out after");
                    Assertions.assertThat((int)count).isEqualTo(128);
                    break;
                }
            }
            Assertions.assertThat((List)this.activityClient.retrieveQueries()).extractingResultOf("getQuery").containsOnlyOnce(new Object[]{"USE \"newkeyspace\""});
        }
        catch (Throwable throwable) {
            MockRequest.completeAll(requests);
            cluster.close();
            throw throwable;
        }
        MockRequest.completeAll(requests);
        cluster.close();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test(groups={"short"})
    public void should_fail_in_borrowConnection_when_setting_keyspace_and_another_set_keyspace_attempt_is_in_flight() throws TimeoutException {
        Cluster cluster = this.createClusterBuilder().build();
        try {
            HostConnectionPool pool = this.createPool(cluster, 1, 1);
            this.primingClient.prime(PrimingRequest.queryBuilder().withQuery("USE \"slowks\"").withThen(PrimingRequest.then().withFixedDelay(Long.valueOf(5000L))));
            Connection connection = (Connection)pool.connections.get(0);
            connection.setKeyspaceAsync("slowks");
            pool.manager.poolsState.setKeyspace("newks");
            MockRequest request = MockRequest.send(pool);
            try {
                Uninterruptibles.getUninterruptibly(request.connectionFuture, (long)5L, (TimeUnit)TimeUnit.SECONDS);
                Assert.fail((String)"Should have thrown exception");
            }
            catch (ExecutionException e) {
                Assertions.assertThat((Throwable)e.getCause()).isInstanceOf(DriverException.class);
                Assertions.assertThat((String)e.getCause().getMessage()).contains(new CharSequence[]{"Aborting attempt to set keyspace to 'newks' since there is already an in flight attempt to set keyspace to 'slowks'."});
            }
        }
        finally {
            cluster.close();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test(groups={"short"})
    public void should_fail_in_dequeue_when_setting_keyspace_and_another_set_keyspace_attempt_is_in_flight() throws ExecutionException, TimeoutException {
        Cluster cluster = this.createClusterBuilder().build();
        List requests = Lists.newArrayList();
        try {
            HostConnectionPool pool = this.createPool(cluster, 1, 1);
            cluster.getConfiguration().getPoolingOptions().setMaxRequestsPerConnection(HostDistance.LOCAL, 100);
            int maxQueueSize = 256;
            Assertions.assertThat((int)pool.connections.size()).isEqualTo(1);
            ArrayList coreConnections = Lists.newArrayList((Iterable)pool.connections);
            requests = MockRequest.sendMany(100, pool);
            this.assertBorrowedConnections(requests, coreConnections);
            MockRequest queuedRequest = MockRequest.send(pool, maxQueueSize);
            for (MockRequest request : requests) {
                Uninterruptibles.getUninterruptibly(request.connectionFuture, (long)5L, (TimeUnit)TimeUnit.SECONDS);
            }
            this.primingClient.prime(PrimingRequest.queryBuilder().withQuery("USE \"slowks\"").withThen(PrimingRequest.then().withFixedDelay(Long.valueOf(5000L))));
            Connection connection = (Connection)pool.connections.get(0);
            connection.setKeyspaceAsync("slowks");
            pool.manager.poolsState.setKeyspace("newkeyspace");
            MockRequest.completeAll(requests);
            try {
                Uninterruptibles.getUninterruptibly(queuedRequest.connectionFuture, (long)5L, (TimeUnit)TimeUnit.SECONDS);
                Assert.fail((String)"Should have thrown exception");
            }
            catch (ExecutionException e) {
                Assertions.assertThat((Throwable)e.getCause()).isInstanceOf(DriverException.class);
                Assertions.assertThat((String)e.getCause().getMessage()).contains(new CharSequence[]{"Aborting attempt to set keyspace to 'newkeyspace' since there is already an in flight attempt to set keyspace to 'slowks'."});
            }
        }
        catch (Throwable throwable) {
            MockRequest.completeAll(requests);
            cluster.close();
            throw throwable;
        }
        MockRequest.completeAll(requests);
        cluster.close();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test(groups={"short"})
    public void variable_size_pool_should_fill_its_connections_and_then_reject() throws Exception {
        Cluster cluster = this.createClusterBuilder().build();
        ArrayList allRequests = Lists.newArrayList();
        try {
            Connection.Factory factory;
            HostConnectionPool pool = this.createPool(cluster, 1, 2);
            cluster.manager.connectionFactory = factory = (Connection.Factory)Mockito.spy((Object)cluster.manager.connectionFactory);
            Assertions.assertThat((int)pool.connections.size()).isEqualTo(1);
            Connection coreConnection = (Connection)pool.connections.get(0);
            List requests = MockRequest.sendMany(HostConnectionPoolTest.NEW_CONNECTION_THRESHOLD, pool);
            this.assertBorrowedConnection(requests, coreConnection);
            allRequests.addAll(requests);
            allRequests.add(MockRequest.send(pool));
            ((Connection.Factory)Mockito.verify((Object)factory, (VerificationMode)Mockito.after((int)2000).times(1))).open((HostConnectionPool)Mockito.any(HostConnectionPool.class));
            this.assertPoolSize(pool, 2);
            for (int i = 0; i < NEW_CONNECTION_THRESHOLD; ++i) {
                MockRequest request = MockRequest.send(pool);
                Assertions.assertThat((Object)request.getConnection()).isNotEqualTo((Object)coreConnection);
                allRequests.add(request);
            }
            allRequests.addAll(MockRequest.sendMany(55, pool));
            MockRequest failedBorrow = MockRequest.send(pool);
            try {
                failedBorrow.getConnection();
                Assert.fail((String)"Expected a BusyPoolException");
            }
            catch (BusyPoolException busyPoolException) {
                // empty catch block
            }
        }
        finally {
            MockRequest.completeAll(allRequests);
            cluster.close();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test(groups={"short"})
    public void should_add_extra_connection_when_core_full() throws Exception {
        Cluster cluster = this.createClusterBuilder().build();
        ArrayList allRequests = Lists.newArrayList();
        try {
            Connection.Factory factory;
            HostConnectionPool pool = this.createPool(cluster, 1, 2);
            cluster.manager.connectionFactory = factory = (Connection.Factory)Mockito.spy((Object)cluster.manager.connectionFactory);
            Connection core = (Connection)pool.connections.get(0);
            List requests = MockRequest.sendMany(HostConnectionPoolTest.NEW_CONNECTION_THRESHOLD, pool);
            this.assertBorrowedConnection(requests, core);
            allRequests.addAll(requests);
            allRequests.add(MockRequest.send(pool));
            ((Connection.Factory)Mockito.verify((Object)factory, (VerificationMode)Mockito.after((int)2000).times(1))).open((HostConnectionPool)Mockito.any(HostConnectionPool.class));
            this.assertPoolSize(pool, 2);
        }
        finally {
            MockRequest.completeAll(allRequests);
            cluster.close();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test(groups={"long"})
    public void should_resurrect_trashed_connection_within_idle_timeout() throws Exception {
        Cluster cluster = this.createClusterBuilder().withPoolingOptions(new PoolingOptions().setIdleTimeoutSeconds(20)).build();
        ArrayList allRequests = Lists.newArrayList();
        try {
            Connection.Factory factory;
            HostConnectionPool pool = this.createPool(cluster, 1, 2);
            cluster.manager.connectionFactory = factory = (Connection.Factory)Mockito.spy((Object)cluster.manager.connectionFactory);
            Connection connection1 = (Connection)pool.connections.get(0);
            List requests = MockRequest.sendMany(HostConnectionPoolTest.NEW_CONNECTION_THRESHOLD, pool);
            this.assertBorrowedConnections(requests, Collections.singletonList(connection1));
            allRequests.addAll(requests);
            allRequests.add(MockRequest.send(pool));
            ((Connection.Factory)Mockito.verify((Object)factory, (VerificationMode)Mockito.after((int)2000).times(1))).open((HostConnectionPool)Mockito.any(HostConnectionPool.class));
            this.assertPoolSize(pool, 2);
            Connection connection2 = (Connection)pool.connections.get(1);
            Assertions.assertThat((int)connection1.inFlight.get()).isEqualTo(101);
            Assertions.assertThat((int)connection2.inFlight.get()).isEqualTo(0);
            MockRequest.completeMany(51, allRequests);
            Assertions.assertThat((int)connection1.inFlight.get()).isEqualTo(50);
            Assertions.assertThat((int)connection2.inFlight.get()).isEqualTo(0);
            Uninterruptibles.sleepUninterruptibly((long)20L, (TimeUnit)TimeUnit.SECONDS);
            Assertions.assertThat((List)pool.connections).containsExactly((Object[])new Connection[]{connection2});
            Assertions.assertThat((Iterable)pool.trash).containsExactly((Object[])new Connection[]{connection1});
            allRequests.addAll(MockRequest.sendMany(50, pool));
            Assertions.assertThat((List)pool.connections).containsExactly((Object[])new Connection[]{connection2});
            Assertions.assertThat((Iterable)pool.trash).containsExactly((Object[])new Connection[]{connection1});
            Assertions.assertThat((int)connection1.inFlight.get()).isEqualTo(50);
            Assertions.assertThat((int)connection2.inFlight.get()).isEqualTo(50);
            allRequests.addAll(MockRequest.sendMany(1, pool));
            ((Connection.Factory)Mockito.verify((Object)factory, (VerificationMode)Mockito.after((int)2000).times(1))).open((HostConnectionPool)Mockito.any(HostConnectionPool.class));
            this.assertPoolSize(pool, 2);
            Assertions.assertThat((List)pool.connections).containsExactly((Object[])new Connection[]{connection2, connection1});
            Assertions.assertThat((Iterable)pool.trash).isEmpty();
            Assertions.assertThat((int)connection1.inFlight.get()).isEqualTo(50);
            Assertions.assertThat((int)connection2.inFlight.get()).isEqualTo(51);
        }
        finally {
            MockRequest.completeAll(allRequests);
            cluster.close();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test(groups={"long"})
    public void should_not_resurrect_trashed_connection_after_idle_timeout() throws Exception {
        Cluster cluster = this.createClusterBuilder().withPoolingOptions(new PoolingOptions().setIdleTimeoutSeconds(20)).build();
        ArrayList allRequests = Lists.newArrayList();
        try {
            Connection.Factory factory;
            HostConnectionPool pool = this.createPool(cluster, 1, 2);
            cluster.manager.connectionFactory = factory = (Connection.Factory)Mockito.spy((Object)cluster.manager.connectionFactory);
            Connection connection1 = (Connection)pool.connections.get(0);
            List requests = MockRequest.sendMany(HostConnectionPoolTest.NEW_CONNECTION_THRESHOLD, pool);
            this.assertBorrowedConnection(requests, connection1);
            allRequests.addAll(requests);
            allRequests.add(MockRequest.send(pool));
            ((Connection.Factory)Mockito.verify((Object)factory, (VerificationMode)Mockito.after((int)2000).times(1))).open((HostConnectionPool)Mockito.any(HostConnectionPool.class));
            this.assertPoolSize(pool, 2);
            Mockito.reset((Object[])new Connection.Factory[]{factory});
            Connection connection2 = (Connection)pool.connections.get(1);
            Assertions.assertThat((int)connection1.inFlight.get()).isEqualTo(101);
            Assertions.assertThat((int)connection2.inFlight.get()).isEqualTo(0);
            MockRequest.completeMany(51, allRequests);
            Assertions.assertThat((int)connection1.inFlight.get()).isEqualTo(50);
            Assertions.assertThat((int)connection2.inFlight.get()).isEqualTo(0);
            Uninterruptibles.sleepUninterruptibly((long)20L, (TimeUnit)TimeUnit.SECONDS);
            Assertions.assertThat((List)pool.connections).containsExactly((Object[])new Connection[]{connection2});
            Assertions.assertThat((Iterable)pool.trash).containsExactly((Object[])new Connection[]{connection1});
            MockRequest.completeMany(50, allRequests);
            Assertions.assertThat((int)connection1.inFlight.get()).isEqualTo(0);
            Uninterruptibles.sleepUninterruptibly((long)30L, (TimeUnit)TimeUnit.SECONDS);
            Assertions.assertThat((List)pool.connections).containsExactly((Object[])new Connection[]{connection2});
            Assertions.assertThat((Iterable)pool.trash).isEmpty();
            Assertions.assertThat((boolean)connection1.isClosed()).isTrue();
            requests = MockRequest.sendMany(HostConnectionPoolTest.NEW_CONNECTION_THRESHOLD, pool);
            this.assertBorrowedConnection(requests, connection2);
            allRequests.addAll(requests);
            allRequests.add(MockRequest.send(pool));
            Assertions.assertThat((int)connection2.inFlight.get()).isEqualTo(101);
            ((Connection.Factory)Mockito.verify((Object)factory, (VerificationMode)Mockito.after((int)2000).times(1))).open((HostConnectionPool)Mockito.any(HostConnectionPool.class));
            this.assertPoolSize(pool, 2);
            MockRequest request = MockRequest.send(pool);
            allRequests.add(request);
            ((AbstractObjectAssert)Assertions.assertThat((Object)request.getConnection()).isNotEqualTo((Object)connection2)).isNotEqualTo((Object)connection1);
        }
        finally {
            MockRequest.completeAll(allRequests);
            cluster.close();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test(groups={"long"})
    public void should_not_close_trashed_connection_until_no_in_flight() throws Exception {
        Cluster cluster = this.createClusterBuilder().withPoolingOptions(new PoolingOptions().setIdleTimeoutSeconds(20)).build();
        ArrayList allRequests = Lists.newArrayList();
        try {
            Connection.Factory factory;
            HostConnectionPool pool = this.createPool(cluster, 1, 2);
            cluster.manager.connectionFactory = factory = (Connection.Factory)Mockito.spy((Object)cluster.manager.connectionFactory);
            Connection connection1 = (Connection)pool.connections.get(0);
            List requests = MockRequest.sendMany(HostConnectionPoolTest.NEW_CONNECTION_THRESHOLD, pool);
            this.assertBorrowedConnections(requests, Collections.singletonList(connection1));
            allRequests.addAll(requests);
            allRequests.add(MockRequest.send(pool));
            ((Connection.Factory)Mockito.verify((Object)factory, (VerificationMode)Mockito.after((int)2000).times(1))).open((HostConnectionPool)Mockito.any(HostConnectionPool.class));
            Assertions.assertThat((List)pool.connections).hasSize(2);
            MockRequest.completeMany(50, allRequests);
            Uninterruptibles.sleepUninterruptibly((long)30L, (TimeUnit)TimeUnit.SECONDS);
            Assertions.assertThat((Iterable)pool.trash).containsExactly((Object[])new Connection[]{connection1});
            Assertions.assertThat((int)connection1.inFlight.get()).isEqualTo(51);
            Assertions.assertThat((boolean)connection1.isClosed()).isFalse();
            MockRequest.completeMany(51, allRequests);
            Uninterruptibles.sleepUninterruptibly((long)30L, (TimeUnit)TimeUnit.SECONDS);
            Assertions.assertThat((boolean)connection1.isClosed()).isTrue();
            Assertions.assertThat((List)pool.connections).doesNotContain((Object[])new Connection[]{connection1});
            Assertions.assertThat((Iterable)pool.trash).doesNotContain((Object[])new Connection[]{connection1});
        }
        finally {
            MockRequest.completeAll(allRequests);
            cluster.close();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test(groups={"short"})
    public void should_trash_on_returning_connection_with_insufficient_streams() throws Exception {
        Cluster cluster = this.createClusterBuilder().build();
        ArrayList allRequests = Lists.newArrayList();
        try {
            Connection.Factory factory;
            HostConnectionPool pool = this.createPool(cluster, 1, 2);
            cluster.manager.connectionFactory = factory = (Connection.Factory)Mockito.spy((Object)cluster.manager.connectionFactory);
            Connection core = (Connection)pool.connections.get(0);
            List requests = MockRequest.sendMany(HostConnectionPoolTest.NEW_CONNECTION_THRESHOLD, pool);
            this.assertBorrowedConnections(requests, Collections.singletonList(core));
            allRequests.addAll(requests);
            allRequests.add(MockRequest.send(pool));
            ((Connection.Factory)Mockito.verify((Object)factory, (VerificationMode)Mockito.after((int)2000).times(1))).open((HostConnectionPool)Mockito.any(HostConnectionPool.class));
            Assertions.assertThat((List)pool.connections).hasSize(2);
            Connection extra1 = (Connection)Mockito.spy(pool.connections.get(1));
            pool.connections.set(1, extra1);
            allRequests.addAll(MockRequest.sendMany(10, pool));
            Assertions.assertThat((List)pool.connections).hasSize(2);
            ((Connection)Mockito.doReturn((Object)0).when((Object)extra1)).maxAvailableStreams();
            Assertions.assertThat((Iterable)pool.trash).hasSize(0);
            pool.returnConnection(extra1);
            Assertions.assertThat((Iterable)pool.trash).hasSize(1);
        }
        finally {
            MockRequest.completeAll(allRequests);
            cluster.close();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test(groups={"short"})
    public void should_keep_host_up_when_one_connection_lost() throws Exception {
        Cluster cluster = this.createClusterBuilder().build();
        try {
            HostConnectionPool pool = this.createPool(cluster, 2, 2);
            Connection core0 = (Connection)pool.connections.get(0);
            Connection core1 = (Connection)pool.connections.get(1);
            this.currentClient.disableListener();
            this.currentClient.closeConnection(ClosedConnectionReport.CloseType.CLOSE, (InetSocketAddress)core0.channel.localAddress());
            Uninterruptibles.sleepUninterruptibly((long)100L, (TimeUnit)TimeUnit.MILLISECONDS);
            Assertions.assertThat((boolean)core0.isClosed()).isTrue();
            Assertions.assertThat((boolean)core1.isClosed()).isFalse();
            Assertions.assertThat((List)pool.connections).doesNotContain((Object[])new Connection[]{core0});
            Assertions.assertThat(cluster).host(1).hasState(Host.State.UP);
            Assertions.assertThat(cluster).hasOpenControlConnection();
        }
        finally {
            cluster.close();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test(groups={"short"})
    public void should_mark_host_down_when_no_connections_remaining() throws Exception {
        int readTimeout = 1000;
        int reconnectInterval = 1000;
        Cluster cluster = this.createClusterBuilder().withSocketOptions(new SocketOptions().setConnectTimeoutMillis(readTimeout).setReadTimeoutMillis(reconnectInterval)).withReconnectionPolicy((ReconnectionPolicy)new ConstantReconnectionPolicy(1000L)).build();
        try {
            Connection.Factory factory;
            cluster.init();
            cluster.manager.connectionFactory = factory = (Connection.Factory)Mockito.spy((Object)cluster.manager.connectionFactory);
            HostConnectionPool pool = this.createPool(cluster, 8, 8);
            ArrayList connections = Lists.newArrayList((Iterable)pool.connections);
            Mockito.reset((Object[])new Connection.Factory[]{factory});
            this.currentClient.disableListener();
            this.currentClient.closeConnections(ClosedConnectionReport.CloseType.CLOSE);
            Assertions.assertThat(cluster).host(1).goesDownWithin(10L, TimeUnit.SECONDS);
            Assertions.assertThat(cluster).hasClosedControlConnection();
            for (Connection connection : connections) {
                Assertions.assertThat((boolean)connection.isClosed()).isTrue();
            }
            ((Connection.Factory)Mockito.verify((Object)factory, (VerificationMode)Mockito.timeout((long)(reconnectInterval * 2)).atLeastOnce())).open(this.host);
            Uninterruptibles.sleepUninterruptibly((long)(readTimeout * 2), (TimeUnit)TimeUnit.MILLISECONDS);
            Assertions.assertThat(cluster).hasClosedControlConnection();
            this.currentClient.enableListener();
            ((Connection.Factory)Mockito.verify((Object)factory, (VerificationMode)Mockito.after((int)(reconnectInterval * 2)).atLeast(2))).open(this.host);
            ((Connection.Factory)Mockito.verify((Object)factory, (VerificationMode)Mockito.timeout((long)(reconnectInterval * 2)))).newConnections((HostConnectionPool)Mockito.any(HostConnectionPool.class), Mockito.eq((int)7));
            Uninterruptibles.sleepUninterruptibly((long)readTimeout, (TimeUnit)TimeUnit.MILLISECONDS);
            Assertions.assertThat(cluster).hasOpenControlConnection();
            Assertions.assertThat(cluster).host(1).hasState(Host.State.UP);
        }
        finally {
            cluster.close();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test(groups={"short"})
    public void should_create_new_connections_when_connection_lost_and_under_core_connections() throws Exception {
        int readTimeout = 1000;
        int reconnectInterval = 1000;
        Cluster cluster = this.createClusterBuilder().withSocketOptions(new SocketOptions().setConnectTimeoutMillis(readTimeout).setReadTimeoutMillis(reconnectInterval)).withReconnectionPolicy((ReconnectionPolicy)new ConstantReconnectionPolicy(1000L)).build();
        ArrayList allRequests = Lists.newArrayList();
        try {
            Connection.Factory factory;
            cluster.init();
            cluster.manager.connectionFactory = factory = (Connection.Factory)Mockito.spy((Object)cluster.manager.connectionFactory);
            TestExecutorService blockingExecutor = new TestExecutorService(cluster.manager.blockingExecutor);
            cluster.manager.blockingExecutor = blockingExecutor;
            HostConnectionPool pool = this.createPool(cluster, 3, 3);
            Connection core0 = (Connection)pool.connections.get(0);
            Connection core1 = (Connection)pool.connections.get(1);
            Connection core2 = (Connection)pool.connections.get(2);
            this.currentClient.disableListener();
            this.currentClient.closeConnection(ClosedConnectionReport.CloseType.CLOSE, (InetSocketAddress)core0.channel.localAddress());
            this.currentClient.closeConnection(ClosedConnectionReport.CloseType.CLOSE, (InetSocketAddress)core2.channel.localAddress());
            Uninterruptibles.sleepUninterruptibly((long)100L, (TimeUnit)TimeUnit.MILLISECONDS);
            Assertions.assertThat(cluster).host(1).hasState(Host.State.UP);
            Assertions.assertThat((List)pool.connections).hasSize(1);
            MockRequest request = MockRequest.send(pool);
            allRequests.add(request);
            Assertions.assertThat((Object)request.getConnection()).isEqualTo((Object)core1);
            ((Connection.Factory)Mockito.verify((Object)factory, (VerificationMode)Mockito.never())).open((HostConnectionPool)Mockito.any(HostConnectionPool.class));
            Uninterruptibles.sleepUninterruptibly((long)reconnectInterval, (TimeUnit)TimeUnit.MILLISECONDS);
            blockingExecutor.reset();
            request = MockRequest.send(pool);
            allRequests.add(request);
            Assertions.assertThat((Object)request.getConnection()).isEqualTo((Object)core1);
            blockingExecutor.blockUntilNextTaskCompleted();
            ((Connection.Factory)Mockito.verify((Object)factory)).open((HostConnectionPool)Mockito.any(HostConnectionPool.class));
            Mockito.reset((Object[])new Connection.Factory[]{factory});
            Uninterruptibles.sleepUninterruptibly((long)((readTimeout + reconnectInterval) * 2), (TimeUnit)TimeUnit.MILLISECONDS);
            this.currentClient.enableListener();
            Uninterruptibles.sleepUninterruptibly((long)reconnectInterval, (TimeUnit)TimeUnit.MILLISECONDS);
            blockingExecutor.reset();
            allRequests.add(MockRequest.send(pool));
            blockingExecutor.blockUntilNextTaskCompleted();
            ((Connection.Factory)Mockito.verify((Object)factory)).open((HostConnectionPool)Mockito.any(HostConnectionPool.class));
            Mockito.reset((Object[])new Connection.Factory[]{factory});
            blockingExecutor.reset();
            allRequests.add(MockRequest.send(pool));
            blockingExecutor.blockUntilNextTaskCompleted();
            ((Connection.Factory)Mockito.verify((Object)factory)).open((HostConnectionPool)Mockito.any(HostConnectionPool.class));
            Mockito.reset((Object[])new Connection.Factory[]{factory});
            allRequests.add(MockRequest.send(pool));
            ((Connection.Factory)Mockito.verify((Object)factory, (VerificationMode)Mockito.after((int)((reconnectInterval + readTimeout) * 2)).never())).open((HostConnectionPool)Mockito.any(HostConnectionPool.class));
        }
        finally {
            MockRequest.completeAll(allRequests);
            cluster.close();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test(groups={"short"})
    public void should_not_schedule_reconnect_when_connection_lost_and_at_core_connections() throws Exception {
        int readTimeout = 1000;
        int reconnectInterval = 1000;
        Cluster cluster = this.createClusterBuilder().withSocketOptions(new SocketOptions().setConnectTimeoutMillis(readTimeout).setReadTimeoutMillis(reconnectInterval)).withReconnectionPolicy((ReconnectionPolicy)new ConstantReconnectionPolicy(1000L)).build();
        ArrayList allRequests = Lists.newArrayList();
        try {
            MockRequest request3;
            Connection.Factory factory;
            cluster.init();
            cluster.manager.connectionFactory = factory = (Connection.Factory)Mockito.spy((Object)cluster.manager.connectionFactory);
            HostConnectionPool pool = this.createPool(cluster, 1, 2);
            Connection core0 = (Connection)pool.connections.get(0);
            ArrayList core0requests = Lists.newArrayList();
            for (int i = 0; i < 101; ++i) {
                MockRequest request2 = MockRequest.send(pool);
                Assertions.assertThat((Object)request2.getConnection()).isEqualTo((Object)core0);
                core0requests.add(request2);
            }
            ((Connection.Factory)Mockito.verify((Object)factory, (VerificationMode)Mockito.after((int)2000).times(1))).open((HostConnectionPool)Mockito.any(HostConnectionPool.class));
            Assertions.assertThat((List)pool.connections).hasSize(2);
            Mockito.reset((Object[])new Connection.Factory[]{factory});
            Connection extra1 = (Connection)pool.connections.get(1);
            this.currentClient.closeConnection(ClosedConnectionReport.CloseType.CLOSE, (InetSocketAddress)core0.channel.localAddress());
            Uninterruptibles.sleepUninterruptibly((long)100L, (TimeUnit)TimeUnit.MILLISECONDS);
            this.currentClient.disableListener();
            for (MockRequest request3 : core0requests) {
                Assertions.assertThat((Comparable)((Comparable)((Object)request3.state.get()))).isEqualTo((Object)MockRequest.State.FAILED);
            }
            Assertions.assertThat(cluster).host(1).hasState(Host.State.UP);
            List requests = MockRequest.sendMany(100, pool);
            this.assertBorrowedConnections(requests, Collections.singletonList(extra1));
            allRequests.addAll(requests);
            Assertions.assertThat((List)pool.connections).hasSize(1);
            ((Connection.Factory)Mockito.verify((Object)factory, (VerificationMode)Mockito.after((int)readTimeout).never())).open((HostConnectionPool)Mockito.any(HostConnectionPool.class));
            request3 = MockRequest.send(pool);
            allRequests.add(request3);
            Assertions.assertThat((Object)request3.getConnection()).isEqualTo((Object)extra1);
            ((Connection.Factory)Mockito.verify((Object)factory, (VerificationMode)Mockito.timeout((long)readTimeout))).open((HostConnectionPool)Mockito.any(HostConnectionPool.class));
            this.assertPoolSize(pool, 1);
            Assertions.assertThat((List)pool.connections).hasSize(1);
            Uninterruptibles.sleepUninterruptibly((long)(readTimeout * 2), (TimeUnit)TimeUnit.MILLISECONDS);
            this.assertPoolSize(pool, 1);
            this.currentClient.enableListener();
            Uninterruptibles.sleepUninterruptibly((long)reconnectInterval, (TimeUnit)TimeUnit.MILLISECONDS);
            request3 = MockRequest.send(pool);
            allRequests.add(request3);
            Assertions.assertThat((Object)request3.getConnection()).isEqualTo((Object)extra1);
            Uninterruptibles.sleepUninterruptibly((long)readTimeout, (TimeUnit)TimeUnit.MILLISECONDS);
            Assertions.assertThat((List)pool.connections).hasSize(2);
            request3 = MockRequest.send(pool);
            allRequests.add(request3);
            ((AbstractObjectAssert)Assertions.assertThat((Object)request3.getConnection()).isNotEqualTo((Object)core0)).isNotEqualTo((Object)extra1);
        }
        finally {
            MockRequest.completeAll(allRequests);
            cluster.close();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test(groups={"short"})
    public void should_not_mark_host_down_if_some_connections_fail_on_init() throws Exception {
        int readTimeout = 1000;
        int reconnectInterval = 1000;
        Cluster cluster = this.createClusterBuilder().withSocketOptions(new SocketOptions().setConnectTimeoutMillis(readTimeout).setReadTimeoutMillis(reconnectInterval)).withReconnectionPolicy((ReconnectionPolicy)new ConstantReconnectionPolicy(1000L)).build();
        ArrayList allRequests = Lists.newArrayList();
        try {
            Connection.Factory factory;
            cluster.init();
            cluster.manager.connectionFactory = factory = (Connection.Factory)Mockito.spy((Object)cluster.manager.connectionFactory);
            this.currentClient.disableListener(4);
            HostConnectionPool pool = this.createPool(cluster, 8, 8);
            Mockito.reset((Object[])new Connection.Factory[]{factory});
            Assertions.assertThat((List)pool.connections).hasSize(4);
            Assertions.assertThat(cluster).host(1).hasState(Host.State.UP);
            Assertions.assertThat(cluster).hasOpenControlConnection();
            this.currentClient.enableListener();
            Uninterruptibles.sleepUninterruptibly((long)reconnectInterval, (TimeUnit)TimeUnit.MILLISECONDS);
            for (int i = 5; i <= 8; ++i) {
                allRequests.add(MockRequest.send(pool));
                ((Connection.Factory)Mockito.verify((Object)factory, (VerificationMode)Mockito.timeout((long)readTimeout))).open((HostConnectionPool)Mockito.any(HostConnectionPool.class));
                Mockito.reset((Object[])new Connection.Factory[]{factory});
                this.assertPoolSize(pool, i);
            }
        }
        finally {
            MockRequest.completeAll(allRequests);
            cluster.close();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test(groups={"short"})
    public void should_throw_exception_if_convicted_and_no_connections_available() {
        int readTimeout = 1000;
        int reconnectInterval = 1000;
        Cluster cluster = this.createClusterBuilder().withSocketOptions(new SocketOptions().setConnectTimeoutMillis(readTimeout).setReadTimeoutMillis(reconnectInterval)).withReconnectionPolicy((ReconnectionPolicy)new ConstantReconnectionPolicy(1000L)).build();
        try {
            Connection.Factory factory;
            cluster.init();
            Assertions.assertThat(cluster).hasOpenControlConnection();
            cluster.manager.connectionFactory = factory = (Connection.Factory)Mockito.spy((Object)cluster.manager.connectionFactory);
            this.currentClient.disableListener();
            HostConnectionPool pool = this.createPool(cluster, 8, 8);
            Mockito.reset((Object[])new Connection.Factory[]{factory});
            Assertions.assertThat((List)pool.connections).hasSize(0);
            Assertions.assertThat(cluster).host(1).hasState(Host.State.UP);
            Assertions.assertThat(cluster).hasOpenControlConnection();
            MockRequest failedBorrow = MockRequest.send(pool);
            try {
                failedBorrow.getConnection();
                Assert.fail((String)"Expected a BusyPoolException");
            }
            catch (BusyPoolException busyPoolException) {
                // empty catch block
            }
        }
        finally {
            cluster.close();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test(groups={"short"})
    public void should_wait_on_connection_if_not_convicted_and_no_connections_available() throws Exception {
        int readTimeout = 1000;
        int reconnectInterval = 1000;
        Cluster cluster = this.createClusterBuilder().withSocketOptions(new SocketOptions().setConnectTimeoutMillis(readTimeout).setReadTimeoutMillis(reconnectInterval)).withReconnectionPolicy((ReconnectionPolicy)new ConstantReconnectionPolicy(1000L)).build();
        try {
            Connection.Factory factory;
            cluster.init();
            Assertions.assertThat(cluster).hasOpenControlConnection();
            cluster.manager.connectionFactory = factory = (Connection.Factory)Mockito.spy((Object)cluster.manager.connectionFactory);
            this.currentClient.disableListener();
            HostConnectionPool pool = this.createPool(cluster, 8, 8);
            Assertions.assertThat((List)pool.connections).hasSize(0);
            Assertions.assertThat(cluster).host(1).hasState(Host.State.UP);
            Assertions.assertThat(cluster).hasOpenControlConnection();
            this.currentClient.enableListener();
            Uninterruptibles.sleepUninterruptibly((long)reconnectInterval, (TimeUnit)TimeUnit.MILLISECONDS);
            Mockito.reset((Object[])new Connection.Factory[]{factory});
            MockRequest request = MockRequest.send(pool, 1);
            ((Connection.Factory)Mockito.verify((Object)factory, (VerificationMode)Mockito.timeout((long)(readTimeout * 8)).times(8))).open((HostConnectionPool)Mockito.any(HostConnectionPool.class));
            this.assertPoolSize(pool, 8);
            request.simulateSuccessResponse();
        }
        finally {
            cluster.close();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test(groups={"short"})
    public void should_wait_on_connection_if_zero_core_connections() throws Exception {
        int readTimeout = 1000;
        int reconnectInterval = 1000;
        Cluster cluster = this.createClusterBuilder().withSocketOptions(new SocketOptions().setConnectTimeoutMillis(readTimeout).setReadTimeoutMillis(reconnectInterval)).withReconnectionPolicy((ReconnectionPolicy)new ConstantReconnectionPolicy(1000L)).build();
        try {
            Connection.Factory factory;
            cluster.init();
            Assertions.assertThat(cluster).hasOpenControlConnection();
            cluster.manager.connectionFactory = factory = (Connection.Factory)Mockito.spy((Object)cluster.manager.connectionFactory);
            HostConnectionPool pool = this.createPool(cluster, 0, 2);
            Assertions.assertThat((List)pool.connections).hasSize(0);
            Assertions.assertThat(cluster).host(1).hasState(Host.State.UP);
            Assertions.assertThat(cluster).hasOpenControlConnection();
            Mockito.reset((Object[])new Connection.Factory[]{factory});
            MockRequest request = MockRequest.send(pool, 1);
            ((Connection.Factory)Mockito.verify((Object)factory, (VerificationMode)Mockito.timeout((long)readTimeout).times(1))).open((HostConnectionPool)Mockito.any(HostConnectionPool.class));
            this.assertPoolSize(pool, 1);
            Uninterruptibles.getUninterruptibly(request.requestInitialized, (long)10L, (TimeUnit)TimeUnit.SECONDS);
            request.simulateSuccessResponse();
        }
        finally {
            cluster.close();
        }
    }

    private HostConnectionPool createPool(Cluster cluster, int coreConnections, int maxConnections) {
        cluster.getConfiguration().getPoolingOptions().setNewConnectionThreshold(HostDistance.LOCAL, 100).setMaxRequestsPerConnection(HostDistance.LOCAL, 128).setMaxConnectionsPerHost(HostDistance.LOCAL, maxConnections).setCoreConnectionsPerHost(HostDistance.LOCAL, coreConnections);
        Session session = cluster.connect();
        Host host = TestUtils.findHost(cluster, 1);
        SessionManager sm = (SessionManager)session;
        return (HostConnectionPool)sm.pools.get(host);
    }

    @Test(groups={"long"}, enabled=false)
    public void open_connections_metric_should_always_be_positive() throws InterruptedException {
        int numberOfInserts = 100000;
        final CountDownLatch pendingInserts = new CountDownLatch(numberOfInserts);
        ExecutorService progressReportExecutor = Executors.newSingleThreadExecutor();
        Runnable progressReporter = new Runnable(){

            @Override
            public void run() {
                pendingInserts.countDown();
            }
        };
        final AtomicBoolean negativeOpenConnectionCountSpotted = new AtomicBoolean(false);
        final Gauge openConnections = this.cluster.getMetrics().getOpenConnections();
        ScheduledExecutorService openConnectionsWatcherExecutor = Executors.newScheduledThreadPool(1);
        Runnable openConnectionsWatcher = new Runnable(){

            @Override
            public void run() {
                Integer value = (Integer)openConnections.getValue();
                if (value < 0) {
                    System.err.println("Negative value spotted for openConnection metric: " + value);
                    negativeOpenConnectionCountSpotted.set(true);
                }
            }
        };
        openConnectionsWatcherExecutor.scheduleAtFixedRate(openConnectionsWatcher, 1L, 1L, TimeUnit.SECONDS);
        PreparedStatement insertStatement = this.session.prepare(this.generateJava349InsertStatement());
        for (int key = 0; key < numberOfInserts; ++key) {
            ResultSetFuture future = this.session.executeAsync((Statement)insertStatement.bind(new Object[]{key}));
            future.addListener(progressReporter, (Executor)progressReportExecutor);
        }
        pendingInserts.await();
        openConnectionsWatcherExecutor.shutdownNow();
        progressReportExecutor.shutdownNow();
        if (negativeOpenConnectionCountSpotted.get()) {
            Assert.fail((String)"Negative value spotted for open connection count");
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test(groups={"short"})
    public void should_not_create_connections_if_zero_core_connections_and_reused_connection_on_reconnect() throws Exception {
        int reconnectInterval = 1000;
        Cluster cluster = this.createClusterBuilder().withReconnectionPolicy((ReconnectionPolicy)new ConstantReconnectionPolicy(1000L)).build();
        Host.StateListener stateListener = (Host.StateListener)Mockito.mock(Host.StateListener.class);
        try {
            Connection.Factory factory;
            cluster.init();
            Assertions.assertThat(cluster).hasOpenControlConnection();
            cluster.manager.connectionFactory = factory = (Connection.Factory)Mockito.spy((Object)cluster.manager.connectionFactory);
            HostConnectionPool pool = this.createPool(cluster, 0, 2);
            cluster.register(stateListener);
            cluster.manager.triggerOnDown(pool.host, true);
            Thread.sleep(reconnectInterval * 3);
            ((Host.StateListener)Mockito.verify((Object)stateListener, (VerificationMode)Mockito.times((int)1))).onUp(pool.host);
            Assertions.assertThat((List)pool.connections).hasSize(0);
            Assertions.assertThat(cluster).host(1).hasState(Host.State.UP);
            Assertions.assertThat(cluster).hasOpenControlConnection();
        }
        finally {
            cluster.close();
        }
    }

    private String generateJava349InsertStatement() {
        int i;
        StringBuilder sb = new StringBuilder("INSERT INTO Java349 (mykey");
        for (i = 0; i < 1000; ++i) {
            sb.append(", column").append(i);
        }
        sb.append(") VALUES (?");
        for (i = 0; i < 1000; ++i) {
            sb.append(", ").append(i);
        }
        sb.append(");");
        PrimingRequest preparedStatementPrime = PrimingRequest.preparedStatementBuilder().withQuery(sb.toString()).withThen(PrimingRequest.then().withVariableTypes(new CqlType[]{PrimitiveType.INT})).build();
        this.primingClient.prime(preparedStatementPrime);
        return sb.toString();
    }

    static class TestExecutorService
    extends ForwardingListeningExecutorService {
        private final ListeningExecutorService delegate;
        private final Semaphore semaphore = new Semaphore(0);

        TestExecutorService(ListeningExecutorService delegate) {
            this.delegate = delegate;
        }

        protected ListeningExecutorService delegate() {
            return this.delegate;
        }

        public void reset() {
            this.semaphore.drainPermits();
        }

        public void blockUntilNextTaskCompleted() throws InterruptedException {
            this.semaphore.tryAcquire(1, 1L, TimeUnit.MINUTES);
        }

        public ListenableFuture<?> submit(Runnable task) {
            ListenableFuture future = super.submit(task);
            GuavaCompatibility.INSTANCE.addCallback(future, (FutureCallback)new FutureCallback<Object>(){

                public void onSuccess(Object result) {
                    TestExecutorService.this.semaphore.release(1);
                }

                public void onFailure(Throwable t) {
                    TestExecutorService.this.semaphore.release(1);
                }
            });
            return future;
        }
    }

    static class MockRequest
    implements Connection.ResponseCallback {
        final ListenableFuture<Connection> connectionFuture;
        final ListenableFuture<Connection.ResponseHandler> requestInitialized;
        private volatile Connection.ResponseHandler responseHandler;
        final AtomicReference<State> state = new AtomicReference<State>(State.START);

        static MockRequest send(HostConnectionPool pool) {
            return MockRequest.send(pool, 0);
        }

        static MockRequest send(HostConnectionPool pool, int maxQueueSize) throws ConnectionException, BusyConnectionException {
            return MockRequest.send(pool, 5000, maxQueueSize);
        }

        static MockRequest send(HostConnectionPool pool, int timeoutMillis, int maxQueueSize) throws ConnectionException, BusyConnectionException {
            return new MockRequest(pool, timeoutMillis, maxQueueSize);
        }

        private static List<MockRequest> sendMany(int count, HostConnectionPool pool) throws ConnectionException {
            return MockRequest.sendMany(count, pool, 0);
        }

        private static List<MockRequest> sendMany(int count, HostConnectionPool pool, int maxQueueSize) throws ConnectionException {
            ArrayList requests = Lists.newArrayList();
            for (int i = 0; i < count; ++i) {
                MockRequest request = MockRequest.send(pool, maxQueueSize);
                requests.add(request);
            }
            return requests;
        }

        private static void completeMany(int count, List<MockRequest> requests) {
            Iterator<MockRequest> requestIt = requests.iterator();
            for (int i = 0; i < count && requestIt.hasNext(); ++i) {
                MockRequest request = requestIt.next();
                request.simulateSuccessResponse();
                requestIt.remove();
            }
        }

        private static void completeAll(List<MockRequest> requests) {
            for (MockRequest request : requests) {
                request.simulateSuccessResponse();
            }
        }

        private MockRequest(HostConnectionPool pool, int timeoutMillis, int maxQueueSize) throws ConnectionException {
            this.connectionFuture = pool.borrowConnection((long)timeoutMillis, TimeUnit.MILLISECONDS, maxQueueSize);
            this.requestInitialized = GuavaCompatibility.INSTANCE.transform(this.connectionFuture, (Function)new Function<Connection, Connection.ResponseHandler>(){

                public Connection.ResponseHandler apply(Connection connection) {
                    MockRequest thisRequest = MockRequest.this;
                    thisRequest.responseHandler = new Connection.ResponseHandler(connection, -1L, (Connection.ResponseCallback)thisRequest, false);
                    connection.dispatcher.add(thisRequest.responseHandler);
                    return MockRequest.this.responseHandler;
                }
            });
        }

        void simulateSuccessResponse() {
            this.onSet(this.getConnection(), null, 0L, 0);
        }

        void simulateErrorResponse() {
            this.onException(this.getConnection(), null, 0L, 0);
        }

        void simulateTimeout() {
            if (this.onTimeout(this.getConnection(), 0L, 0)) {
                this.responseHandler.cancelHandler();
            }
        }

        Connection getConnection() {
            try {
                return (Connection)Uninterruptibles.getUninterruptibly(this.connectionFuture, (long)500L, (TimeUnit)TimeUnit.MILLISECONDS);
            }
            catch (ExecutionException e) {
                throw Throwables.propagate((Throwable)e.getCause());
            }
            catch (TimeoutException e) {
                Assert.fail((String)"Timed out getting connection");
                return null;
            }
        }

        public void onSet(Connection connection, Message.Response response, long latency, int retryCount) {
            Assertions.assertThat((boolean)this.connectionFuture.isDone()).isTrue();
            try {
                Assertions.assertThat((Object)Uninterruptibles.getUninterruptibly(this.connectionFuture)).isNotNull();
            }
            catch (ExecutionException e) {
                Throwables.propagate((Throwable)e.getCause());
            }
            if (this.state.compareAndSet(State.START, State.COMPLETED)) {
                connection.dispatcher.removeHandler(this.responseHandler, true);
                connection.release();
            }
        }

        public void onException(Connection connection, Exception exception, long latency, int retryCount) {
            if (this.state.compareAndSet(State.START, State.FAILED)) {
                connection.dispatcher.removeHandler(this.responseHandler, true);
                connection.release();
            }
        }

        public boolean onTimeout(Connection connection, long latency, int retryCount) {
            return this.state.compareAndSet(State.START, State.TIMED_OUT);
        }

        public Message.Request request() {
            return null;
        }

        public int retryCount() {
            return 0;
        }

        static enum State {
            START,
            COMPLETED,
            FAILED,
            TIMED_OUT;

        }
    }
}

