/*
 * Decompiled with CFR 0.152.
 */
package com.couchbase.lite;

import com.couchbase.lite.BlobKey;
import com.couchbase.lite.BlobStore;
import com.couchbase.lite.support.Base64;
import com.couchbase.lite.support.security.SymmetricKey;
import com.couchbase.lite.support.security.SymmetricKeyException;
import com.couchbase.lite.util.Log;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Locale;

public class BlobStoreWriter {
    private BlobStore store = null;
    private int length = 0;
    private BlobKey blobKey = null;
    private byte[] md5DigestResult = null;
    private MessageDigest sha1Digest = null;
    private MessageDigest md5Digest = null;
    private BufferedOutputStream outStream = null;
    private File tempFile = null;
    private SymmetricKey.Encryptor encryptor = null;

    public BlobStoreWriter(BlobStore store) {
        this.store = store;
        try {
            this.sha1Digest = MessageDigest.getInstance("SHA-1");
            this.sha1Digest.reset();
            this.md5Digest = MessageDigest.getInstance("MD5");
            this.md5Digest.reset();
        }
        catch (NoSuchAlgorithmException e) {
            throw new IllegalStateException(e);
        }
        try {
            this.openTempFile();
        }
        catch (IOException e) {
            throw new IllegalStateException(e);
        }
        SymmetricKey encryptionKey = store.getEncryptionKey();
        if (encryptionKey != null) {
            try {
                this.encryptor = encryptionKey.createEncryptor();
            }
            catch (SymmetricKeyException e) {
                throw new IllegalStateException(e);
            }
        }
    }

    private void openTempFile() throws IOException {
        File tempDir = this.store.tempDir();
        String filePrefix = "tmp";
        String fileExtension = ".blobtmp";
        this.tempFile = File.createTempFile(filePrefix, fileExtension, tempDir);
        this.outStream = new BufferedOutputStream(new FileOutputStream(this.tempFile));
    }

    public void appendData(byte[] data) throws IOException, SymmetricKeyException {
        if (data == null) {
            return;
        }
        this.appendData(data, 0, data.length);
    }

    public void appendData(byte[] data, int off, int len) throws IOException, SymmetricKeyException {
        if (data == null) {
            return;
        }
        this.length += len;
        this.sha1Digest.update(data, off, len);
        this.md5Digest.update(data, off, len);
        if (this.encryptor != null) {
            if ((data = this.encryptor.encrypt(data, off, len)) != null) {
                this.outStream.write(data, 0, data.length);
            }
        } else {
            this.outStream.write(data, off, len);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void appendInputStream(InputStream inputStream) throws IOException, SymmetricKeyException {
        byte[] buffer = new byte[1024];
        this.length = 0;
        try {
            int len;
            while ((len = inputStream.read(buffer)) != -1) {
                this.appendData(buffer, 0, len);
            }
        }
        finally {
            try {
                if (inputStream != null) {
                    inputStream.close();
                }
            }
            catch (IOException e) {
                Log.w("BlobStore", "Exception closing input stream", e);
            }
        }
    }

    public void finish() throws IOException, SymmetricKeyException {
        if (this.outStream != null) {
            if (this.encryptor != null) {
                this.outStream.write(this.encryptor.encrypt(null));
            }
            this.outStream.close();
            this.outStream = null;
            this.blobKey = new BlobKey(this.sha1Digest.digest());
            this.md5DigestResult = this.md5Digest.digest();
        }
    }

    public void cancel() {
        try {
            if (this.outStream != null) {
                this.outStream.close();
                this.outStream = null;
            }
            this.encryptor = null;
        }
        catch (IOException e) {
            Log.w("BlobStore", "Exception closing buffered output stream", e);
        }
        this.tempFile.delete();
    }

    public boolean install() {
        if (this.tempFile == null) {
            return true;
        }
        String destPath = this.store.getRawPathForKey(this.blobKey);
        File destPathFile = new File(destPath);
        if (this.tempFile.renameTo(destPathFile)) {
            this.tempFile = null;
        } else {
            this.cancel();
        }
        return true;
    }

    public String mD5DigestString() {
        String base64Md5Digest = Base64.encodeBytes(this.md5DigestResult);
        return String.format(Locale.ENGLISH, "md5-%s", base64Md5Digest);
    }

    public String sHA1DigestString() {
        String base64Sha1Digest = Base64.encodeBytes(this.blobKey.getBytes());
        return String.format(Locale.ENGLISH, "sha1-%s", base64Sha1Digest);
    }

    public int getLength() {
        return this.length;
    }

    public BlobKey getBlobKey() {
        return this.blobKey;
    }

    public String getFilePath() {
        return this.tempFile.getPath();
    }
}

