/*
 * Decompiled with CFR 0.152.
 */
package com.couchbase.lite;

import com.couchbase.lite.BlobKey;
import com.couchbase.lite.BlobStoreWriter;
import com.couchbase.lite.Context;
import com.couchbase.lite.CouchbaseLiteException;
import com.couchbase.lite.Misc;
import com.couchbase.lite.support.FileDirUtils;
import com.couchbase.lite.support.action.Action;
import com.couchbase.lite.support.action.ActionBlock;
import com.couchbase.lite.support.action.ActionException;
import com.couchbase.lite.support.security.SymmetricKey;
import com.couchbase.lite.support.security.SymmetricKeyException;
import com.couchbase.lite.util.Log;
import com.couchbase.lite.util.TextUtils;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStream;
import java.io.RandomAccessFile;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Set;

public class BlobStore {
    public static final String FILE_EXTENSION = ".blob";
    public static final String TMP_FILE_EXTENSION = ".blobtmp";
    public static final String TMP_FILE_PREFIX = "tmp";
    public static final String ENCRYPTION_ALGORITHM = "AES";
    public static final String ENCRYPTION_MARKER_FILENAME = "_encryption";
    private Context context;
    private String path;
    private SymmetricKey encryptionKey;
    private BlobStore tempStore;

    public BlobStore(Context context, String path, SymmetricKey encryptionKey) throws CouchbaseLiteException {
        this(context, path, encryptionKey, false);
    }

    public BlobStore(Context context, String path, SymmetricKey encryptionKey, boolean autoMigrate) throws CouchbaseLiteException {
        this.context = context;
        this.path = path;
        this.encryptionKey = encryptionKey;
        File directory = new File(path);
        if (directory.exists()) {
            if (!directory.isDirectory()) {
                throw new CouchbaseLiteException("BlobStore: Blobstore is not a directory", 592);
            }
            this.verifyExistingStore();
        } else {
            if (!directory.mkdirs()) {
                Log.w("Database", "BlobStore: Unable to make directory: %s", directory);
                throw new CouchbaseLiteException("Unable to create a blobstore", 592);
            }
            if (encryptionKey != null) {
                this.markEncrypted(true);
            }
        }
        if (autoMigrate) {
            BlobStore.migrateBlobstoreFilenames(directory);
        }
    }

    private void verifyExistingStore() throws CouchbaseLiteException {
        SymmetricKey encKey;
        File markerFile = new File(this.path, ENCRYPTION_MARKER_FILENAME);
        boolean isMarkerExists = markerFile.exists();
        String encryptionAlg = null;
        if (isMarkerExists) {
            try {
                encryptionAlg = TextUtils.read(markerFile);
            }
            catch (IOException e) {
                throw new CouchbaseLiteException(e.getCause(), 491);
            }
        }
        if (encryptionAlg != null) {
            if (this.encryptionKey == null) {
                Log.w("Database", "BlobStore: Opening encrypted blob-store without providing a key");
                throw new CouchbaseLiteException(401);
            }
            if (!encryptionAlg.equals(ENCRYPTION_ALGORITHM)) {
                Log.w("Database", "BlobStore: Blob store uses unrecognized encryption '" + encryptionAlg + '\'');
                throw new CouchbaseLiteException(401);
            }
        } else if (!isMarkerExists && (encKey = this.encryptionKey) != null) {
            Log.i("Database", "BlobStore: BlobStore should be encrypted; do it now...");
            this.encryptionKey = null;
            try {
                this.changeEncryptionKey(encKey);
            }
            catch (ActionException e) {
                throw new CouchbaseLiteException("Cannot change the attachment encryption key", (Throwable)e, 592);
            }
        }
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private void markEncrypted(boolean encrypted) throws CouchbaseLiteException {
        File markerFile = new File(this.path, ENCRYPTION_MARKER_FILENAME);
        if (encrypted) {
            try {
                TextUtils.write(ENCRYPTION_ALGORITHM, markerFile);
                if (markerFile.exists()) return;
                Log.w("Database", "BlobStore: Unable to save the encryption marker file into the blob store");
                return;
            }
            catch (IOException e) {
                Log.w("Database", "BlobStore: Unable to save the encryption marker file into the blob store");
                throw new CouchbaseLiteException(e.getCause(), 592);
            }
        } else {
            if (!markerFile.exists() || markerFile.delete()) return;
            Log.w("Database", "BlobStore: Unable to delete the encryption marker file in the blob store");
            throw new CouchbaseLiteException("Unable to delete the encryption marker file in the Blob-store", 592);
        }
    }

    public Action actionToChangeEncryptionKey(final SymmetricKey newKey) {
        File tempStoreDir;
        File[] blobs;
        Action action = new Action();
        final SymmetricKey oldKey = this.encryptionKey;
        File directory = new File(this.path);
        File[] files = null;
        if (directory.exists() && directory.isDirectory()) {
            files = directory.listFiles(new FilenameFilter(){

                @Override
                public boolean accept(File dir, String name) {
                    return name.endsWith(BlobStore.FILE_EXTENSION);
                }
            });
        }
        if ((blobs = files) == null || blobs.length == 0) {
            action.add(new ActionBlock(){

                @Override
                public void execute() throws ActionException {
                    Log.i("Database", "BlobStore: " + (newKey != null ? "encrypting" : "decrypting") + ' ' + BlobStore.this.path);
                    Log.i("Database", "BlobStore: **No blobs to copy; done.**");
                    BlobStore.this.encryptionKey = newKey;
                    try {
                        BlobStore.this.markEncrypted(newKey != null);
                    }
                    catch (CouchbaseLiteException e) {
                        throw new ActionException(e);
                    }
                }
            }, new ActionBlock(){

                @Override
                public void execute() throws ActionException {
                    BlobStore.this.encryptionKey = oldKey;
                }
            }, null);
            return action;
        }
        File tempDir = new File(this.context.getTempDir(), Misc.CreateUUID());
        File file = tempStoreDir = tempDir.mkdirs() ? tempDir : null;
        if (tempStoreDir != null) {
            tempStoreDir.deleteOnExit();
        }
        action.add(new ActionBlock(){

            @Override
            public void execute() throws ActionException {
                Log.i("Database", "BlobStore: " + (newKey != null ? "encrypting" : "decrypting") + ' ' + BlobStore.this.path);
                if (tempStoreDir == null) {
                    throw new ActionException("Cannot create a temporary directory");
                }
            }
        }, new ActionBlock(){

            @Override
            public void execute() throws ActionException {
                if (!FileDirUtils.deleteRecursive(tempStoreDir)) {
                    throw new ActionException("Cannot delete a temporary directory " + tempStoreDir.getAbsolutePath());
                }
            }
        }, null);
        action.add(new ActionBlock(){

            @Override
            public void execute() throws ActionException {
                try {
                    BlobStore.this.tempStore = new BlobStore(BlobStore.this.context, tempStoreDir.getAbsolutePath(), newKey);
                    BlobStore.this.tempStore.markEncrypted(newKey != null);
                }
                catch (CouchbaseLiteException e) {
                    throw new ActionException(e);
                }
            }
        }, null, null);
        action.add(new ActionBlock(){

            @Override
            public void execute() throws ActionException {
                for (File blob : blobs) {
                    InputStream readStream = null;
                    BlobStoreWriter writer = null;
                    try {
                        Log.i("Database", "BlobStore: Copying " + blob);
                        readStream = BlobStore.this.encryptionKey.decryptStream(new FileInputStream(blob));
                        writer = new BlobStoreWriter(BlobStore.this.tempStore);
                        writer.appendInputStream(readStream);
                        writer.finish();
                        writer.install();
                    }
                    catch (Exception e) {
                        if (writer != null) {
                            writer.cancel();
                        }
                        throw new ActionException(e);
                    }
                    finally {
                        if (writer != null) {
                            new File(BlobStore.this.tempStore.getRawPathForKey(writer.getBlobKey())).deleteOnExit();
                        }
                        try {
                            if (readStream != null) {
                                readStream.close();
                            }
                        }
                        catch (IOException iOException) {}
                    }
                }
            }
        }, null, null);
        action.add(Action.moveAndReplaceFile(tempStoreDir.getAbsolutePath(), this.path, this.context.getTempDir().getAbsolutePath()));
        action.add(new ActionBlock(){

            @Override
            public void execute() throws ActionException {
                BlobStore.this.encryptionKey = newKey;
            }
        }, new ActionBlock(){

            @Override
            public void execute() throws ActionException {
                BlobStore.this.encryptionKey = oldKey;
            }
        }, null);
        return action;
    }

    public void changeEncryptionKey(SymmetricKey newKey) throws ActionException {
        this.actionToChangeEncryptionKey(newKey).run();
    }

    protected static void migrateBlobstoreFilenames(File directory) {
        File[] files;
        if (directory == null || !directory.isDirectory()) {
            return;
        }
        for (File file : files = directory.listFiles(new FilenameFilter(){

            @Override
            public boolean accept(File dir, String name) {
                return name.endsWith(BlobStore.FILE_EXTENSION);
            }
        })) {
            String name = file.getName();
            name = name.substring(0, name.indexOf(FILE_EXTENSION));
            File dest = new File(directory, name.toUpperCase() + FILE_EXTENSION);
            file.renameTo(dest);
        }
    }

    public static BlobKey keyForBlob(byte[] data) {
        MessageDigest md;
        try {
            md = MessageDigest.getInstance("SHA-1");
        }
        catch (NoSuchAlgorithmException e) {
            Log.e("Database", "BlobStore: Error, SHA-1 getDigest is unavailable.");
            return null;
        }
        byte[] sha1hash = new byte[40];
        md.update(data, 0, data.length);
        sha1hash = md.digest();
        BlobKey result = new BlobKey(sha1hash);
        return result;
    }

    public static BlobKey keyForBlobFromFile(File file) {
        MessageDigest md;
        try {
            md = MessageDigest.getInstance("SHA-1");
        }
        catch (NoSuchAlgorithmException e) {
            Log.e("Database", "BlobStore: Error, SHA-1 getDigest is unavailable.");
            return null;
        }
        byte[] sha1hash = new byte[40];
        try {
            FileInputStream fis = new FileInputStream(file);
            byte[] buffer = new byte[65536];
            int lenRead = fis.read(buffer);
            while (lenRead > 0) {
                md.update(buffer, 0, lenRead);
                lenRead = fis.read(buffer);
            }
            fis.close();
        }
        catch (IOException e) {
            Log.e("Database", "BlobStore: Error reading tmp file to compute key");
        }
        sha1hash = md.digest();
        BlobKey result = new BlobKey(sha1hash);
        return result;
    }

    public String getBlobPathForKey(BlobKey key) {
        if (this.encryptionKey != null) {
            return null;
        }
        return this.getRawPathForKey(key);
    }

    public String getRawPathForKey(BlobKey key) {
        String lowercaseFilename;
        File lowercaseFile;
        String hexKey = BlobKey.convertToHex(key.getBytes());
        String filename = this.path + File.separator + hexKey + FILE_EXTENSION;
        File file = new File(filename);
        if (!file.exists() && (lowercaseFile = new File(lowercaseFilename = this.path + File.separator + hexKey.toLowerCase() + FILE_EXTENSION)).exists()) {
            filename = lowercaseFilename;
            Log.w("Database", "BlobStore: Found the older attachment blobstore file. Recommend to set auto migration:\n\tManagerOptions options = new ManagerOptions();\n\toptions.setAutoMigrateBlobStoreFilename(true);\n\tManager manager = new Manager(..., options);\n");
        }
        return filename;
    }

    public long getSizeOfBlob(BlobKey key) {
        String path = this.getRawPathForKey(key);
        File file = new File(path);
        return file.length();
    }

    public boolean getKeyForFilename(BlobKey outKey, String filename) {
        if (!filename.endsWith(FILE_EXTENSION)) {
            return false;
        }
        String rest = filename.substring(this.path.length() + 1, filename.length() - FILE_EXTENSION.length());
        outKey.setBytes(BlobKey.convertFromHex(rest));
        return true;
    }

    public boolean hasBlobForKey(BlobKey key) {
        if (key == null) {
            return false;
        }
        String path = this.getRawPathForKey(key);
        File file = new File(path);
        return file.isFile() && file.exists();
    }

    public byte[] blobForKey(BlobKey key) {
        if (key == null) {
            return null;
        }
        String path = this.getRawPathForKey(key);
        File file = new File(path);
        byte[] blob = null;
        try {
            BlobKey decodedKey;
            blob = BlobStore.getBytesFromFile(file);
            if (this.encryptionKey != null && blob != null) {
                blob = this.encryptionKey.decryptData(blob);
            }
            if (!key.equals(decodedKey = BlobStore.keyForBlob(blob))) {
                Log.w("Database", "BlobStore: Attachment " + path + " decoded incorrectly!");
                blob = null;
            }
        }
        catch (OutOfMemoryError e) {
            blob = null;
            Log.e("Database", "BlobStore: Error reading file", e);
        }
        catch (IOException e) {
            blob = null;
            Log.e("Database", "BlobStore: Error reading file", e);
        }
        catch (SymmetricKeyException e) {
            blob = null;
            Log.e("Database", "BlobStore: Attachment " + path + " decoded incorrectly!", e);
        }
        return blob;
    }

    public InputStream blobStreamForKey(BlobKey key) {
        String path = this.getRawPathForKey(key);
        File file = new File(path);
        if (file.canRead()) {
            try {
                FileInputStream is = new FileInputStream(file);
                if (this.encryptionKey != null) {
                    return this.encryptionKey.decryptStream(is);
                }
                return is;
            }
            catch (FileNotFoundException e) {
                Log.e("Database", "BlobStore: Unexpected file not found in blob store", e);
                return null;
            }
            catch (SymmetricKeyException e) {
                Log.e("Database", "BlobStore: Attachment stream " + path + " cannot be decoded!", e);
                return null;
            }
        }
        return null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean storeBlob(byte[] data, BlobKey outKey) {
        BlobKey newKey = BlobStore.keyForBlob(data);
        outKey.setBytes(newKey.getBytes());
        String path = this.getRawPathForKey(outKey);
        File file = new File(path);
        if (file.canRead()) {
            return true;
        }
        if (this.encryptionKey != null) {
            try {
                data = this.encryptionKey.encryptData(data);
            }
            catch (SymmetricKeyException e) {
                Log.w("Database", "BlobStore: Failed to encode data for " + path, e);
                return false;
            }
        }
        FileOutputStream fos = null;
        try {
            fos = new FileOutputStream(file);
            fos.write(data);
        }
        catch (FileNotFoundException e) {
            Log.e("Database", "BlobStore: Error opening file for output", e);
            boolean bl = false;
            return bl;
        }
        catch (IOException ioe) {
            Log.e("Database", "BlobStore: Error writing to file", ioe);
            boolean bl = false;
            return bl;
        }
        finally {
            if (fos != null) {
                try {
                    fos.close();
                }
                catch (IOException iOException) {}
            }
        }
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static byte[] getBytesFromFile(File file) throws IOException {
        FileInputStream is = null;
        try {
            int offset;
            is = new FileInputStream(file);
            long length = file.length();
            if (length > Integer.MAX_VALUE) {
                throw new OutOfMemoryError("The file is too large to read into a byte array.");
            }
            byte[] bytes = new byte[(int)length];
            if (bytes == null) {
                throw new OutOfMemoryError("The file is too large to read into a byte array.");
            }
            int numRead = 0;
            for (offset = 0; offset < bytes.length && (numRead = ((InputStream)is).read(bytes, offset, bytes.length - offset)) >= 0; offset += numRead) {
            }
            if (offset < bytes.length) {
                throw new IOException("Could not completely read file " + file.getName());
            }
            byte[] byArray = bytes;
            return byArray;
        }
        finally {
            if (is != null) {
                ((InputStream)is).close();
            }
        }
    }

    public Set<BlobKey> allKeys() {
        File[] contents;
        HashSet<BlobKey> result = new HashSet<BlobKey>();
        File file = new File(this.path);
        for (File attachment : contents = file.listFiles()) {
            if (attachment.isDirectory()) continue;
            BlobKey attachmentKey = new BlobKey();
            this.getKeyForFilename(attachmentKey, attachment.getPath());
            result.add(attachmentKey);
        }
        return result;
    }

    public int count() {
        File file = new File(this.path);
        File[] contents = file.listFiles();
        return contents.length;
    }

    public long totalDataSize() {
        File[] contents;
        long total = 0L;
        File file = new File(this.path);
        for (File attachment : contents = file.listFiles()) {
            total += attachment.length();
        }
        return total;
    }

    public int deleteBlobsExceptWithKeys(List<BlobKey> keysToKeep) {
        File[] contents;
        int numDeleted = 0;
        File file = new File(this.path);
        File[] fileArray = contents = file.listFiles();
        int n = fileArray.length;
        for (int i = 0; i < n; ++i) {
            BlobKey attachmentKey = new BlobKey();
            File attachment = fileArray[i];
            if (!this.getKeyForFilename(attachmentKey, attachment.getPath()) || keysToKeep.contains(attachmentKey)) continue;
            boolean result = attachment.delete();
            if (result) {
                ++numDeleted;
                continue;
            }
            Log.e("Database", "BlobStore: Error deleting attachment: %s", attachment);
        }
        return numDeleted;
    }

    public int deleteBlobs() {
        return this.deleteBlobsExceptWithKeys(new ArrayList<BlobKey>());
    }

    public boolean isGZipped(BlobKey key) {
        int magic = 0;
        String path = this.getRawPathForKey(key);
        File file = new File(path);
        if (file.canRead()) {
            try {
                RandomAccessFile raf = new RandomAccessFile(file, "r");
                magic = raf.read() & 0xFF | raf.read() << 8 & 0xFF00;
                raf.close();
            }
            catch (Throwable e) {
                Log.e("BlobStore", "Failed to read from RandomAccessFile", e);
            }
        }
        return magic == 35615;
    }

    public File tempDir() {
        File directory = new File(this.path);
        File tempDirectory = new File(directory, "temp_attachments");
        if (!tempDirectory.exists()) {
            tempDirectory.mkdirs();
        }
        if (!tempDirectory.isDirectory()) {
            throw new IllegalStateException(String.format(Locale.ENGLISH, "Unable to create directory for: %s", tempDirectory));
        }
        return tempDirectory;
    }

    public String getPath() {
        return this.path;
    }

    public SymmetricKey getEncryptionKey() {
        return this.encryptionKey;
    }

    public boolean isEncrypted() {
        return this.encryptionKey != null;
    }
}

