/*
 * Copyright 2021 Couchbase, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.couchbase.client.kotlin.codec

import com.couchbase.client.core.api.kv.CoreEncodedContent
import com.couchbase.client.core.deps.io.netty.buffer.ByteBufUtil
import com.couchbase.client.core.deps.io.netty.buffer.Unpooled
import com.couchbase.client.core.msg.kv.CodecFlags
import com.couchbase.client.core.msg.kv.CodecFlags.hasCommonFormat
import com.couchbase.client.kotlin.internal.toStringUtf8

/**
 * The encoded form of a KV value, with flags indicating the data type.
 *
 * @param flags a bitfield indicating the type of content.
 * See [CodecFlags] for standard values.
 */
public class Content(public val bytes: ByteArray, public val flags: Int) {
    public companion object {
        /**
         * A JSON document.
         */
        public fun json(value: ByteArray): Content = Content(value, CodecFlags.JSON_COMPAT_FLAGS)

        /**
         * A JSON document.
         */
        public fun json(value: String): Content = json(value.toByteArray())

        /**
         * Arbitrary binary content.
         */
        public fun binary(value: ByteArray): Content = Content(value, CodecFlags.BINARY_COMPAT_FLAGS)

        /**
         * Arbitrary textual content (UTF-8).
         */
        public fun string(value: String): Content = Content(value.toByteArray(), CodecFlags.STRING_COMPAT_FLAGS)

        /**
         * The serialized form of a Java object as generated by [java.io.ObjectOutputStream].
         */
        public fun serializedJavaObject(value: ByteArray): Content = Content(value, CodecFlags.SERIALIZED_COMPAT_FLAGS)
    }

    override fun toString(): String {
        val type = when {
            hasCommonFormat(flags, CodecFlags.JSON_COMMON_FLAGS) -> "json"
            hasCommonFormat(flags, CodecFlags.BINARY_COMMON_FLAGS) -> "binary"
            hasCommonFormat(flags, CodecFlags.STRING_COMMON_FLAGS) -> "string"

            // sample bucket documents look like this
            flags == 0 && bytes.isNotEmpty() && bytes[0].toInt().toChar() == '{' -> "json?"

            else -> "?"
        }

        return "Content(length=${bytes.size}, flags=$flags, type=${type} value=${convertContentToString()})"
    }

    // Don't implement CoreEncodedContent directly, since that would add a redundant "encoded" getter.
    internal fun toCore() = CoreEncodedContent.of(bytes, flags)

    private fun convertContentToString(): String {
        if (bytes.isEmpty()) return ""

        val printable = hasCommonFormat(flags, CodecFlags.JSON_COMMON_FLAGS)
                || hasCommonFormat(flags, CodecFlags.STRING_COMMON_FLAGS)
                || flags == 0 && bytes[0].toInt().toChar() == '{'
        return if (printable) {
            bytes.toStringUtf8()
        } else {
            val result = ByteBufUtil.prettyHexDump(Unpooled.wrappedBuffer(bytes))
            "\n$result\n"
        }
    }
}
