package co.saltpay.epos.request.handlerlib.receiver

import android.content.BroadcastReceiver
import android.content.Context
import android.content.Intent
import co.saltpay.epos.models.internal.InternalRequestModel
import co.saltpay.epos.models.internal.toInternalRequestModel
import co.saltpay.epos.models.mappers.INTENT_BACKGROUND_REQUEST_ACTION
import co.saltpay.epos.models.mappers.OldPayAppConfigRequest
import co.saltpay.epos.models.mappers.backgroundToForegroundRequestIntent
import co.saltpay.epos.models.mappers.isInternalRequest
import co.saltpay.epos.models.mappers.requestId
import co.saltpay.epos.models.mappers.requestingLibVersion
import co.saltpay.epos.models.mappers.requestingPackage
import co.saltpay.epos.models.mappers.toOldRequestModel
import co.saltpay.epos.models.mappers.toRequestModel
import co.saltpay.epos.models.request.BackgroundRequest
import co.saltpay.epos.models.request.ForegroundRequest
import co.saltpay.epos.models.request.PayAppConfigRequest
import co.saltpay.epos.request.handlerlib.asLog
import co.saltpay.epos.request.handlerlib.di.CompositionRootAccessor
import co.saltpay.epos.request.handlerlib.di.compositionRoot
import co.saltpay.epos.request.handlerlib.events.SenderBackgroundRequest
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch

internal class RequestBroadcastReceiver : BroadcastReceiver(), CompositionRootAccessor {

    private val publisher get() = compositionRoot.eventsPublisher
    private val processScope get() = compositionRoot.processScope
    private val logger get() = compositionRoot.logger

    override fun onReceive(context: Context?, intent: Intent) {
        if (intent.action != INTENT_BACKGROUND_REQUEST_ACTION) {
            return
        }

        val asyncFinisher = goAsync()
        if (intent.isInternalRequest) {
            asyncFinisher.handleInternalRequest(intent)
        } else {
            asyncFinisher.handleEposRequest(intent)
        }
    }

    private fun PendingResult.handleEposRequest(intent: Intent) {
        val backgroundRequest = kotlin.runCatching {
            when (val request = intent.toRequestModel()) {
                is ForegroundRequest -> {
                    // Version mismatch that we will try to recover
                    // Sender epos is considering this request as background, but we consider it
                    // foreground. Since this is only a signal for the communication libs
                    // we should be able to just redirect the intent to the activity
                    compositionRoot.app.startActivity(
                        intent.backgroundToForegroundRequestIntent(compositionRoot.app)
                    )
                    return
                }

                is BackgroundRequest -> request.toValidBackgroundRequest(intent)
            }
        }.recoverCatching {
            val recoveredOldRequest = when (val oldRequest = intent.toOldRequestModel()) {
                is OldPayAppConfigRequest -> PayAppConfigRequest(
                    requestId = oldRequest.requestId,
                    receiptHandlingMode = if (oldRequest.enableReceiptFullControl) {
                        PayAppConfigRequest.ReceiptHandlingMode.UnifiedEposWithoutDCC
                    } else {
                        PayAppConfigRequest.ReceiptHandlingMode.IncludedOnPaymentFlows
                    },
                    allowPayAppTips = oldRequest.allowPayAppTips
                )
            }

            recoveredOldRequest.toValidBackgroundRequest(intent).also {
                logger.i("RequestBroadcastReceiver", "Recovered request from old version. Recovered ${it.backgroundRequest}")
            }
        }.getOrElse {
            val request = SenderBackgroundRequest.Invalid(
                requestingPackage = intent.requestingPackage,
                requestingLibVersion = intent.requestingLibVersion,
                requestId = intent.requestId
            )

            logger.e(
                "RequestBroadcastReceiver",
                "Couldn't parse received intent. Reason: ${it.asLog()}, invalid request: $request"
            )

            processScope.launch(Dispatchers.IO) {
                publisher.publish(request)
                finish()
            }
            return
        }

        logger.i("RequestBroadcastReceiver", "Request received = $backgroundRequest")

        processScope.launch(Dispatchers.IO) {
            publisher.publish(backgroundRequest)
            finish()
        }
    }

    private fun BackgroundRequest.toValidBackgroundRequest(
        intent: Intent
    ) = SenderBackgroundRequest.Valid(
        backgroundRequest = this,
        requestingPackage = intent.requestingPackage,
        requestingLibVersion = intent.requestingLibVersion,
        requestId = intent.requestId
    )

    private fun PendingResult.handleInternalRequest(intent: Intent) {
        when (val internalRequest = intent.toInternalRequestModel()) {
            is InternalRequestModel.Log -> internalRequest.handle()
        }

        finish()
    }

    private fun InternalRequestModel.Log.handle() {
        val relatedRequestInfo = relatedRequestId?.let { " Related request id $it" } ?: ""
        val msg = "[EPOS_LOG]$relatedRequestInfo: $message"
        when (severity) {
            InternalRequestModel.Log.Severity.INFO -> logger.i(tag, msg)
            InternalRequestModel.Log.Severity.WARN -> logger.w(tag, msg)
            InternalRequestModel.Log.Severity.ERROR -> logger.e(tag, msg)
        }
    }
}