package co.saltpay.epos.request.handlerlib

import android.app.Application
import android.content.Intent
import co.saltpay.epos.models.mappers.*
import co.saltpay.epos.models.request.ForegroundRequest
import co.saltpay.epos.models.response.LibrariesVersionMismatch
import co.saltpay.epos.models.response.PayAppBusy
import co.saltpay.epos.models.response.ResponseModel
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import co.saltpay.epos.models.mappers.BuildConfig as MappersBuildConfig

internal class ForegroundRequestProcessor(
    private val processScope: CoroutineScope,
    private val app: Application,
    private val logger: Logger
) {

    private var processingRequestId: String? = null

    fun process(
        goToBackground: () -> Unit,
        intent: Intent,
        handlerProvider: (ForegroundRequest) -> ForegroundRequestHandler
    ) {
        when (val parsedRequest = intent.parse()) {
            is IntentParseResult.EposRequest -> parsedRequest.handleAndRespond(intent, handlerProvider)

            is IntentParseResult.ErrorParsingIntent -> {
                goToBackground()
                app.sendBroadcast(parsedRequest.response.toIntent(intent.requestingPackage, intent.requestingLibVersion))
            }

            is IntentParseResult.NonEposIntent -> Unit
        }
    }

    private fun IntentParseResult.EposRequest.handleAndRespond(
        intent: Intent,
        handlerProvider: (ForegroundRequest) -> ForegroundRequestHandler
    ) {
        logger.i("ForegroundRequestProcessor", "Request received = $request")
        processScope.launch {
            val response = handleRequest(handlerProvider, request, intent.requestingPackage)
                ?: return@launch

            app.sendBroadcast(
                response.toIntent(
                    intent.requestingPackage,
                    intent.requestingLibVersion
                )
            )
        }
    }

    private suspend fun handleRequest(
        handlerProvider: (ForegroundRequest) -> ForegroundRequestHandler,
        request: ForegroundRequest,
        requestingPackage: String
    ): ResponseModel? = withContext(Dispatchers.Main.immediate) {
        processingRequestId?.let { requestIdInProgress ->
            if (requestIdInProgress == request.requestId) {
                logger.i("ForegroundRequestProcessor", "Received request with id $requestIdInProgress same id we are processing now, ignoring..")
                return@withContext null
            }

            return@withContext PayAppBusy(
                requestId = request.requestId,
                requestIdInProgress = requestIdInProgress
            )
        }

        processingRequestId = request.requestId
        val response = handlerProvider(request).handle(request, requestingPackage)
        processingRequestId = null
        return@withContext response
    }

    private fun Intent.parse(): IntentParseResult {
        if (action != INTENT_FOREGROUND_REQUEST_ACTION) {
            return IntentParseResult.NonEposIntent
        }

        val foregroundRequest = kotlin.runCatching {
            toRequestModel() as ForegroundRequest
        }.getOrElse {
            val response = LibrariesVersionMismatch(
                requestId = requestId,
                eposLibVersion = requestingLibVersion,
                payAppLibVersion = MappersBuildConfig.VERSION_NAME
            )

            logger.e("ForegroundRequestProcessor", "Couldn't parse received intent. Reason: ${it.asLog()}, responding: $response")

            return IntentParseResult.ErrorParsingIntent(response)
        }

        return IntentParseResult.EposRequest(foregroundRequest)
    }

    private sealed interface IntentParseResult {
        object NonEposIntent: IntentParseResult

        data class ErrorParsingIntent(val response: ResponseModel): IntentParseResult

        data class EposRequest(val request: ForegroundRequest): IntentParseResult
    }

}