package co.saltpay.epos.request.handlerlib

import android.app.Activity
import android.app.Application
import android.content.Intent
import co.saltpay.epos.models.mappers.*
import co.saltpay.epos.models.request.ForegroundRequest
import co.saltpay.epos.models.response.LibrariesVersionMismatch
import co.saltpay.epos.models.response.PayAppBusy
import co.saltpay.epos.models.response.ResponseModel
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import co.saltpay.epos.models.mappers.BuildConfig as MappersBuildConfig

internal class ForegroundRequestProcessor(
    private val processScope: CoroutineScope,
    private val app: Application,
    private val logger: Logger
) {

    private var processingRequestId: String? = null

    fun process(
        activity: Activity,
        intent: Intent,
        handlerProvider: (ForegroundRequest) -> ForegroundRequestHandler
    ) {
        when (val result = intent.parse()) {
            is IntentParseResult.EposRequest -> {
                processScope.launch {
                    val response = handleRequest(activity, handlerProvider, result.request)
                    app.sendBroadcast(response.toIntent(intent.requestingPackage, intent.requestingLibVersion))
                }
            }

            is IntentParseResult.ErrorParsingIntent -> {
                activity.moveTaskToBack(true)
                app.sendBroadcast(result.response.toIntent(intent.requestingPackage, intent.requestingLibVersion))
            }

            is IntentParseResult.NonEposIntent -> Unit
        }
    }

    private suspend fun handleRequest(
        activity: Activity,
        handlerProvider: (ForegroundRequest) -> ForegroundRequestHandler,
        request: ForegroundRequest
    ): ResponseModel = withContext(Dispatchers.Main.immediate) {
        processingRequestId?.let {
            activity.moveTaskToBack(true)
            return@withContext PayAppBusy(
                requestId = request.requestId,
                requestIdInProgress = it
            )
        }

        processingRequestId = request.requestId
        val response = handlerProvider(request).handle(request)
        processingRequestId = null
        return@withContext response
    }

    private fun Intent.parse(): IntentParseResult {
        if (action != INTENT_FOREGROUND_REQUEST_ACTION) {
            return IntentParseResult.NonEposIntent
        }

        val foregroundRequest = kotlin.runCatching {
            toRequestModel() as ForegroundRequest
        }.getOrElse {
            val response = LibrariesVersionMismatch(
                requestId = requestId,
                eposLibVersion = requestingLibVersion,
                payAppLibVersion = MappersBuildConfig.VERSION_NAME
            )

            logger.e("ForegroundRequestProcessor", "Couldn't parse received intent. Reason: ${it.asLog()}, responding: $response")

            return IntentParseResult.ErrorParsingIntent(response)
        }

        return IntentParseResult.EposRequest(foregroundRequest)
    }

    private sealed interface IntentParseResult {
        object NonEposIntent: IntentParseResult

        data class ErrorParsingIntent(val response: ResponseModel): IntentParseResult

        data class EposRequest(val request: ForegroundRequest): IntentParseResult
    }

}