package cn.zzq0324.radish.web.handler;

import cn.zzq0324.radish.common.code.CommonStatusCode;
import cn.zzq0324.radish.common.dto.response.Response;
import cn.zzq0324.radish.common.exception.BusinessException;
import cn.zzq0324.radish.common.exception.PermissionDeniedException;
import cn.zzq0324.radish.common.exception.UnauthorizedException;
import cn.zzq0324.radish.web.constant.RadishWebConstants;
import java.util.Arrays;
import java.util.stream.Collectors;
import javax.servlet.http.HttpServletRequest;
import javax.validation.ValidationException;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;
import org.springframework.http.HttpStatus;
import org.springframework.web.HttpMediaTypeNotSupportedException;
import org.springframework.web.HttpRequestMethodNotSupportedException;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.MissingServletRequestParameterException;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestControllerAdvice;
import org.springframework.web.method.annotation.MethodArgumentTypeMismatchException;
import org.springframework.web.servlet.NoHandlerFoundException;

/**
 * 通用异常拦截处理
 *
 * @author: zzq0324
 * @since : 1.0.0
 */
@Slf4j
@RestControllerAdvice
@Order(Ordered.HIGHEST_PRECEDENCE)
@ConditionalOnProperty(value = RadishWebConstants.ENABLE_EXCEPTION_HANDLER, havingValue = "true", matchIfMissing =
    true)
public class RestExceptionHandler {

  /**
   * 参数类型不匹配，为便于前端查看错误信息，采用BaseResponse返回
   */
  @ExceptionHandler(MethodArgumentTypeMismatchException.class)
  public Response handleArgumentTypeMismatch(MethodArgumentTypeMismatchException e, HttpServletRequest request) {
    log.warn("argument type mismatch, request url: {}", getRequestURL(request), e);

    return Response.with(CommonStatusCode.BAD_REQUEST.args(e.getName(), e.getRequiredType()));
  }

  /**
   * 方法参数校验不通过
   */
  @ExceptionHandler(value = {MethodArgumentNotValidException.class})
  public Response handleMethodArgumentNotValidException(
      MethodArgumentNotValidException e, HttpServletRequest request) {
    log.warn("request url: {}, method argument not valid.", getRequestURL(request), e);

    // 拼接错误信息，多个以分号隔开
    String error = e.getBindingResult().getAllErrors().stream().map(s -> s.getDefaultMessage())
        .collect(Collectors.joining(";"));

    return Response.with(CommonStatusCode.BAD_REQUEST.args(error));
  }

  /**
   * 参数校验不通过
   */
  @ExceptionHandler(value = {ValidationException.class})
  public Response handleValidationException(ValidationException e, HttpServletRequest request) {
    log.warn("request url: {}, argument not valid.", getRequestURL(request), e);

    return Response.with(CommonStatusCode.BAD_REQUEST.args(e.getMessage()));
  }

  /**
   * 参数非法
   */
  @ExceptionHandler(value = {IllegalArgumentException.class})
  public Response handleIllegalArgumentException(IllegalArgumentException e, HttpServletRequest request) {
    log.warn("request url: {}, illegal argument: {}", getRequestURL(request), e);

    return Response.with(CommonStatusCode.BAD_REQUEST.args(e.getMessage()));
  }


  /**
   * 业务异常包装为BaseResponse返回
   */
  @ExceptionHandler(value = BusinessException.class)
  public Response handleBusinessException(BusinessException e, HttpServletRequest request) {
    log.warn("request url: {}, error code: {}", getRequestURL(request), e.getStatusCode(), e);

    return Response.with(e.getStatusCode());
  }

  /**
   * 未授权
   */
  @ExceptionHandler(value = UnauthorizedException.class)
  @ResponseStatus(HttpStatus.UNAUTHORIZED)
  public void handleUnauthorizedException(UnauthorizedException e, HttpServletRequest request) {
    log.warn("request url: {} unauthorized", getRequestURL(request), e);
  }

  @ExceptionHandler(value = PermissionDeniedException.class)
  @ResponseStatus(HttpStatus.FORBIDDEN)
  public void handlePermissionDeniedException(PermissionDeniedException e, HttpServletRequest request) {
    log.error("request url: {} forbidden", getRequestURL(request), e);
  }

  /**
   * 找不到对应的handler，HTTP状态码返回404
   */
  @ExceptionHandler(value = NoHandlerFoundException.class)
  @ResponseStatus(HttpStatus.NOT_FOUND)
  public void handleNoHandlerFoundException(HttpServletRequest request) {
    log.error("not handler found, request method: {}, url: {}.", request.getMethod(),
        request.getRequestURL());
  }

  /**
   * 请求方法不支持，例如仅支持post方法，调用了get方法。为便于前端查看错误信息，采用BaseResponse返回
   */
  @ExceptionHandler(HttpRequestMethodNotSupportedException.class)
  public Response handleRequestMethodNotSupported(HttpRequestMethodNotSupportedException e,
      HttpServletRequest request) {
    String supportedMethods = Arrays.toString(e.getSupportedMethods());

    log.warn("request method not supported. request url: {}, request method:{}, support method:{}",
        request.getRequestURL(), e.getMethod(), supportedMethods);

    return Response.with(CommonStatusCode.METHOD_NOT_SUPPORT.args(request.getMethod(), supportedMethods));
  }

  /**
   * Media类型不匹配，为便于前端查看错误信息，采用BaseResponse返回
   */
  @ExceptionHandler(HttpMediaTypeNotSupportedException.class)
  public Response handleMediaTypeNotSupported(HttpMediaTypeNotSupportedException e,
      HttpServletRequest request) {
    log.warn("media type not supported, request url: {}, content type: {}", request.getRequestURL(),
        request.getContentType(), e);

    return Response.with(CommonStatusCode.NOT_ACCEPTABLE);
  }

  /**
   * 请求参数缺失
   */
  @ExceptionHandler(value = {MissingServletRequestParameterException.class})
  public Response handleMissingServletRequestParameterException(
      MissingServletRequestParameterException e, HttpServletRequest request) {
    log.warn("request url: {}, request param missing exception.", request.getRequestURL(), e);

    return Response.with(CommonStatusCode.BAD_REQUEST.args(e.getMessage()));
  }

  /**
   * 未知异常处理
   */
  @ExceptionHandler(value = Throwable.class)
  @ResponseStatus(HttpStatus.INTERNAL_SERVER_ERROR)
  public void handleUnknownException(Throwable e, HttpServletRequest request) {
    log.error("unknown exception occur, request url: {}", getRequestURL(request), e);
  }

  private String getRequestURL(HttpServletRequest request) {
    return request.getRequestURL().toString();
  }
}
