package cn.zzq0324.radish.extension;

import java.util.Objects;
import lombok.Getter;
import org.springframework.util.Assert;
import org.springframework.util.ConcurrentLruCache;
import org.springframework.util.StringUtils;

/**
 * 业务场景
 *
 * @author  zzq0324
 * @since : 1.0.0
 */
@Getter
public class BusinessScenario {

  public static final String SPLIT = "#";
  public static final String IDENTIFY_TPL = "%s#%s#%s";

  // 增加场景缓存，提高查找效率，减少重复创建对象进而减少GC
  private static ConcurrentLruCache<String, BusinessScenario> CACHE = new ConcurrentLruCache<>(1024, scenarioStr -> {
    String[] arr = StringUtils.delimitedListToStringArray(scenarioStr, SPLIT);
    return new BusinessScenario(arr[0], arr[1], arr[2]);
  });

  private String business;
  private String useCase;
  private String scenario;

  private BusinessScenario(String business, String useCase, String scenario) {
    this.business = business;
    this.useCase = useCase;
    this.scenario = scenario;
  }

  public static BusinessScenario of(String business, String useCase, String scenario) {
    Assert.doesNotContain(business, SPLIT, "business must not contain #");
    Assert.doesNotContain(useCase, SPLIT, "useCase must not contain #");
    Assert.doesNotContain(scenario, SPLIT, "scenario must not contain #");

    return CACHE.get(getScenarioId(business, useCase, scenario));
  }

  public static BusinessScenario of(String business, String useCase) {
    return of(business, useCase, ExtensionConstant.DEFAULT_SCENARIO);
  }

  public static BusinessScenario of(String business) {
    return of(business, ExtensionConstant.DEFAULT_USE_CASE, ExtensionConstant.DEFAULT_SCENARIO);
  }

  public static BusinessScenario of() {
    return of(ExtensionConstant.DEFAULT_BUSINESS, ExtensionConstant.DEFAULT_USE_CASE,
        ExtensionConstant.DEFAULT_SCENARIO);
  }

  public static String getScenarioId(String business, String useCase, String scenario) {
    return String.format(IDENTIFY_TPL, business, useCase, scenario);
  }

  @Override
  public String toString() {
    return String.format(IDENTIFY_TPL, business, useCase, scenario);
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }

    if (o == null || getClass() != o.getClass()) {
      return false;
    }

    BusinessScenario that = (BusinessScenario) o;
    return business.equals(that.business) && useCase.equals(that.useCase) && scenario.equals(that.scenario);
  }

  @Override
  public int hashCode() {
    return Objects.hash(business, useCase, scenario);
  }
}
