package cn.zxinrun.excel.util;

import cn.hutool.core.convert.Convert;
import cn.zxinrun.core.util.DateUtils;
import cn.zxinrun.core.util.StringUtils;
import cn.zxinrun.core.util.file.FileTypeUtils;
import cn.zxinrun.core.util.file.ImageUtils;
import cn.zxinrun.core.util.reflect.ReflectUtils;
import cn.zxinrun.excel.adapter.ExcelHandlerAdapter;
import cn.zxinrun.excel.annotation.Excel;
import cn.zxinrun.excel.annotation.Excels;
import cn.zxinrun.excel.enums.ColumnTypeEnum;
import cn.zxinrun.excel.enums.TypeEnum;
import jakarta.servlet.http.HttpServletResponse;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.RegExUtils;
import org.apache.poi.ss.usermodel.*;
import org.apache.poi.ss.util.CellRangeAddress;
import org.apache.poi.ss.util.CellRangeAddressList;
import org.apache.poi.util.IOUtils;
import org.apache.poi.xssf.streaming.SXSSFWorkbook;
import org.apache.poi.xssf.usermodel.XSSFClientAnchor;
import org.apache.poi.xssf.usermodel.XSSFDataValidation;

import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.math.BigDecimal;
import java.text.DecimalFormat;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;

/**
 * Excel工具类
 * @author Mr.Zhang
 * @since 1.0
 **/
@Slf4j
public class ExcelUtils<T> {

    public static final String FORMULA_REGEX_STR = "=|-|\\+|@";

    public static final String[] FORMULA_STR = {"=", "-", "+", "@"};

    /**
     * Excel sheet最大行数，默认65536
     */
    public static final int sheetSize = 65536;

    /**
     * 工作表名称
     */
    private String sheetName;

    /**
     * 导出类型（EXPORT:导出数据；IMPORT：导入模板）
     */
    private TypeEnum type;

    /**
     * 工作薄对象
     */
    private Workbook wb;

    /**
     * 工作表对象
     */
    private Sheet sheet;

    /**
     * 样式列表
     */
    private Map<String, CellStyle> styles;

    /**
     * 导入导出数据列表
     */
    private List<T> list;

    /**
     * 注解列表
     */
    private List<Object[]> fields;

    /**
     * 当前行号
     */
    private int rownum;

    /**
     * 标题
     */
    private String title;

    /**
     * 最大高度
     */
    private short maxHeight;

    /**
     * 统计列表
     */
    private Map<Integer, Double> statistics = new HashMap<Integer, Double>();

    /**
     * 数字格式
     */
    private static final DecimalFormat DOUBLE_FORMAT = new DecimalFormat("######0.00");

    /**
     * 实体对象
     */
    public Class<T> clazz;

    /**
     * 构造方法，初始化工具类
     * @param clazz 实体类类型
     */
    public ExcelUtils(Class<T> clazz) {
        this.clazz = clazz;
    }

    /**
     * 初始化 Excel 工具类，设置数据源、工作表名称、标题和导出类型。
     * <ul>
     *   <li>如果数据源为空，则初始化为空列表</li>
     *   <li>根据类的字段创建 Excel 字段映射</li>
     *   <li>创建工作簿和标题</li>
     * </ul>
     * @param list      数据源
     * @param sheetName 工作表名称
     * @param title     标题
     * @param type      导出类型（EXPORT 或 IMPORT）
     */
    public void init(List<T> list, String sheetName, String title, TypeEnum type) {
        if (list == null) {
            list = new ArrayList<T>();
        }
        this.list = list;
        this.sheetName = sheetName;
        this.type = type;
        this.title = title;
        createExcelField();
        createWorkbook();
        createTitle();
    }

    /**
     * 创建 Excel 的标题行（第一行）。
     * <ul>
     *   <li>如果设置了标题（title 不为空），则在第一行创建标题单元格，并合并单元格区域。</li>
     *   <li>设置标题行的高度和样式。</li>
     *   <li>合并单元格范围为：第 0 行到第 0 行，第 0 列到最后一列。</li>
     * </ul>
     */
    public void createTitle() {
        if (StringUtils.isNotEmpty(title)) {
            Row titleRow = sheet.createRow(rownum == 0 ? rownum++ : 0);
            titleRow.setHeightInPoints(30);
            Cell titleCell = titleRow.createCell(0);
            titleCell.setCellStyle(styles.get("title"));
            titleCell.setCellValue(title);
            sheet.addMergedRegion(new CellRangeAddress(titleRow.getRowNum(), titleRow.getRowNum(), titleRow.getRowNum(),
                    this.fields.size() - 1));
        }
    }

    /**
     * 将 Excel 数据导入为对象列表。
     * <ul>
     *   <li>默认读取第一个工作表，标题占用一行</li>
     *   <li>根据 Excel 注解自动映射字段</li>
     * </ul>
     * @param is 输入流
     * @return 转换后的对象列表
     * @throws Exception 如果解析失败
     */
    public List<T> importExcel(InputStream is) throws Exception {
        return importExcel(is, 0);
    }

    /**
     * 将 Excel 数据导入为对象列表。
     * <ul>
     *   <li>可指定标题占用的行数（titleNum），从指定行数后开始读取数据</li>
     *   <li>根据 Excel 注解自动映射字段</li>
     * </ul>
     * @param is       输入流
     * @param titleNum 标题占用行数
     * @return 转换后的对象列表
     * @throws Exception 如果解析失败
     */
    public List<T> importExcel(InputStream is, int titleNum) throws Exception {
        return importExcel(StringUtils.EMPTY, is, titleNum);
    }

    /**
     * 将 Excel 数据导入为对象列表。
     * <ul>
     *   <li>可指定工作表名称（sheetName）和标题占用的行数（titleNum）</li>
     *   <li>从指定的 sheet 和标题行后开始读取数据</li>
     *   <li>根据 @Excel 注解自动映射字段</li>
     * </ul>
     * @param sheetName 工作表名称
     * @param is        输入流
     * @param titleNum  标题占用行数
     * @return 转换后的对象列表
     * @throws Exception 如果解析失败
     */
    public List<T> importExcel(String sheetName, InputStream is, int titleNum) throws Exception {
        this.type = TypeEnum.IMPORT;
        this.wb = WorkbookFactory.create(is);
        List<T> list = new ArrayList<T>();
        // 如果指定sheet名,则取指定sheet中的内容 否则默认指向第1个sheet
        Sheet sheet = StringUtils.isNotEmpty(sheetName) ? wb.getSheet(sheetName) : wb.getSheetAt(0);
        if (sheet == null) {
            throw new IOException("文件sheet不存在");
        }

        // 获取最后一个非空行的行下标，比如总行数为n，则返回的为n-1
        int rows = sheet.getLastRowNum();

        if (rows > 0) {
            // 定义一个map用于存放excel列的序号和field.
            Map<String, Integer> cellMap = new HashMap<String, Integer>();
            // 获取表头
            Row heard = sheet.getRow(titleNum);
            for (int i = 0; i < heard.getPhysicalNumberOfCells(); i++) {
                Cell cell = heard.getCell(i);
                if (cell != null) {
                    String value = this.getCellValue(heard, i).toString();
                    cellMap.put(value, i);
                } else {
                    cellMap.put(null, i);
                }
            }
            // 有数据时才处理 得到类的所有field.
            List<Object[]> fields = this.getFields();
            Map<Integer, Object[]> fieldsMap = new HashMap<Integer, Object[]>();
            for (Object[] objects : fields) {
                Excel attr = (Excel) objects[1];
                Integer column = cellMap.get(attr.name());
                if (column != null) {
                    fieldsMap.put(column, objects);
                }
            }
            for (int i = titleNum + 1; i <= rows; i++) {
                // 从第2行开始取数据,默认第一行是表头.
                Row row = sheet.getRow(i);
                // 判断当前行是否是空行
                if (isRowEmpty(row)) {
                    continue;
                }
                T entity = null;
                for (Map.Entry<Integer, Object[]> entry : fieldsMap.entrySet()) {
                    Object val = this.getCellValue(row, entry.getKey());

                    // 如果不存在实例则新建.
                    entity = (entity == null ? clazz.newInstance() : entity);
                    // 从map中得到对应列的field.
                    Field field = (Field) entry.getValue()[0];
                    Excel attr = (Excel) entry.getValue()[1];
                    // 取得类型,并根据对象类型设置值.
                    Class<?> fieldType = field.getType();
                    if (String.class == fieldType) {
                        String s = Convert.toStr(val);
                        if (org.apache.commons.lang3.StringUtils.endsWith(s, ".0")) {
                            val = org.apache.commons.lang3.StringUtils.substringBefore(s, ".0");
                        } else {
                            String dateFormat = field.getAnnotation(Excel.class).dateFormat();
                            if (StringUtils.isNotEmpty(dateFormat)) {
                                val = parseDateToStr(dateFormat, val);
                            } else {
                                val = Convert.toStr(val);
                            }
                        }
                    } else if ((Integer.TYPE == fieldType || Integer.class == fieldType) && StringUtils.isNumeric(Convert.toStr(val))) {
                        val = Convert.toInt(val);
                    } else if ((Long.TYPE == fieldType || Long.class == fieldType) && StringUtils.isNumeric(Convert.toStr(val))) {
                        val = Convert.toLong(val);
                    } else if (Double.TYPE == fieldType || Double.class == fieldType) {
                        val = Convert.toDouble(val);
                    } else if (Float.TYPE == fieldType || Float.class == fieldType) {
                        val = Convert.toFloat(val);
                    } else if (BigDecimal.class == fieldType) {
                        val = Convert.toBigDecimal(val);
                    } else if (Date.class == fieldType) {
                        if (val instanceof String) {
                            val = DateUtils.parseDate(val);
                        } else if (val instanceof Double) {
                            val = DateUtil.getJavaDate((Double) val);
                        }
                    } else if (Boolean.TYPE == fieldType || Boolean.class == fieldType) {
                        val = Convert.toBool(val, false);
                    }
                    String propertyName = field.getName();
                    if (StringUtils.isNotEmpty(attr.targetAttr())) {
                        propertyName = field.getName() + "." + attr.targetAttr();
                    } else if (StringUtils.isNotEmpty(attr.readConverterExp())) {
                        val = reverseByExp(Convert.toStr(val), attr.readConverterExp(), attr.separator());
                    } else if (!attr.handler().equals(ExcelHandlerAdapter.class)) {
                        val = dataFormatHandlerAdapter(val, attr);
                    }
                    ReflectUtils.invokeSetter(entity, propertyName, val);
                }
                list.add(entity);
            }
        }
        return list;
    }

    /**
     * 导出数据到 Excel 文件。
     * <ul>
     *   <li>初始化 Excel 工具类，设置数据源、工作表名称和导出类型</li>
     *   <li>将生成的 Excel 文件写入 HTTP 响应流，供客户端下载</li>
     * </ul>
     * @param response  HTTP 响应对象
     * @param list      数据源
     * @param sheetName 工作表名称
     */
    public void exportExcel(HttpServletResponse response, List<T> list, String sheetName) {
        exportExcel(response, list, sheetName, StringUtils.EMPTY);
    }

    /**
     * 导出数据到 Excel 文件，并设置标题。
     * <ul>
     *   <li>初始化 Excel 工具类，设置数据源、工作表名称、标题和导出类型</li>
     *   <li>将生成的 Excel 文件写入 HTTP 响应流，供客户端下载</li>
     * </ul>
     * @param response  HTTP 响应对象
     * @param list      数据源
     * @param sheetName 工作表名称
     * @param title     标题
     */
    public void exportExcel(HttpServletResponse response, List<T> list, String sheetName, String title) {
        response.setContentType("application/vnd.openxmlformats-officedocument.spreadsheetml.sheet");
        response.setCharacterEncoding("utf-8");
        this.init(list, sheetName, title, TypeEnum.EXPORT);
        exportExcel(response);
    }

    /**
     * 对list数据源将其里面的数据导入到excel表单
     * @param response  HTTP 响应对象
     * @param sheetName 工作表名称
     */
    public void importTemplateExcel(HttpServletResponse response, String sheetName) {
        importTemplateExcel(response, sheetName, StringUtils.EMPTY);
    }

    /**
     * 对list数据源将其里面的数据导入到excel表单
     * @param response  HTTP 响应对象
     * @param sheetName 工作表名称
     * @param title     标题
     */
    public void importTemplateExcel(HttpServletResponse response, String sheetName, String title) {
        response.setContentType("application/vnd.openxmlformats-officedocument.spreadsheetml.sheet");
        response.setCharacterEncoding("utf-8");
        this.init(null, sheetName, title, TypeEnum.IMPORT);
        exportExcel(response);
    }

    /**
     * 将当前工作簿内容写入 HTTP 响应流，实现 Excel 文件导出。
     * <ul>
     *   <li>写入数据到 sheet</li>
     *   <li>将生成的 Excel 文件写入响应流，供客户端下载</li>
     *   <li>异常时记录日志</li>
     *   <li>最后关闭工作簿资源</li>
     * </ul>
     * @param response HTTP 响应对象
     */
    public void exportExcel(HttpServletResponse response) {
        try {
            writeSheet();
            wb.write(response.getOutputStream());
        } catch (Exception e) {
            log.error("导出Excel异常{}", e.getMessage());
        } finally {
            IOUtils.closeQuietly(wb);
        }
    }

    /**
     * 写入数据到 Excel 工作表。
     * <ul>
     *   <li>根据数据量动态创建多个 Sheet，每个 Sheet 填充部分数据。</li>
     *   <li>为每个 Sheet 创建表头行并写入字段名称。</li>
     *   <li>如果为导出模式（EXPORT），则填充数据和统计行。</li>
     * </ul>
     */
    public void writeSheet() {
        // 取出一共有多少个sheet.
        int sheetNo = Math.max(1, (int) Math.ceil(list.size() * 1.0 / sheetSize));
        for (int index = 0; index < sheetNo; index++) {
            createSheet(sheetNo, index);

            // 产生一行
            Row row = sheet.createRow(rownum);
            int column = 0;
            // 写入各个字段的列头名称
            for (Object[] os : fields) {
                Excel excel = (Excel) os[1];
                this.createCell(excel, row, column++);
            }
            if (TypeEnum.EXPORT.equals(type)) {
                fillExcelData(index, row);
                addStatisticsRow();
            }
        }
    }

    /**
     * 填充 Excel 数据到当前 Sheet。
     * <ul>
     *   <li>根据 sheet 索引计算数据起止位置，分批写入数据。</li>
     *   <li>为每一行创建单元格，并根据字段注解设置值和样式。</li>
     *   <li>每个字段调用 addCell 方法写入单元格内容。</li>
     * </ul>
     * @param index 当前 Sheet 的索引
     * @param row   当前表头行对象
     */
    public void fillExcelData(int index, Row row) {
        int startNo = index * sheetSize;
        int endNo = Math.min(startNo + sheetSize, list.size());
        for (int i = startNo; i < endNo; i++) {
            row = sheet.createRow(i + 1 + rownum - startNo);
            // 得到导出对象.
            T vo = (T) list.get(i);
            int column = 0;
            for (Object[] os : fields) {
                Field field = (Field) os[0];
                Excel excel = (Excel) os[1];
                this.addCell(excel, row, vo, field, column++);
            }
        }
    }

    /**
     * 创建并返回 Excel 各类单元格样式的映射表。
     * <ul>
     *   <li>包括标题样式、数据样式、表头样式和统计行样式</li>
     *   <li>每种样式设置字体、对齐方式、边框、颜色等属性</li>
     *   <li>样式以字符串 key 存储，便于后续按需获取</li>
     * </ul>
     * @param wb 工作簿对象
     * @return 样式映射表，key 为样式名，value 为 CellStyle
     */
    private Map<String, CellStyle> createStyles(Workbook wb) {
        // 写入各条记录,每条记录对应excel表中的一行
        Map<String, CellStyle> styles = new HashMap<String, CellStyle>();
        CellStyle style = wb.createCellStyle();
        style.setAlignment(HorizontalAlignment.CENTER);
        style.setVerticalAlignment(VerticalAlignment.CENTER);
        Font titleFont = wb.createFont();
        titleFont.setFontName("Arial");
        titleFont.setFontHeightInPoints((short) 16);
        titleFont.setBold(true);
        style.setFont(titleFont);
        styles.put("title", style);

        style = wb.createCellStyle();
        style.setAlignment(HorizontalAlignment.CENTER);
        style.setVerticalAlignment(VerticalAlignment.CENTER);
        style.setBorderRight(BorderStyle.THIN);
        style.setRightBorderColor(IndexedColors.BLACK1.getIndex());
        style.setBorderLeft(BorderStyle.THIN);
        style.setLeftBorderColor(IndexedColors.BLACK1.getIndex());
        style.setBorderTop(BorderStyle.THIN);
        style.setTopBorderColor(IndexedColors.BLACK1.getIndex());
        style.setBorderBottom(BorderStyle.THIN);
        style.setBottomBorderColor(IndexedColors.BLACK1.getIndex());
        Font dataFont = wb.createFont();
        dataFont.setFontName("Arial");
        dataFont.setFontHeightInPoints((short) 10);
        style.setFont(dataFont);
        styles.put("data", style);

        style = wb.createCellStyle();
        style.cloneStyleFrom(styles.get("data"));
        style.setAlignment(HorizontalAlignment.CENTER);
        style.setVerticalAlignment(VerticalAlignment.CENTER);
//        style.setFillForegroundColor(IndexedColors.GREY_25_PERCENT.getIndex());// 浅灰
        style.setFillForegroundColor(IndexedColors.GREY_50_PERCENT.getIndex());// 深灰
        style.setFillPattern(FillPatternType.SOLID_FOREGROUND);

        Font headerFont = wb.createFont();
        headerFont.setFontName("Arial");
        headerFont.setFontHeightInPoints((short) 10);
        headerFont.setBold(true);
        headerFont.setColor(IndexedColors.WHITE.getIndex());
        style.setFont(headerFont);
        styles.put("header", style);

        style = wb.createCellStyle();
        style.setAlignment(HorizontalAlignment.CENTER);
        style.setVerticalAlignment(VerticalAlignment.CENTER);
        Font totalFont = wb.createFont();
        totalFont.setFontName("Arial");
        totalFont.setFontHeightInPoints((short) 10);
        style.setFont(totalFont);
        styles.put("total", style);

        return styles;
    }

    /**
     * 创建单元格并设置表头样式。
     * <ul>
     *   <li>根据 @Excel 注解设置单元格的列名、样式和数据校验。</li>
     *   <li>为表头行创建单元格，并应用 header 样式。</li>
     *   <li>如有下拉选项或提示信息，则为该列添加数据校验。</li>
     * </ul>
     * @param attr   Excel 注解
     * @param row    行对象
     * @param column 列索引
     * @return 创建的单元格对象
     */
    public Cell createCell(Excel attr, Row row, int column) {
        // 创建列
        Cell cell = row.createCell(column);
        // 写入列信息
        cell.setCellValue(attr.name());
        setDataValidation(attr, row, column);
        cell.setCellStyle(styles.get("header"));
        return cell;
    }

    /**
     * 设置单元格的值和样式。
     * <ul>
     *   <li>字符串类型：防止CSV注入，支持默认值和后缀</li>
     *   <li>数字类型：根据内容设置为整数或小数</li>
     *   <li>图片类型：插入图片到单元格</li>
     * </ul>
     * @param value 单元格值
     * @param attr  Excel注解
     * @param cell  单元格对象
     */
    public void setCellVo(Object value, Excel attr, Cell cell) {
        if (ColumnTypeEnum.STRING == attr.cellType()) {
            String cellValue = Convert.toStr(value);
            // 对于任何以表达式触发字符 =-+@开头的单元格，直接使用tab字符作为前缀，防止CSV注入。
            if (org.apache.commons.lang3.StringUtils.startsWithAny(cellValue, FORMULA_STR)) {
                cellValue = RegExUtils.replaceFirst(cellValue, FORMULA_REGEX_STR, "\t$0");
            }
            cell.setCellValue(cellValue != null ? attr.defaultValue() : cellValue + attr.suffix());
        } else if (ColumnTypeEnum.NUMERIC == attr.cellType()) {
            if (value != null) {
                cell.setCellValue(StringUtils.contains(Convert.toStr(value), ".") ? Convert.toDouble(value) : Convert.toInt(value));
            }
        } else if (ColumnTypeEnum.IMAGE == attr.cellType()) {
            ClientAnchor anchor = new XSSFClientAnchor(0, 0, 0, 0, (short) cell.getColumnIndex(), cell.getRow().getRowNum(), (short) (cell.getColumnIndex() + 1), cell.getRow().getRowNum() + 1);
            String imagePath = Convert.toStr(value);
            if (StringUtils.isNotEmpty(imagePath)) {
                byte[] data = ImageUtils.getImage(imagePath);
                getDrawingPatriarch(cell.getSheet()).createPicture(anchor,
                        cell.getSheet().getWorkbook().addPicture(data, getImageType(data)));
            }
        }
    }

    /**
     * 获取或创建指定 Sheet 的画布对象（Drawing）。
     * <ul>
     *   <li>如果当前 Sheet 尚未存在画布，则创建一个新的画布并返回</li>
     *   <li>如果已存在画布，则直接返回该画布</li>
     *   <li>画布用于插入图片、图形等对象</li>
     * </ul>
     * @param sheet 工作表对象
     * @return 画布对象（Drawing）
     */
    public static Drawing<?> getDrawingPatriarch(Sheet sheet) {
        if (sheet.getDrawingPatriarch() == null) {
            sheet.createDrawingPatriarch();
        }
        return sheet.getDrawingPatriarch();
    }

    /**
     * 根据图片字节内容判断图片类型，返回 POI 对应的图片类型常量。
     * <ul>
     *   <li>JPG 类型返回 {@link Workbook#PICTURE_TYPE_JPEG}</li>
     *   <li>PNG 类型返回 {@link Workbook#PICTURE_TYPE_PNG}</li>
     *   <li>其他类型默认返回 JPEG</li>
     * </ul>
     * @param value 图片字节数组
     * @return POI 图片类型常量
     */
    public int getImageType(byte[] value) {
        String type = FileTypeUtils.getFileExtendName(value);
        if ("JPG".equalsIgnoreCase(type)) {
            return Workbook.PICTURE_TYPE_JPEG;
        } else if ("PNG".equalsIgnoreCase(type)) {
            return Workbook.PICTURE_TYPE_PNG;
        }
        return Workbook.PICTURE_TYPE_JPEG;
    }

    /**
     * 设置单元格的数据校验和列宽。
     * <ul>
     *   <li>如果列名包含“注：”，设置固定宽度</li>
     *   <li>否则根据注解配置设置列宽</li>
     *   <li>如有下拉选项或提示信息，则为该列添加下拉框或提示框</li>
     * </ul>
     * @param attr   Excel注解
     * @param row    行对象
     * @param column 列索引
     */
    public void setDataValidation(Excel attr, Row row, int column) {
        if (attr.name().indexOf("注：") >= 0) {
            sheet.setColumnWidth(column, 6000);
        } else {
            // 设置列宽
            sheet.setColumnWidth(column, (int) ((attr.width() + 0.72) * 256));
        }
        if (StringUtils.isNotEmpty(attr.prompt()) || attr.combo().length > 0) {
            // 提示信息或只能选择不能输入的列内容.
            setPromptOrValidation(sheet, attr.combo(), attr.prompt(), 1, 100, column, column);
        }
    }

    /**
     * 添加单元格数据到指定行和列。
     * <ul>
     *   <li>根据 @Excel 注解配置设置单元格值和格式</li>
     *   <li>支持日期、表达式转换、自定义处理器、BigDecimal 精度等</li>
     *   <li>根据注解决定是否导出该列</li>
     *   <li>支持统计行数据累加</li>
     * </ul>
     * @param attr   Excel 注解
     * @param row    行对象
     * @param vo     数据对象
     * @param field  字段对象
     * @param column 列索引
     * @return 创建的单元格对象
     */
    public Cell addCell(Excel attr, Row row, T vo, Field field, int column) {
        Cell cell = null;
        try {
            // 设置行高
            row.setHeight(maxHeight);
            // 根据Excel中设置情况决定是否导出,有些情况需要保持为空,希望用户填写这一列.
            if (attr.isExport()) {
                // 创建cell
                cell = row.createCell(column);
                setDataCell(cell, attr);

                // 用于读取对象中的属性
                Object value = getTargetValue(vo, field, attr);
                String dateFormat = attr.dateFormat();
                String readConverterExp = attr.readConverterExp();
                String separator = attr.separator();
                if (StringUtils.isNotEmpty(dateFormat) && value != null) {
                    cell.setCellValue(parseDateToStr(dateFormat, value));
                } else if (StringUtils.isNotEmpty(readConverterExp) && value != null) {
                    cell.setCellValue(convertByExp(Convert.toStr(value), readConverterExp, separator));
                } else if (value instanceof BigDecimal && -1 != attr.scale()) {
                    cell.setCellValue((((BigDecimal) value).setScale(attr.scale(), attr.roundingMode())).toString());
                } else if (!attr.handler().equals(ExcelHandlerAdapter.class)) {
                    cell.setCellValue(dataFormatHandlerAdapter(value, attr));
                } else {
                    // 设置列类型
                    setCellVo(value, attr, cell);
                }
                addStatisticsData(column, Convert.toStr(value), attr);
            }
        } catch (Exception e) {
            log.error("创建单元格失败{}", e.getMessage());
        }
        return cell;
    }

    /**
     * 设置单元格的样式。
     * <ul>
     *   <li>根据 @Excel 注解配置对齐方式、字体、边框和字体颜色等样式属性</li>
     *   <li>应用于数据单元格，确保导出 Excel 时格式统一</li>
     * </ul>
     * @param cell  单元格对象
     * @param excel Excel 注解信息
     */
    public void setDataCell(Cell cell, Excel excel) {
        CellStyle style = wb.createCellStyle();
        style.setAlignment(HorizontalAlignment.CENTER);
        style.setVerticalAlignment(VerticalAlignment.CENTER);
        style.setBorderRight(BorderStyle.THIN);
        style.setRightBorderColor(IndexedColors.GREY_50_PERCENT.getIndex());
        style.setBorderLeft(BorderStyle.THIN);
        style.setLeftBorderColor(IndexedColors.GREY_50_PERCENT.getIndex());
        style.setBorderTop(BorderStyle.THIN);
        style.setTopBorderColor(IndexedColors.GREY_50_PERCENT.getIndex());
        style.setBorderBottom(BorderStyle.THIN);
        style.setBottomBorderColor(IndexedColors.GREY_50_PERCENT.getIndex());
        style.setAlignment(excel.align());
        Font dataFont = wb.createFont();
        dataFont.setFontName("Arial");
        dataFont.setFontHeightInPoints((short) 10);
        dataFont.setColor(excel.color().index);
        style.setFont(dataFont);
        cell.setCellStyle(style);
    }

    /**
     * 为指定的 Excel 区域设置单元格下拉框或提示框数据校验。
     * <ul>
     *   <li>如果 textlist 非空，则为单元格添加下拉选项。</li>
     *   <li>如果 promptContent 非空，则为单元格添加鼠标悬停提示。</li>
     *   <li>支持设置起止行列范围。</li>
     *   <li>自动处理 XSSF 兼容性问题。</li>
     * </ul>
     * @param sheet         工作表对象
     * @param textlist      下拉框选项内容数组
     * @param promptContent 提示内容
     * @param firstRow      起始行
     * @param endRow        结束行
     * @param firstCol      起始列
     * @param endCol        结束列
     */
    public void setPromptOrValidation(Sheet sheet, String[] textlist, String promptContent, int firstRow, int endRow,
                                      int firstCol, int endCol) {
        DataValidationHelper helper = sheet.getDataValidationHelper();
        DataValidationConstraint constraint = textlist.length > 0 ? helper.createExplicitListConstraint(textlist) : helper.createCustomConstraint("DD1");
        CellRangeAddressList regions = new CellRangeAddressList(firstRow, endRow, firstCol, endCol);
        DataValidation dataValidation = helper.createValidation(constraint, regions);
        if (StringUtils.isNotEmpty(promptContent)) {
            // 如果设置了提示信息则鼠标放上去提示
            dataValidation.createPromptBox("", promptContent);
            dataValidation.setShowPromptBox(true);
        }
        // 处理Excel兼容性问题
        if (dataValidation instanceof XSSFDataValidation) {
            dataValidation.setSuppressDropDownArrow(true);
            dataValidation.setShowErrorBox(true);
        } else {
            dataValidation.setSuppressDropDownArrow(false);
        }
        sheet.addValidationData(dataValidation);
    }

    /**
     * 根据表达式将属性值转换为对应的显示值。
     * <ul>
     *   <li>支持多值分隔（如 1,2 转换为 男,女）</li>
     *   <li>表达式格式如：0=男,1=女,2=未知</li>
     *   <li>单值直接匹配返回，分隔符存在时遍历匹配所有值</li>
     * </ul>
     * @param propertyValue 属性值
     * @param converterExp  转换表达式（如 0=男,1=女）
     * @param separator     多值分隔符
     * @return 转换后的显示值
     */
    public static String convertByExp(String propertyValue, String converterExp, String separator) {
        StringBuilder propertyString = new StringBuilder();
        String[] convertSource = converterExp.split(",");
        for (String item : convertSource) {
            String[] itemArray = item.split("=");
            if (StringUtils.containsAny(separator, propertyValue)) {
                for (String value : propertyValue.split(separator)) {
                    if (itemArray[0].equals(value)) {
                        propertyString.append(itemArray[1] + separator);
                        break;
                    }
                }
            } else {
                if (itemArray[0].equals(propertyValue)) {
                    return itemArray[1];
                }
            }
        }
        return org.apache.commons.lang3.StringUtils.stripEnd(propertyString.toString(), separator);
    }

    /**
     * 根据表达式将显示值反向转换为属性值。
     * <ul>
     *   <li>支持多值分隔（如 男,女 转换为 1,2）</li>
     *   <li>表达式格式如：0=男,1=女,2=未知</li>
     *   <li>单值直接匹配返回，分隔符存在时遍历匹配所有值</li>
     * </ul>
     * @param propertyValue 显示值
     * @param converterExp  转换表达式（如 0=男,1=女）
     * @param separator     多值分隔符
     * @return 反向转换后的属性值
     */
    public static String reverseByExp(String propertyValue, String converterExp, String separator) {
        StringBuilder propertyString = new StringBuilder();
        String[] convertSource = converterExp.split(",");
        for (String item : convertSource) {
            String[] itemArray = item.split("=");
            if (StringUtils.containsAny(separator, propertyValue)) {
                for (String value : propertyValue.split(separator)) {
                    if (itemArray[1].equals(value)) {
                        propertyString.append(itemArray[0] + separator);
                        break;
                    }
                }
            } else {
                if (itemArray[1].equals(propertyValue)) {
                    return itemArray[0];
                }
            }
        }
        return org.apache.commons.lang3.StringUtils.stripEnd(propertyString.toString(), separator);
    }

    /**
     * 使用自定义数据处理器格式化单元格数据。
     * <ul>
     *   <li>通过 @Excel 注解指定的 handler 类，对数据进行自定义格式化处理</li>
     *   <li>调用 handler 的 format 方法，传入原始值和参数数组</li>
     *   <li>格式化失败时记录日志，返回字符串类型的结果</li>
     * </ul>
     * @param value 原始数据值
     * @param excel Excel 注解信息（包含 handler 和参数）
     * @return 格式化后的字符串结果
     */
    public String dataFormatHandlerAdapter(Object value, Excel excel) {
        try {
            Object instance = excel.handler().newInstance();
            Method formatMethod = excel.handler().getMethod("format", new Class[]{Object.class, String[].class});
            value = formatMethod.invoke(instance, value, excel.args());
        } catch (Exception e) {
            log.error("不能格式化数据 " + excel.handler(), e.getMessage());
        }
        return Convert.toStr(value);
    }

    /**
     * 累加统计列的数据值，用于生成统计行。
     * <ul>
     *   <li>仅当字段注解设置 isStatistics=true 时才进行累加</li>
     *   <li>将指定列的数值累加到 statistics 映射中</li>
     *   <li>非数字内容自动忽略</li>
     * </ul>
     * @param index  列索引
     * @param text   当前单元格的文本值
     * @param entity Excel 注解对象（用于判断是否统计）
     */
    private void addStatisticsData(Integer index, String text, Excel entity) {
        if (entity != null && entity.isStatistics()) {
            Double temp = 0D;
            if (!statistics.containsKey(index)) {
                statistics.put(index, temp);
            }
            try {
                temp = Double.valueOf(text);
            } catch (NumberFormatException e) {
            }
            statistics.put(index, statistics.get(index) + temp);
        }
    }

    /**
     * 创建统计行并添加到当前工作表的末尾。
     * <ul>
     *   <li>如果存在需要统计的列，则在表格末尾新增一行“合计”</li>
     *   <li>将所有已统计的列的总和写入对应单元格</li>
     *   <li>写入后清空统计数据</li>
     * </ul>
     */
    public void addStatisticsRow() {
        if (statistics.size() > 0) {
            Row row = sheet.createRow(sheet.getLastRowNum() + 1);
            Set<Integer> keys = statistics.keySet();
            Cell cell = row.createCell(0);
            cell.setCellStyle(styles.get("total"));
            cell.setCellValue("合计");

            for (Integer key : keys) {
                cell = row.createCell(key);
                cell.setCellStyle(styles.get("total"));
                cell.setCellValue(DOUBLE_FORMAT.format(statistics.get(key)));
            }
            statistics.clear();
        }
    }

    /**
     * 获取对象指定字段的值，支持多级属性（通过 @Excel 注解的 targetAttr 配置）。
     * <ul>
     *   <li>如果 targetAttr 为空，直接返回字段值</li>
     *   <li>如果 targetAttr 包含点号，按多级属性依次获取嵌套对象的值</li>
     *   <li>否则获取指定属性名的值</li>
     * </ul>
     * @param vo    实体对象
     * @param field 字段
     * @param excel Excel 注解
     * @return 字段的最终属性值
     * @throws Exception 反射获取属性值失败时抛出
     */
    private Object getTargetValue(T vo, Field field, Excel excel) throws Exception {
        Object o = field.get(vo);
        if (StringUtils.isNotEmpty(excel.targetAttr())) {
            String target = excel.targetAttr();
            if (target.contains(".")) {
                String[] targets = target.split("[.]");
                for (String name : targets) {
                    o = getValue(o, name);
                }
            } else {
                o = getValue(o, target);
            }
        }
        return o;
    }

    /**
     * 以类的属性名获取对象的属性值（通过反射）。
     * <ul>
     *   <li>支持获取私有字段的值</li>
     *   <li>如果对象或属性名为空，直接返回原对象</li>
     *   <li>抛出异常时由调用方处理</li>
     * </ul>
     * @param o    目标对象
     * @param name 属性名
     * @return 属性值
     * @throws Exception 反射获取属性值失败时抛出
     */
    private Object getValue(Object o, String name) throws Exception {
        if (o != null && StringUtils.isNotEmpty(name)) {
            Class<?> clazz = o.getClass();
            Field field = clazz.getDeclaredField(name);
            field.setAccessible(true);
            o = field.get(o);
        }
        return o;
    }

    /**
     * 初始化 Excel 字段映射列表，并按排序规则排序。
     * <ul>
     *   <li>获取所有带有 @Excel 或 @Excels 注解的字段</li>
     *   <li>根据 @Excel 注解的 sort 属性对字段进行排序</li>
     *   <li>获取所有字段中配置的最大行高并设置为 maxHeight</li>
     * </ul>
     */
    private void createExcelField() {
        this.fields = getFields();
        this.fields = this.fields.stream().sorted(Comparator.comparing(objects -> ((Excel) objects[1]).sort())).collect(Collectors.toList());
        // 获取到 @Excel(height = 14) 注解上配置最大的高度信息
        this.maxHeight = getRowHeight();
    }

    /**
     * 获取类中所有带有 @Excel 或 @Excels 注解的字段信息。
     * <ul>
     *   <li>遍历父类和当前类的所有字段</li>
     *   <li>收集符合导入/导出类型的注解字段</li>
     *   <li>支持单个 @Excel 注解和多个 @Excel 注解（@Excels）</li>
     *   <li>返回字段和注解对象的数组列表</li>
     * </ul>
     * @return 字段和注解对象的列表，每项为 Object[]{Field, Excel}
     */
    public List<Object[]> getFields() {
        List<Object[]> fields = new ArrayList<Object[]>();
        List<Field> tempFields = new ArrayList<>();
        tempFields.addAll(Arrays.asList(clazz.getSuperclass().getDeclaredFields()));
        tempFields.addAll(Arrays.asList(clazz.getDeclaredFields()));
        for (Field field : tempFields) {
            // 单注解
            if (field.isAnnotationPresent(Excel.class)) {
                Excel attr = field.getAnnotation(Excel.class);
                if (attr != null && (attr.type() == TypeEnum.ALL || attr.type() == type)) {
                    field.setAccessible(true);
                    fields.add(new Object[]{field, attr});
                }
            }

            // 多注解
            if (field.isAnnotationPresent(Excels.class)) {
                Excels attrs = field.getAnnotation(Excels.class);
                Excel[] excels = attrs.value();
                for (Excel attr : excels) {
                    if (attr != null && (attr.type() == TypeEnum.ALL || attr.type() == type)) {
                        field.setAccessible(true);
                        fields.add(new Object[]{field, attr});
                    }
                }
            }
        }
        return fields;
    }

    /**
     * 根据注解获取最大行高
     * @return 最大行高度
     */
    public short getRowHeight() {
        double maxHeight = 0;
        for (Object[] os : this.fields) {
            Excel excel = (Excel) os[1];
            maxHeight = Math.max(maxHeight, excel.height());
        }
        return (short) (maxHeight * 20);
    }

    /**
     * 创建一个工作簿
     */
    public void createWorkbook() {
        // 创建一个带有 500 行内存窗口的流式工作簿，适合大数据量导出。
        this.wb = new SXSSFWorkbook(500);
        // 在工作簿中新建一个工作表。
        this.sheet = wb.createSheet();
        // 设置第一个工作表的名称为 sheetName。
        wb.setSheetName(0, sheetName);
        // 初始化并缓存单元格样式。
        this.styles = createStyles(wb);
    }

    /**
     * 创建工作表
     * 在需要多个 sheet 时（sheetNo 大于 1），并且当前是第 2 个及以后的 sheet（index 大于 0）
     * 为工作簿创建一个新工作表，设置表头标题，并将该 sheet 命名为 sheetName+index
     * 这样可以实现多 sheet 导出时，每个 sheet 都有独立的标题和名称。
     * @param sheetNo sheet数量
     * @param index   序号
     */
    public void createSheet(int sheetNo, int index) {
        // 设置工作表的名称.
        if (sheetNo > 1 && index > 0) {
            // 新建一个工作表并赋值给当前 sheet。
            this.sheet = wb.createSheet();
            // 在新建的 sheet 上创建标题行。
            this.createTitle();
            // 设置新建 sheet 的名称为 sheetName 加上当前索引。
            wb.setSheetName(index, sheetName + index);
        }
    }

    /**
     * 获取指定行和列的单元格值，并根据单元格类型进行转换。
     * <ul>
     *   <li>数值或公式类型：如为日期格式则转为 Date，否则按数字或整数处理</li>
     *   <li>字符串类型：返回字符串</li>
     *   <li>布尔类型：返回 boolean</li>
     *   <li>错误类型：返回错误码</li>
     *   <li>单元格为 null：返回空字符串</li>
     * </ul>
     * @param row    行对象
     * @param column 列号
     * @return 单元格的值（Object 类型）
     */
    public Object getCellValue(Row row, int column) {
        if (row == null) {
            return row;
        }
        Object val = "";
        try {
            Cell cell = row.getCell(column);
            if (cell != null) {
                if (cell.getCellType() == CellType.NUMERIC || cell.getCellType() == CellType.FORMULA) {
                    val = cell.getNumericCellValue();
                    if (DateUtil.isCellDateFormatted(cell)) {
                        val = DateUtil.getJavaDate((Double) val); // POI Excel 日期格式转换
                    } else {
                        if ((Double) val % 1 != 0) {
                            val = new BigDecimal(val.toString());
                        } else {
                            val = new DecimalFormat("0").format(val);
                        }
                    }
                } else if (cell.getCellType() == CellType.STRING) {
                    val = cell.getStringCellValue();
                } else if (cell.getCellType() == CellType.BOOLEAN) {
                    val = cell.getBooleanCellValue();
                } else if (cell.getCellType() == CellType.ERROR) {
                    val = cell.getErrorCellValue();
                }

            }
        } catch (Exception e) {
            return val;
        }
        return val;
    }

    /**
     * 判断指定的行是否为空行。
     * 如果该行不存在或所有单元格均为 BLANK 类型，则认为是空行。
     * @param row 需要判断的行
     * @return 如果是空行返回 true，否则返回 false
     */
    private boolean isRowEmpty(Row row) {
        if (row == null) {
            return true;
        }
        for (int i = row.getFirstCellNum(); i < row.getLastCellNum(); i++) {
            Cell cell = row.getCell(i);
            if (cell != null && cell.getCellType() != CellType.BLANK) {
                return false;
            }
        }
        return true;
    }

    /**
     * 根据指定格式将不同类型的日期对象格式化为字符串。
     * 支持 Date、LocalDateTime、LocalDate 类型，其他类型则直接调用 toString()。
     * <ul>
     *   <li>val 为 null 时，返回空字符串</li>
     *   <li>val 为 Date 类型，直接格式化</li>
     *   <li>val 为 LocalDateTime 或 LocalDate 类型，先转为 Date 再格式化</li>
     *   <li>其他类型，直接转为字符串</li>
     * </ul>
     * @param dateFormat 日期格式字符串
     * @param val        需要格式化的日期对象
     * @return 格式化后的日期字符串
     */
    public String parseDateToStr(String dateFormat, Object val) {
        if (val == null) {
            return "";
        }
        String str;
        if (val instanceof Date) {
            str = DateUtils.parseDateToStr(dateFormat, (Date) val);
        } else if (val instanceof LocalDateTime) {
            str = DateUtils.parseDateToStr(dateFormat, DateUtils.toDate((LocalDateTime) val));
        } else if (val instanceof LocalDate) {
            str = DateUtils.parseDateToStr(dateFormat, DateUtils.toDate((LocalDate) val));
        } else {
            str = val.toString();
        }
        return str;
    }

}
