package cn.zxinrun.core.util;

import cn.hutool.core.util.StrUtil;

/**
 * 字符串工具类
 * @author Mr.Zhang
 * @since 1.0
 **/
public class StringUtils extends StrUtil {

    // 百分号
    public static final String PERCENT_SIGN = "%";

    /**
     * 将一个字符串，去除前后的空格
     * @param str 原始字符串
     * @return 去除空格后的字符串
     */
    public static String trim(String str) {
        return (str == null ? "" : str.trim());
    }

    /**
     * 判断是否为空，并且不是空白字符
     * <pre>
     * StringUtil.hasText("")       =  false
     * StringUtil.hasText(" ")      =  false
     * StringUtil.hasText("abc")    =  true
     * StringUtil.hasText("123")    =  true
     * </pre>
     * @param str 要判断的value
     * @return 结果
     */
    public static boolean hasText(String str) {
        return (str != null && !str.isEmpty() && containsText(str));
    }

    private static boolean containsText(CharSequence str) {
        int strLen = str.length();
        for (int i = 0; i < strLen; i++) {
            if (!Character.isWhitespace(str.charAt(i))) {
                return true;
            }
        }
        return false;
    }

    /**
     * 是否包含字符串
     * <pre>
     * StringUtil.inStringIgnoreCase("abc", "abc aaa")      =  false
     * StringUtil.inStringIgnoreCase("abc", "abc", "aaa")   =  true
     * </pre>
     * @param str  验证字符串
     * @param strs 字符串组
     * @return 包含返回true
     */
    public static boolean inStringIgnoreCase(String str, String... strs) {
        if (str != null && strs != null) {
            for (String s : strs) {
                if (str.equalsIgnoreCase(trim(s))) {
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * 驼峰转下划线命名
     * <pre>
     * StringUtil.toUnderScoreCase("abcAbc")    =  "abc_abc"
     * </pre>
     * @param str 原始字符串
     * @return 转换好的字符串
     */
    public static String toUnderScoreCase(String str) {
        if (str == null) {
            return null;
        }
        StringBuilder sb = new StringBuilder();
        // 前置字符是否大写
        boolean preCharIsUpperCase = true;
        // 当前字符是否大写
        boolean curreCharIsUpperCase = true;
        // 下一字符是否大写
        boolean nexteCharIsUpperCase = true;
        for (int i = 0; i < str.length(); i++) {
            char c = str.charAt(i);
            if (i > 0) {
                preCharIsUpperCase = Character.isUpperCase(str.charAt(i - 1));
            } else {
                preCharIsUpperCase = false;
            }
            curreCharIsUpperCase = Character.isUpperCase(c);
            if (i < (str.length() - 1)) {
                nexteCharIsUpperCase = Character.isUpperCase(str.charAt(i + 1));
            }
            if (preCharIsUpperCase && curreCharIsUpperCase && !nexteCharIsUpperCase) {
                sb.append(UNDERLINE);
            } else if ((i != 0 && !preCharIsUpperCase) && curreCharIsUpperCase) {
                sb.append(UNDERLINE);
            }
            sb.append(Character.toLowerCase(c));
        }
        return sb.toString();
    }

    /**
     * 将下划线方式命名的字符串转换为大驼峰式。
     * 如果转换前的下划线大写方式命名的字符串为空，则返回空字符串。
     * <pre>
     * StringUtil.convertToCamelCase("abc_abc")   =  "AbcAbc"
     * StringUtil.convertToCamelCase("ABC_ABC")   =  "AbcAbc"
     * StringUtil.convertToCamelCase("Abc_aBC")   =  "AbcAbc"
     * StringUtil.convertToCamelCase("abc_ABC")   =  "AbcAbc"
     * </pre>
     * @param name 转换前的下划线大写方式命名的字符串
     * @return 转换后的驼峰式命名的字符串
     */
    public static String convertToCamelCase(String name) {
        StringBuilder result = new StringBuilder();
        // 快速检查
        if (name == null || name.isEmpty()) {
            // 没必要转换
            return "";
        } else if (!name.contains("_")) {
            // 不含下划线，仅将首字母大写
            return name.substring(0, 1).toUpperCase() + name.substring(1);
        }
        // 用下划线将原始字符串分割
        String[] camels = name.split("_");
        for (String camel : camels) {
            // 跳过原始字符串中开头、结尾的下换线或双重下划线
            if (camel.isEmpty()) {
                continue;
            }
            // 首字母大写
            result.append(camel.substring(0, 1).toUpperCase());
            result.append(camel.substring(1).toLowerCase());
        }
        return result.toString();
    }

    /**
     * 将下划线命名转换为小驼峰式命名法
     * <pre>
     * StringUtil.convertToCamelCase("abc_abc")     =  "abcAbc"
     * StringUtil.convertToCamelCase("name")        =  "name"
     * </pre>
     * @param str 原始字符串
     * @return 转换后的字符串
     */
    public static String toCamelCase(String str) {
        if (str == null) {
            return null;
        }
        str = str.toLowerCase();
        StringBuilder sb = new StringBuilder(str.length());
        boolean upperCase = false;
        for (int i = 0; i < str.length(); i++) {
            char c = str.charAt(i);
            if (String.valueOf(c).equals(UNDERLINE)) {
                upperCase = true;
            } else if (upperCase) {
                sb.append(Character.toUpperCase(c));
                upperCase = false;
            } else {
                sb.append(c);
            }
        }
        return sb.toString();
    }

    /**
     * 数字左边补齐0，使之达到指定长度。注意，如果数字转换为字符串后，长度大于size，则只保留 最后size个字符。
     * <pre>
     * StringUtil.padl(123, 5)      =  "00123"
     * StringUtil.padl(66, 5)       =  "00066"
     * StringUtil.padl(666666, 5)   =  "666666"
     * </pre>
     * @param num  数字对象
     * @param size 字符串指定长度
     * @return 返回数字的字符串格式，该字符串为指定长度。
     */
    public static String padl(final Number num, final int size) {
        return padl(num.toString(), size, '0');
    }

    /**
     * 字符串左补齐。如果原始字符串s长度大于size，则只保留最后size个字符。
     * <pre>
     * StringUtil.padl("123", 5, 'a')      =  "aa123"
     * StringUtil.padl("66", 5, 'a')       =  "aaa66"
     * StringUtil.padl("666666", 5, 'a')   =  "666666"
     * </pre>
     * @param s    原始字符串
     * @param size 字符串指定长度
     * @param c    用于补齐的字符
     * @return 返回指定长度的字符串，由原字符串左补齐或截取得到。
     */
    public static String padl(final String s, final int size, final char c) {
        final StringBuilder sb = new StringBuilder(size);
        if (s != null) {
            final int len = s.length();
            if (s.length() <= size) {
                for (int i = size - len; i > 0; i--) {
                    sb.append(c);
                }
                sb.append(s);
            } else {
                return s.substring(len - size, len);
            }
        } else {
            for (int i = size; i > 0; i--) {
                sb.append(c);
            }
        }
        return sb.toString();
    }

}
