package cn.zxinrun.core.util;

import cn.zxinrun.core.constant.CommonConstants;
import lombok.Data;

import java.io.Serializable;

/**
 * 通用请求返回主体
 * @author Mr.Zhang
 * @since 1.0
 **/
@Data
public class Result<T> implements Serializable {

    private static final long serialVersionUID = 1L;

    // 状态码
    private int code;
    // 提示信息
    private String msg;
    // 返回数据
    private T data;

    /**
     * 成功
     * @param <T> 指定的类型
     * @return Result
     */
    public static <T> Result<T> success() {
        return restResult(null, CommonConstants.SUCCESS, CommonConstants.SUCCESS_TEXT);
    }

    /**
     * 成功
     * @param data 返回数据
     * @param <T>  指定的类型
     * @return Result
     */
    public static <T> Result<T> success(T data) {
        return restResult(data, CommonConstants.SUCCESS, CommonConstants.SUCCESS_TEXT);
    }

    /**
     * 成功
     * @param data 返回数据
     * @param msg  提示信息
     * @param <T>  指定的类型
     * @return Result
     */
    public static <T> Result<T> success(T data, String msg) {
        return restResult(data, CommonConstants.SUCCESS, msg);
    }

    /**
     * 失败
     * @param <T> 指定的类型
     * @return Result
     */
    public static <T> Result<T> failed() {
        return restResult(null, CommonConstants.FAIL, CommonConstants.FAIL_TEXT);
    }

    /**
     * 失败
     * @param msg 提示信息
     * @param <T> 指定的类型
     * @return Result
     */
    public static <T> Result<T> failed(String msg) {
        return restResult(null, CommonConstants.FAIL, msg);
    }

    /**
     * 失败
     * @param data 返回数据
     * @param <T>  指定的类型
     * @return Result
     */
    public static <T> Result<T> failed(T data) {
        return restResult(data, CommonConstants.FAIL, CommonConstants.FAIL_TEXT);
    }

    /**
     * 失败
     * @param data 返回数据
     * @param msg  提示信息
     * @param <T>  指定的类型
     * @return Result
     */
    public static <T> Result<T> failed(T data, String msg) {
        return restResult(data, CommonConstants.FAIL, msg);
    }

    /**
     * 自定义返回
     * @param data 返回数据
     * @param code 状态码
     * @param msg  提示信息
     * @param <T>  指定的类型
     * @return Result
     */
    public static <T> Result<T> result(T data, int code, String msg) {
        return restResult(data, code, msg);
    }

    /**
     * 自定义返回
     * @param data 返回数据
     * @param code 状态码
     * @param msg  提示信息
     * @param <T>  指定的类型
     * @return Result
     */
    public static <T> Result<T> restResult(T data, int code, String msg) {
        Result<T> apiResult = new Result<>();
        apiResult.setCode(code);
        apiResult.setData(data);
        apiResult.setMsg(msg);
        return apiResult;
    }

    /**
     * 请求是否成功
     * @return true:成功 false:失败
     */
    public Boolean isSuccess() {
        return (this.code == CommonConstants.SUCCESS);
    }

}
