package cn.zxinrun.core.util;

import cn.hutool.core.date.DateUtil;
import org.apache.commons.lang3.time.DateFormatUtils;

import java.lang.management.ManagementFactory;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.time.*;
import java.util.Date;

/**
 * 日期工具类
 * @author Mr.Zhang
 * @since 1.0
 **/
public class DateUtils extends DateUtil {

    // 纯年份
    public static final String YYYY = "yyyy";
    // 年份-月份
    public static final String YYYY_MM = "yyyy-MM";
    // 年份-月份-日期
    public static final String YYYY_MM_DD = "yyyy-MM-dd";
    // 年月日时分秒
    public static final String YYYYMMDDHHMMSS = "yyyyMMddHHmmss";
    // 年-月-日 时-分-秒
    public static final String YYYY_MM_DD_HH_MM_SS = "yyyy-MM-dd HH:mm:ss";

    private static final String[] PARSE_PATTERNS = {
            "yyyy-MM-dd", "yyyy-MM-dd HH:mm:ss", "yyyy-MM-dd HH:mm", "yyyy-MM",
            "yyyy/MM/dd", "yyyy/MM/dd HH:mm:ss", "yyyy/MM/dd HH:mm", "yyyy/MM",
            "yyyy.MM.dd", "yyyy.MM.dd HH:mm:ss", "yyyy.MM.dd HH:mm", "yyyy.MM"
    };

    /**
     * SimpleDateFormat 是线程不安全的类，一般不要定义为 static 变量，如果定义为 static，必须加锁
     */
    private static final ThreadLocal<DateFormat> df = new ThreadLocal<DateFormat>() {
        @Override
        protected DateFormat initialValue() {
            return new SimpleDateFormat("yyyy-MM-dd");
        }
    };

    /**
     * 获取当前Date型日期
     * @return 当前日期
     */
    public static Date getNowDate() {
        return new Date();
    }

    /**
     * 获取当前日期
     * @return 返回 yyyy-MM-dd 格式的当前日期
     */
    public static String getDate() {
        return dateTimeNow(YYYY_MM_DD);
    }

    /**
     * 获取当前时间
     * @return 返回 yyyy-MM-dd HH:mm:ss 格式的当前时间
     */
    public static String getTime() {
        return dateTimeNow(YYYY_MM_DD_HH_MM_SS);
    }

    /**
     * 获取当前时间
     * @return 返回 yyyyMMddHHmmss 格式的当前时间
     */
    public static String dateTimeNow() {
        return dateTimeNow(YYYYMMDDHHMMSS);
    }

    /**
     * 将当前日期或时间，格式化某种字符形式
     * @param format 日期格式字符串
     * @return 格式化后的当前日期
     */
    public static String dateTimeNow(final String format) {
        return parseDateToStr(format, new Date());
    }

    /**
     * 将某个日期格式化成 yyyy-MM-dd
     * @param date 日期/时间
     * @return 格式化后的日期字符串
     */
    public static String dateTime(final Date date) {
        return parseDateToStr(YYYY_MM_DD, date);
    }

    /**
     * 将某个日期格式化成指定的字符串格式
     * @param format 格式化字符串
     * @param date   日期/时间
     * @return 格式化后的日期字符串
     */
    public static String parseDateToStr(final String format, final Date date) {
        return new SimpleDateFormat(format).format(date);
    }

    /**
     * 将当前日期格式化成 yyyy/MM/dd
     * @return 日期路径 即年/月/日 如2018/08/08
     */
    public static String datePath() {
        Date now = new Date();
        return DateFormatUtils.format(now, "yyyy/MM/dd");
    }

    /**
     * 将当前日期格式化成 yyyyMMdd
     * @return 日期路径 即年/月/日 如20180808
     */
    public static String dateTime() {
        Date now = new Date();
        return DateFormatUtils.format(now, "yyyyMMdd");
    }

    /**
     * 日期型字符串转化为日期
     * @param str 日期
     * @return 转换后的日期
     */
    public static Date parseDate(Object str) {
        if (str == null) {
            return null;
        }
        return parseDate(str.toString());
    }

    /**
     * 获取服务器启动时间
     * @return 返回服务器启动的时间
     */
    public static Date getServerStartDate() {
        long time = ManagementFactory.getRuntimeMXBean().getStartTime();
        return new Date(time);
    }

    /**
     * 计算两个时间差
     * @param endDate 结束日期
     * @param nowDate 当前日期
     * @return xx天xx小时xx分钟 格式的字符串
     */
    public static String getDatePoor(Date endDate, Date nowDate) {
        long nd = 1000 * 24 * 60 * 60;
        long nh = 1000 * 60 * 60;
        long nm = 1000 * 60;
        // long ns = 1000;
        // 获得两个时间的毫秒时间差异
        long diff = endDate.getTime() - nowDate.getTime();
        // 计算差多少天
        long day = diff / nd;
        // 计算差多少小时
        long hour = diff % nd / nh;
        // 计算差多少分钟
        long min = diff % nd % nh / nm;
        // 计算差多少秒//输出结果
        // long sec = diff % nd % nh % nm / ns;
        return day + "天" + hour + "小时" + min + "分钟";
    }

    /**
     * 将 LocalDateTime 类型转换为 Date 类型
     * @param temporalAccessor LocalDateTime 格式的日期
     * @return 日期格式
     */
    public static Date localDateTimeToDate(LocalDateTime temporalAccessor) {
        ZonedDateTime zdt = temporalAccessor.atZone(ZoneId.systemDefault());
        return Date.from(zdt.toInstant());
    }

    /**
     * 将 LocalDate 类型转换为 Date 类型
     * @param temporalAccessor LocalDate 格式的日期
     * @return 日期格式
     */
    public static Date localDateToDate(LocalDate temporalAccessor) {
        LocalDateTime localDateTime = LocalDateTime.of(temporalAccessor, LocalTime.of(0, 0, 0));
        return localDateTimeToDate(localDateTime);
    }

    /**
     * 增加 LocalDateTime 转换 Date
     * @param temporalAccessor LocalDateTime 格式的日期
     * @return Date
     */
    public static Date toDate(LocalDateTime temporalAccessor) {
        ZonedDateTime zdt = temporalAccessor.atZone(ZoneId.systemDefault());
        return Date.from(zdt.toInstant());
    }

    /**
     * 增加 LocalDate 转换 Date
     * @param temporalAccessor LocalDateTime 格式的日期
     * @return Date
     */
    public static Date toDate(LocalDate temporalAccessor) {
        LocalDateTime localDateTime = LocalDateTime.of(temporalAccessor, LocalTime.of(0, 0, 0));
        return toDate(localDateTime);
    }

}
