package com.kyle.component.kdb.impl;

import android.content.Context;
import android.database.sqlite.SQLiteDatabase;

import com.kyle.component.kdb.KAppDb;
import com.kyle.component.kdb.condition.ORMMapper;
import com.kyle.component.kdb.core.BaseEntity;
import com.kyle.component.kdb.error.DBException;
import com.kyle.component.kdb.interfaces.IDAO;
import com.kyle.component.kdb.interfaces.IDBListener;
import com.kyle.component.kdb.manager.KDbManager;

import java.io.File;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Created by Kyle on 2020/12/22.
 */

public class KDbManagerImpl<T extends BaseEntity> implements KDbManager<T> {
    private static HashMap<String, KDbManagerImpl> daoMap = new HashMap();
    private Context mContext;
    private IDAO<T> tutuDbDAO;
    public KDbManager.DAOConfig daoConfig;
    private String BASE_DIR;


    public KDbManagerImpl(KDbManager.DAOConfig daoConfig) {
        this.daoConfig = daoConfig;
        if (this.mContext == null)
            this.mContext = KAppDb.getAppDb().getContext();
        BASE_DIR = mContext.getFilesDir() + File.separator;
        createDatabase();
    }

    public static synchronized KDbManager getInstance(KDbManager.DAOConfig daoConfig) {
        if (daoConfig == null) {
            daoConfig = new DAOConfig();
        }
        KDbManagerImpl manager = daoMap.get(daoConfig.getDbName());
        if (manager == null) {
            manager = new KDbManagerImpl(daoConfig);
            daoMap.put(daoConfig.getDbName(), manager);
        }
        try {
            SQLiteDatabase db = manager.getDatabase();
            int oldVersion = db.getVersion();
            int newVersion = daoConfig.getDbVersion();
            if (oldVersion != newVersion) {
                if (oldVersion != 0) {
                    IDBListener upgradeListener = daoConfig.getIdbListener();
                    if (upgradeListener != null) {
                        upgradeListener.onUpdate(manager.getDao(), oldVersion, newVersion);
                    } else {
                        try {
                            manager.getDao().dropDb();
                        } catch (DBException e) {
                        }
                    }
                }
                db.setVersion(newVersion);

            }
            manager.closeDb();
        } catch (DBException e) {
            e.printStackTrace();
            return null;
        }
        return manager;
    }

    protected IDAO<T> getDao() {
        return tutuDbDAO;
    }

    @Override
    public synchronized void save(T entity) throws DBException {
        tutuDbDAO.save(entity);
    }

    @Override
    public synchronized void saveAll(List<T> entityList) throws DBException {
        tutuDbDAO.saveAll(entityList);
    }

    @Override
    public synchronized List<T> findAll(Class<T> entity) throws DBException {
        return tutuDbDAO.findAll(entity);
    }

    @Override
    public synchronized List<T> find(Class<T> entity, String condition) throws DBException {
        return tutuDbDAO.find(entity, condition);
    }

    @Override
    public synchronized List<T> find(Class<T> entity, ORMMapper mapper) throws DBException {
        return tutuDbDAO.find(entity, mapper.generateSql());
    }

    @Override
    public synchronized T findFirst(Class<T> entity, String condition) throws DBException {
        return tutuDbDAO.findFirst(entity, condition);
    }

    @Override
    public synchronized T findFirst(Class<T> entity, ORMMapper mapper) throws DBException {
        return findFirst(entity, mapper.generateSql());
    }

    @Override
    public synchronized void delete(T entity) throws DBException {
        tutuDbDAO.delete(entity);
    }

    @Override
    public synchronized void delete(T[] entity) throws DBException {
        tutuDbDAO.delete(entity);
    }

    @Override
    public synchronized void delete(Class<T> clazz, String condition) throws DBException {
        tutuDbDAO.delete(clazz, condition);
    }

    @Override
    public synchronized void delete(Class<T> clazz, ORMMapper mapper) throws DBException {
        delete(clazz, mapper.generateSql());
    }

    @Override
    public synchronized void deleteAll(Class<T> clazz) throws DBException {
        tutuDbDAO.deleteAll(clazz);
    }

    @Override
    public synchronized void update(T entity) throws DBException {
        tutuDbDAO.update(entity);
    }

    @Override
    public synchronized void update(T entity, String condition) throws DBException {
        tutuDbDAO.update(entity, condition);
    }

    @Override
    public void update(T entity, ORMMapper mapper) throws DBException {
        update(entity, mapper.generateSql());
    }

    @Override
    public synchronized void update(T entity, String condition, String name, String value) throws DBException {
        tutuDbDAO.update(entity, condition, name, value);
    }

    @Override
    public synchronized void update(T entity, String condition, String name, int value) throws DBException {
        tutuDbDAO.update(entity, condition, name, value);
    }

    @Override
    public synchronized int update(T entity, String condition, Map<String, Object> params) throws DBException {
        return tutuDbDAO.update(entity, condition, params);
    }

    @Override
    public synchronized int update(T entity, ORMMapper mapper, Map<String, Object> params) throws DBException {
        return update(entity, mapper.generateSql(), params);
    }

    @Override
    public synchronized int update(Class<T> clazz, String condition, Map<String, Object> params) throws DBException {
        List<T> list = find(clazz, condition);
        int count = 0;
        if (list != null && list.size() > 0) {
            for (T t : list) {
                count += update(t, condition, params);
            }
        }
        return count;
    }

    @Override
    public synchronized int update(Class<T> clazz, ORMMapper mapper, Map<String, Object> params) throws DBException {
        return update(clazz, mapper.generateSql(), params);
    }

    @Override
    public synchronized void updateOrSave(T entity, String condition) throws DBException {
        tutuDbDAO.updateOrSave(entity, condition);
    }

    @Override
    public synchronized void updateOrSave(T entity, ORMMapper mapper) throws DBException {
        update(entity, mapper.generateSql());
    }

    @Override
    public synchronized long getCount(Class<T> clazz) throws DBException {
        return tutuDbDAO.getCount(clazz);
    }

    @Override
    public synchronized void closeDb() throws DBException {
        tutuDbDAO.closeDb();
    }

    @Override
    public synchronized String getDatabaseFullName(String dbName) {
        return BASE_DIR + dbName;
    }

    @Override
    public synchronized SQLiteDatabase getDatabase() throws DBException {
        return tutuDbDAO.getDatabase();
    }

    @Override
    public synchronized DAOConfig getDAOConfig() {
        return daoConfig;
    }


    private synchronized IDAO<T> createDatabase() {
        KDbDAO<T> dao = new KDbDAO<>(mContext, daoConfig);
        if (dao.info.isUpdate) {
            dao.info.isUpdate = false;
            if (daoConfig.getIdbListener() != null) {
                daoConfig.getIdbListener().onUpdate(dao, dao.info.from, dao.info.to);
            }
        }
        this.tutuDbDAO = dao;
        return dao;
    }


    @Override
    public boolean isDatabaseExists(String dbName) {
        return false;
    }

    @Override
    public boolean deleteDatabase(String dbName) {
        return false;
    }
}
