package com.kyle.component.kdb.impl;

import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteOpenHelper;
import android.text.TextUtils;

import com.kyle.component.kdb.core.BaseEntity;
import com.kyle.component.kdb.core.ORMMethodUtils;
import com.kyle.component.kdb.core.ORMUtils;
import com.kyle.component.kdb.error.DBError;
import com.kyle.component.kdb.error.DBException;
import com.kyle.component.kdb.interfaces.IDAO;
import com.kyle.component.kdb.manager.KDbManager;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * Created by Kyle on 2020/12/22.
 */

public class KDbDAO<T extends BaseEntity> extends SQLiteOpenHelper implements IDAO<T> {
    private Context mContext;
    private int version;
    private SQLiteDatabase database;
    private KDbManager.DAOConfig daoConfig;
    public DBUpdateInfo info = new DBUpdateInfo();
    private AtomicInteger dbLock = new AtomicInteger();

    public KDbDAO(Context context, KDbManager.DAOConfig daoConfig) {
        this(context, daoConfig.getDbName(), null, daoConfig.getDbVersion());
        this.daoConfig = daoConfig;
    }

    public KDbDAO(Context context, String name, SQLiteDatabase.CursorFactory factory, int version) {
        super(context, name, factory, version);
    }

    @Override
    public void onCreate(SQLiteDatabase db) {

    }

    @Override
    public void onUpgrade(SQLiteDatabase db, int oldVersion, int newVersion) {
        info.isUpdate = true;
        info.from = oldVersion;
        info.to = newVersion;

    }

    @Override
    public synchronized SQLiteDatabase getDatabase() throws DBException {
        openDB(true);
        return database;
    }

    @Override
    public synchronized void createTable(Class clazz) throws DBException {
        try {
            openDB(true);
            String sql = ORMUtils.genCreateTableSQL(clazz);
            database.execSQL(sql);
        } catch (Exception e) {
            e.printStackTrace();
            throw new DBException(DBError.ERR_CREATE_TABLE, e);
        } finally {
            closeDB();
        }
    }

    @Override
    public synchronized void update(T entity) throws DBException {
        if (entity == null) {
            throw new DBException(DBError.ERR_UPDATE_PARAM);
        }
        String tableName = ORMMethodUtils.getTableName(entity.getClass());
        Field[] fields = ORMMethodUtils.getDeclaredField(entity.getClass());
        Field pkField = ORMMethodUtils.getPKField(fields);
        if (pkField == null) {
            throw new DBException(DBError.ERR_GET_PRIMARY_KEY);
        }
        Object pkValue;
        try {
            if (!pkField.isAccessible())
                pkField.setAccessible(true);
            pkValue = pkField.get(entity);
        } catch (IllegalAccessException e) {
            e.printStackTrace();
            throw new DBException(null);
        }
        if (pkValue == null) {
            throw new DBException(DBError.ERR_GET_PRIMARY_KEY_VALUE);
        }
        String pkValueColumn = ORMMethodUtils.getColumnName(pkField);
        ContentValues values;

        try {
            openDB(true);
            values = ORMMethodUtils.putValue(fields, entity);
            if (values.size() <= 0) {
                throw new DBException(DBError.ERR_UPDATE_PARAM);
            }

            int flag = database.update(tableName, values, pkValueColumn + "= ?",
                    new String[]{String.valueOf(pkValue)});
            if (flag < 1) {
                throw new DBException(DBError.ERR_UPDATE_PARAM);
            }
        } catch (Exception e) {
            throw new DBException(DBError.ERR_UPDATE, e);
        } finally {
            closeDB();
        }
    }

    @Override
    public synchronized int update(T entity, String condition, Map<String, Object> paramsMap) throws DBException {
        if (entity == null) {
            throw new DBException(DBError.ERR_UPDATE_PARAM);
        }
        String tableName = ORMMethodUtils.getTableName(entity.getClass());
        if (!isTableExist(tableName)) {
            return 0;
        }
        ContentValues values = new ContentValues();
        try {
            openDB(true);
            for (String key : paramsMap.keySet()) {
                Object obj = paramsMap.get(key);
                if (obj instanceof Integer) {
                    values.put(key, (Integer) obj);
                } else if (obj instanceof String) {
                    values.put(key, (String) obj);
                } else if (obj instanceof Float) {
                    values.put(key, (Float) obj);
                } else if (obj instanceof Byte) {
                    values.put(key, (Byte) obj);
                } else if (obj instanceof Short) {
                    values.put(key, (Short) obj);
                } else if (obj instanceof Short) {
                    values.put(key, (Short) obj);
                } else if (obj instanceof Double) {
                    values.put(key, (Double) obj);
                } else if (obj instanceof byte[]) {
                    values.put(key, (byte[]) obj);
                } else if (obj instanceof Long) {
                    values.put(key, (Long) obj);
                }
            }
            if (values.size() <= 0) {
                throw new DBException(DBError.ERR_UPDATE_PARAM);
            }
            int flag = database.update(tableName, values, condition, null);
            if (flag < 1) {
                throw new DBException(DBError.ERR_UPDATE_PARAM);
            }
            return flag;

        } catch (Exception e) {
            throw new DBException(DBError.ERR_UPDATE, e);
        } finally {
            closeDB();
        }
    }

    public synchronized void update(T entity, String condition, String name, String value) throws DBException {
        if (entity == null) {
            throw new DBException(DBError.ERR_UPDATE_PARAM);
        }
        String tableName = ORMMethodUtils.getTableName(entity.getClass());
        if (!isTableExist(tableName)) {
            return;
        }
        ContentValues values = new ContentValues();
        try {
            openDB(true);
            values.put(name, value);
            if (values.size() <= 0) {
                throw new DBException(DBError.ERR_UPDATE_PARAM);
            }
            int flag = database.update(tableName, values, condition, null);
            if (flag < 1) {
                throw new DBException(DBError.ERR_UPDATE_PARAM);
            }

        } catch (Exception e) {
            throw new DBException(DBError.ERR_UPDATE, e);
        } finally {
            closeDB();
        }
    }

    public synchronized void update(T entity, String condition, String name, int value) throws DBException {
        if (entity == null) {
            throw new DBException(DBError.ERR_UPDATE_PARAM);
        }
        String tableName = ORMMethodUtils.getTableName(entity.getClass());
        if (!isTableExist(tableName)) {
            return;
        }
        ContentValues values = new ContentValues();
        try {
            openDB(true);
            values.put(name, value);
            if (values.size() <= 0) {
                throw new DBException(DBError.ERR_UPDATE_PARAM);
            }
            int flag = database.update(tableName, values, condition, null);
            if (flag < 1) {
                throw new DBException(DBError.ERR_UPDATE_PARAM);
            }

        } catch (Exception e) {
            throw new DBException(DBError.ERR_UPDATE, e);
        } finally {
            closeDB();
        }
    }

    @Override
    public synchronized void update(T entity, String condition) throws DBException {
        if (entity == null) {
            throw new DBException(DBError.ERR_UPDATE_PARAM);
        }
        String tableName = ORMMethodUtils.getTableName(entity.getClass());
        if (!isTableExist(tableName)) {
            save(entity);
            return;
        }
        ContentValues values;

        try {
            openDB(true);
            Field[] fields = ORMMethodUtils.getDeclaredField(entity.getClass());
            values = ORMMethodUtils.putValue(fields, entity);
            Field pkField = ORMMethodUtils.getPKField(fields);
            values.remove(ORMMethodUtils.getColumnName(pkField));
            if (values.size() <= 0) {
                throw new DBException(DBError.ERR_UPDATE_PARAM);
            }
            int flag = database.update(tableName, values, condition, null);
            if (flag < 1) {
                throw new DBException(DBError.ERR_UPDATE_PARAM);
            }

        } catch (Exception e) {
            throw new DBException(DBError.ERR_UPDATE, e);
        } finally {
            closeDB();
        }
    }

    @Override
    public synchronized void updateOrSave(T entity, String condition) throws DBException {
        if (entity == null) {
            throw new DBException(DBError.ERR_UPDATE_PARAM);
        }
        String tableName = ORMMethodUtils.getTableName(entity.getClass());
        if (!isTableExist(tableName)) {
            save(entity);
            return;
        }
        List<T> entityList = find((Class<T>) entity.getClass(), condition);
        if (entityList == null || entityList.size() <= 0) {
            save(entity);
        } else {
            try {
                openDB(true);
                Field[] fields = ORMMethodUtils.getDeclaredField(entity.getClass());
                ContentValues values = ORMMethodUtils.putValue(fields, entity);
                Field pkField = ORMMethodUtils.getPKField(fields);
                values.remove(ORMMethodUtils.getColumnName(pkField));
                if (values.size() <= 0) {
                    throw new DBException(DBError.ERR_UPDATE_PARAM);
                }
                int flag = database.update(tableName, values, condition, null);
                if (flag < 1) {
                    throw new DBException(DBError.ERR_UPDATE_PARAM);
                }
            } catch (Exception e) {
                throw new DBException(DBError.ERR_UPDATE, e);
            } finally {
                closeDB();
            }
        }
    }

    @Override
    public synchronized List<T> findAll(Class<T> clazz) throws DBException {
        List<T> objList;
        try {
            String tableName = createTableIfNotExist(clazz);
            openDB(true);
            String sql = "SELECT * FROM " + tableName;
            Cursor cursor = database.rawQuery(sql, new String[]{});
            objList = ORMMethodUtils.cursor2Entity(clazz, cursor);
            if (objList != null && objList.size() > 0) {
                return objList;
            }
        } catch (Exception e) {
            throw new DBException(DBError.ERR_FIND, e);
        } finally {
            closeDB();
        }
        return objList;
    }

    @Override
    public synchronized List<T> find(Class<T> clazz, String condition) throws DBException {
        if (TextUtils.isEmpty(condition)) {
            throw new DBException(DBError.ERR_FIND_CONDITION);
        }

        String tableName = createTableIfNotExist(clazz);
        try {
            openDB(true);
            String sql;
            if (condition.trim().startsWith("ORDER") || condition.trim().startsWith("order")) {
                sql = "SELECT * FROM " + tableName + " " + condition;
            } else {
                sql = "SELECT * FROM " + tableName + " WHERE " + condition;
            }
            Cursor cursor = database.rawQuery(sql, new String[]{});
            List<T> objList = ORMMethodUtils.cursor2Entity(clazz, cursor);
            if (objList != null && objList.size() > 0) {
                return objList;
            }
        } catch (Exception e) {
            e.printStackTrace();
            throw new DBException(DBError.ERR_FIND);
        } finally {
            closeDB();
        }
        return new ArrayList<>();
    }

    @Override
    public synchronized T findFirst(Class<T> clazz, String condition) throws DBException {
        if (TextUtils.isEmpty(condition)) {
            throw new DBException(DBError.ERR_FIND_CONDITION);
        }

        String tableName = createTableIfNotExist(clazz);
        try {
            openDB(true);

            String sql = "SELECT * FROM " + tableName + " WHERE " + condition + " LIMIT 1";
            Cursor cursor = database.rawQuery(sql, new String[]{});
            List<T> objList = ORMMethodUtils.cursor2Entity(clazz, cursor);
            if (objList != null && objList.size() > 0) {
                return objList.get(0);
            }
        } catch (Exception e) {
            throw new DBException(DBError.ERR_FIND);
        } finally {
            closeDB();
        }
        return null;
    }

    @Override
    public synchronized void deleteAll(Class<T> clazz) throws DBException {
        String tableName = ORMMethodUtils.getTableName(clazz);
        if (!isTableExist(tableName))
            return;
        try {
            openDB(true);

            String delSql = "DELETE FROM " + tableName;
            String revSeqSql = "UPDATE SQLITE_SEQUENCE SET SEQ=0 WHERE NAME='" + tableName + "'";

            database.beginTransaction();
            database.execSQL(delSql);
            database.execSQL(revSeqSql);
            database.setTransactionSuccessful();
        } catch (Exception e) {
            throw new DBException(DBError.ERR_DEL, e);
        } finally {
            database.endTransaction();
            closeDB();
        }
    }

    @Override
    public synchronized void delete(T t) throws DBException {
//        String tableName = ORMMethodUtils.getTableName(t.getClass());
        String tableName = createTableIfNotExist(t.getClass());
        Field pkField = ORMMethodUtils.getPKField(t.getClass());
        if (pkField == null) {
            throw new DBException(DBError.ERR_GET_PRIMARY_KEY);
        }
        String columnName = ORMMethodUtils.getColumnName(pkField);

        try {
            openDB(true);
            StringBuilder sbSql = new StringBuilder("DELETE FROM " + tableName + " WHERE " + columnName);
            sbSql.append(" = ");
            sbSql.append(t.getIncrementId());
            database.execSQL(sbSql.toString());
        } catch (Exception e) {
            throw new DBException(DBError.ERR_DEL, e);
        } finally {
            closeDB();
        }
    }

    @Override
    public synchronized void delete(T[] ts) throws DBException {
        int length = ts == null ? 0 : ts.length;
        if (length <= 0) {
            throw new DBException(DBError.ERR_DEL_PARAM);
        }
        T entity = ts[0];
        String tableName = createTableIfNotExist(entity.getClass());
        Field pkField = ORMMethodUtils.getPKField(entity.getClass());
        if (pkField == null) {
            throw new DBException(DBError.ERR_GET_PRIMARY_KEY);
        }

        Class pkType = pkField.getType();
        String columnName = ORMMethodUtils.getColumnName(pkField);

        try {
            openDB(true);

            StringBuilder sbSql = new StringBuilder("DELETE FROM " + tableName + " WHERE " + columnName);

            if (ts.length == 1) {
                sbSql.append(" = ");
            } else {
                sbSql.append(" in (");
            }

            String strSep = "";
            if (pkType == String.class) {
                strSep = "'";
            }

            StringBuilder strEntityIds = new StringBuilder("");
            for (T t : ts) {
                strEntityIds.append(strSep);
                strEntityIds.append(t.getIncrementId());
                strEntityIds.append(strSep);
                strEntityIds.append(",");
            }
            strEntityIds.deleteCharAt(strEntityIds.length() - 1);
            sbSql.append(strEntityIds.toString());

            if (ts.length > 1) {
                sbSql.append(")");
            }
            database.execSQL(sbSql.toString());
        } catch (Exception e) {
            throw new DBException(DBError.ERR_DEL, e);
        } finally {
            closeDB();
        }
    }

    @Override
    public synchronized void delete(Class<T> clazz, String condition) throws DBException {
        String tableName = createTableIfNotExist(clazz);
        try {
            openDB(true);
            StringBuilder sbSql = new StringBuilder("DELETE FROM " + tableName + " WHERE " + condition);
            database.execSQL(sbSql.toString());
        } catch (Exception e) {
            throw new DBException(DBError.ERR_DEL, e);
        } finally {
            closeDB();
        }
    }

    @Override
    public synchronized boolean isTableExist(String tableName) throws DBException {
        boolean isExist = false;
        if (TextUtils.isEmpty(tableName))
            return isExist;

        try {
            openDB(false);
            String sql = "SELECT COUNT(*) FROM Sqlite_master WHERE TYPE ='table' AND NAME ='" + tableName.trim() + "' ";
            Cursor cursor = database.rawQuery(sql, null);
            if (cursor.moveToNext()) {
                int count = cursor.getInt(0);
                if (count > 0) {
                    isExist = true;
                }
            }
            return isExist;
        } catch (Exception e) {
            e.printStackTrace();
            throw new DBException(DBError.ERR_IS_TABLE_EXISTS);
        } finally {
            closeDB();
        }
    }

    private synchronized String createTableIfNotExist(Class clazz) throws DBException {
        String tableName = ORMMethodUtils.getTableName(clazz);
        if (!isTableExist(tableName)) {
            createTable(clazz);
        }
        return tableName;
    }

    @Override
    public synchronized void save(T entity) throws DBException {
        String tableName = createTableIfNotExist(entity.getClass());
        ContentValues values;
        try {
            openDB(true);
            Field[] fields = ORMMethodUtils.getDeclaredField(entity.getClass());
            values = ORMMethodUtils.putValue(fields, entity);
            if (values == null || values.size() <= 0) {
                throw new DBException(DBError.ERR_SAVE_PARAM);
            }
            long flag = database.insert(tableName, null, values);

            if (flag < 1) {
                throw new DBException(DBError.ERR_SAVE_PARAM);
            }
            Field pkField = ORMMethodUtils.getPKField(fields);
            if (pkField != null) {
                try {
                    if (!pkField.isAccessible())
                        pkField.setAccessible(true);
                    pkField.set(entity, (int) flag);
                } catch (IllegalAccessException e) {
                    e.printStackTrace();
                    throw new DBException(null);
                }
            }

        } catch (Exception e) {
            throw new DBException(DBError.ERR_SAVE_PARAM, e);
        } finally {
            closeDB();
        }
    }

    @Override
    public synchronized void saveAll(List<T> entityList) throws DBException {
        if (entityList.size() > 0) {
            T firstEntity = entityList.get(0);
            String tableName = createTableIfNotExist(firstEntity.getClass());
            Iterator iterator = entityList.iterator();
            try {
                openDB(true);
                while (iterator.hasNext()) {
                    T entity = (T) iterator.next();
                    ContentValues values;
                    Field[] fields = ORMMethodUtils.getDeclaredField(entity.getClass());
                    values = ORMMethodUtils.putValue(fields, entity);
                    if (values == null || values.size() <= 0) {
                        throw new DBException(DBError.ERR_SAVE_PARAM);
                    }
                    long flag = database.insert(tableName, null, values);
                    if (flag < 1) {
                        throw new DBException(DBError.ERR_SAVE_PARAM);
                    }

                }
            } catch (Exception e) {
                throw new DBException(DBError.ERR_SAVE_PARAM, e);
            } finally {
                closeDB();
            }
        }
    }

    @Override
    public synchronized long getCount(Class<T> clazz) throws DBException {
        String tableName = ORMMethodUtils.getTableName(clazz);
        if (TextUtils.isEmpty(tableName) || !isTableExist(tableName))
            return 0;
        try {
            openDB(true);

            String sql = "SELECT COUNT(*) FROM " + tableName;
            Cursor cursor = database.rawQuery(sql, null);
            cursor.moveToFirst();
            long total = cursor.getLong(0);
            return total;
        } catch (Exception e) {
            throw new DBException(DBError.ERR_GET_COUNT);
        } finally {
            closeDB();
        }
    }

    @Override
    public synchronized void dropDb() throws DBException {
        openDB(true);
        Cursor cursor = database.rawQuery("SELECT name FROM sqlite_master WHERE type=\'table\' AND name<>\'sqlite_sequence\'", null);
        if (cursor != null) {
            try {
                while (cursor.moveToNext()) {
                    String e = cursor.getString(0);
                    database.rawQuery("DROP TABLE " + e, null);
                }
            } catch (Exception e) {
                throw new DBException(DBError.ERR_DROP_TABLE);
            } finally {
                closeDB();
            }
        }
    }

    @Override
    public void closeDb() throws DBException {
        closeDB();
    }

    private synchronized void openDB(boolean checkTable) throws DBException {
        if (checkTable && TextUtils.isEmpty(daoConfig.getDbName())) {
            throw new DBException(DBError.ERR_IS_TABLE_EXISTS);
        }
        if (dbLock.incrementAndGet() == 1 || database == null) {
            database = getWritableDatabase();
        }
    }

    private synchronized void closeDB() {
        if (dbLock.decrementAndGet() == 0) {
            try {
                if (database != null && database.isOpen()) {
                    database.close();
                }
            } catch (Exception e) {

            }

        }
    }

    public static boolean isTableExist(SQLiteDatabase db, String tableName) {
        Cursor cursor = null;
        try {
            cursor = db.rawQuery("SELECT count(*) FROM sqlite_master WHERE type='table' AND name=?", new String[]{tableName});
            boolean hasNext = cursor.moveToNext();
            return hasNext && cursor.getInt(0) > 0;
        } finally {
            if (cursor != null) {
                cursor.close();
            }
        }
    }

    public static boolean isColumnExist(SQLiteDatabase db, String tableName, String columnName) {
        Cursor cursor = null;
        try {
            cursor = db.rawQuery("SELECT count(*) FROM sqlite_master WHERE tbl_name = ? AND (sql LIKE ? OR sql LIKE ?);",
                    new String[]{tableName, "%(" + columnName + "%", "%, " + columnName + " %"});
            boolean hasNext = cursor.moveToNext();
            return hasNext && cursor.getInt(0) > 0;
        } finally {
            if (cursor != null) {
                cursor.close();
            }
        }
    }

    private static boolean checkColumnExists(SQLiteDatabase db, String tableName
            , String columnName) {
        boolean result = false;
        Cursor cursor = null;

        try {
            cursor = db.rawQuery("select * from sqlite_master where name = ? and sql like ?"
                    , new String[]{tableName, "%" + columnName + "%"});
            result = null != cursor && cursor.moveToFirst();
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            if (null != cursor && !cursor.isClosed()) {
                cursor.close();
            }
        }

        return result;
    }


    class DBUpdateInfo {
        public boolean isUpdate = false;
        public int from;
        public int to;
    }

}
