package com.kyle.component.kdb.core;

import android.content.ContentValues;
import android.database.Cursor;
import android.database.sqlite.SQLiteException;
import android.text.TextUtils;

import com.kyle.component.kdb.annotation.Column;
import com.kyle.component.kdb.annotation.Id;
import com.kyle.component.kdb.annotation.Table;
import com.kyle.component.kdb.error.DBException;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.List;

/**
 * Created by Kyle on 2020/12/22.
 */

public class ORMMethodUtils {

    /**
     * 根据指定类获取对应的表名
     *
     * @param clazz
     * @return
     */
    public static String getTableName(Class clazz) {
        String tableName = clazz.getSimpleName();

        Class superClass = getAnnotationTableName(clazz);
        if (superClass != null) {
            Table ts = (Table) superClass.getAnnotation(Table.class);
            String annotationTable = ts.tableName();
            //如果获取的值为@Table的默认值，则使用类名来做为表名
            if (annotationTable != null && !annotationTable.equals("className"))
                tableName = annotationTable;
        }
        return tableName;
    }

    /**
     * 递归查找存在Table注解的父类
     *
     * @param clazz
     * @return
     */
    private static Class getAnnotationTableName(Class clazz) {
        if (clazz == null) {
            return null;
        }
        if (clazz.isAnnotationPresent(Table.class)) {
            return clazz;
        } else {
            return getAnnotationTableName(clazz.getSuperclass());
        }
    }

    /**
     * 获取类和父类所有的方法
     *
     * @param clazz
     * @return
     */
    public static Field[] getDeclaredField(Class clazz) {
        if (clazz == null)
            return null;
        Class superClazz = clazz.getSuperclass();
        List<Field> superFieldsList = new ArrayList();
        while (superClazz != null) {
            Field[] superFields = superClazz.getDeclaredFields();
            for (int i = 0; i < superFields.length; i++)
                superFieldsList.add(superFields[i]);
            superClazz = superClazz.getSuperclass();
        }
        Field[] superFields = new Field[superFieldsList.size()];
        for (int i = 0; i < superFieldsList.size(); i++) {
            superFields[i] = superFieldsList.get(i);
        }
        Field[] fields = clazz.getDeclaredFields();

        Field[] fullFields = new Field[superFields.length + fields.length];
        System.arraycopy(superFields, 0, fullFields, 0, superFields.length);
        System.arraycopy(fields, 0, fullFields, superFields.length, fields.length);
        return fullFields;
    }

    /**
     * 获取该方法的映射值
     *
     * @param field
     * @return
     */
    public static String getColumnName(Field field) {
        String columnName = null;
        if (field.isAnnotationPresent(Id.class)) {
            Id ids = field.getAnnotation(Id.class);
            columnName = ids.id();
        } else if (field.isAnnotationPresent(Column.class)) {
            Column column = field.getAnnotation(Column.class);
            columnName = column.name();
        }
        return columnName;
    }

    interface DataType {
        String INTEGER = "INTEGER";

        String BLOB = "BLOB";

        String TEXT = "TEXT";

        String REAL = "REAL";
    }

    public static String getDataTypeByField(Field field) {
        Class<?> dataTypeClass = field.getType();

        // all number type will be treat as INTEGER in SQLite
        if ((dataTypeClass == Integer.class || dataTypeClass == int.class)) {
            return DataType.INTEGER;
        } else if (dataTypeClass == Long.class || dataTypeClass == long.class) {
            return DataType.INTEGER;
        } else if (dataTypeClass == String.class) {
            return DataType.TEXT;
        } else if (dataTypeClass == Short.class || dataTypeClass == short.class) {
            return DataType.INTEGER;
        } else if (dataTypeClass == Double.class || dataTypeClass == double.class) {
            return DataType.REAL;
        } else if (dataTypeClass == Float.class || dataTypeClass == float.class) {
            return DataType.REAL;
        } else if (dataTypeClass == Boolean.class || dataTypeClass == boolean.class) {
            return DataType.INTEGER;
        } else if (dataTypeClass == Byte[].class || dataTypeClass == byte[].class) {
            return DataType.BLOB;
        } else {
            throw new SQLiteException("field [" + field.getName() + "] is a not supported data type.");
        }
    }


    public static String toUpperCaseFirstOne(String name) {
        if (TextUtils.isEmpty(name)) {
            return name;
        }
        StringBuffer sb = new StringBuffer(name);
        String firstChar = name.substring(0, 1).toUpperCase();
        sb.replace(0, 1, firstChar);
        return sb.toString();
    }

    public static <T extends BaseEntity> List<T> cursor2Entity(Class<T> clazz, Cursor cursor) throws DBException {
        List<T> objList = new ArrayList<>();
        Field[] fields = getDeclaredField(clazz);
        try {
            if (cursor.moveToFirst()) {
                while (!cursor.isAfterLast()) {
                    T obj = clazz.newInstance();

                    for (int i = 0; i < cursor.getColumnCount(); i++) {
                        String strColName = cursor.getColumnName(i);

                        for (Field field : fields) {
                            String fieldColumnName = getColumnName(field);
                            if (fieldColumnName == null)
                                continue;
                            if (getColumnName(field).equals(strColName)) {
                                strColName = toUpperCaseFirstOne(strColName);
                                if (cursor.getType(i) == Cursor.FIELD_TYPE_NULL) {
                                    continue;
                                } else if (cursor.getType(i) == Cursor.FIELD_TYPE_FLOAT) {
                                    field.setAccessible(true);
                                    field.set(obj, cursor.getFloat(i));
                                } else if (cursor.getType(i) == Cursor.FIELD_TYPE_INTEGER) {
                                    if (field.getGenericType().toString().equals("class java.lang.Boolean")
                                            || field.getGenericType().toString().equals("boolean")) {
                                        field.setAccessible(true);
                                        field.set(obj, cursor.getInt(i) == 1 ? true : false);
                                    } else if (field.getGenericType().toString().equals("class java.lang.Integer")
                                            || field.getGenericType().toString().equals("int")) {
                                        field.setAccessible(true);
                                        field.set(obj, cursor.getInt(i));
                                    } else if (field.getGenericType().toString().equals("class java.lang.Long")
                                            || field.getGenericType().toString().equals("long")) {
                                        field.setAccessible(true);
                                        field.set(obj, cursor.getLong(i));
                                    } else if (field.getGenericType().toString().equals("class java.lang.Short")
                                            || field.getGenericType().toString().equals("short")) {
                                        field.setAccessible(true);
                                        field.set(obj, (short) cursor.getInt(i));
                                    } else if (field.getGenericType().toString().equals("class java.lang.Byte")
                                            || field.getGenericType().toString().equals("byte")) {
                                        field.setAccessible(true);
                                        field.set(obj, (byte) cursor.getInt(i));
                                    }
                                } else if (cursor.getType(i) == Cursor.FIELD_TYPE_STRING) {
                                    field.setAccessible(true);
                                    field.set(obj, cursor.getString(i));
                                } else if (cursor.getType(i) == Cursor.FIELD_TYPE_BLOB) {
                                    field.setAccessible(true);
                                    field.set(obj, cursor.getBlob(i));
                                } else {
                                    throw new DBException(null);
                                }
                                break;
                            }
                        }
                    }
                    objList.add(obj);
                    cursor.moveToNext();
                }
                return objList;
            }
        } catch (Exception e) {
            e.printStackTrace();
            throw new DBException(null);
        }
        return objList;
    }

    /**
     * 通过反射成员变量并存储进数据库
     *
     * @param fields
     * @param entity
     * @param <T>
     * @return
     * @throws DBException
     */
    public static <T> ContentValues putValue(Field[] fields, T entity) throws DBException {
        ContentValues values = new ContentValues();

        for (Field field : fields) {
            if (!field.isAccessible())
                field.setAccessible(true);

            String strColName = getColumnName(field);
            if (strColName == null)
                continue;

            try {
                if (field.getGenericType().toString().equals("class java.lang.String")) {
                    values.put(strColName, (String) (field.get(entity)));
                } else if (field.getGenericType().toString().equals("class java.lang.Boolean")
                        || field.getGenericType().toString().equals("boolean")) {
                    values.put(strColName, (((Boolean) (field.get(entity))) ? 1 : 0));
                } else if (field.getGenericType().toString().equals("class java.lang.Byte")
                        || field.getGenericType().toString().equals("byte")) {
                    values.put(strColName, (Byte) field.get(entity));
                } else if (field.getGenericType().toString().equals("class [B")) {
                    values.put(strColName, (byte[]) field.get(entity));
                } else if (field.getGenericType().toString().equals("class java.lang.Double")
                        || field.getGenericType().toString().equals("double")) {
                    values.put(strColName, (Double) field.get(entity));
                } else if (field.getGenericType().toString().equals("class java.lang.Float")
                        || field.getGenericType().toString().equals("float")) {
                    values.put(strColName, (Float) field.get(entity));
                } else if (field.getGenericType().toString().equals("class java.lang.Integer")
                        || field.getGenericType().toString().equals("int")) {
                    values.put(strColName, (Integer) field.get(entity));
                } else if (field.getGenericType().toString().equals("class java.lang.Long")
                        || field.getGenericType().toString().equals("long")) {
                    values.put(strColName, (Long) field.get(entity));
                } else if (field.getGenericType().toString().equals("class java.lang.Short")
                        || field.getGenericType().toString().equals("short")) {
                    values.put(strColName, (Short) field.get(entity));
                } else {
                    throw new DBException(null);
                }
            } catch (IllegalAccessException e) {
                e.printStackTrace();
                throw new DBException(null);
            }
        }
        return values;
    }


    public static Field getPKField(Class clazz) {
        Field[] fields = getDeclaredField(clazz);
        return getPKField(fields);
    }

    /**
     * 得到主键成员变量
     *
     * @param fields
     * @return
     */
    public static Field getPKField(Field[] fields) {
        for (Field field : fields) {
            if (field.isAnnotationPresent(Id.class))
                return field;
        }
        return null;
    }

}
