package com.sprite.framework.entity.support.spring;

import com.sprite.framework.component.SpriteConfig;
import com.sprite.framework.component.SpriteConfigLoader;
import com.sprite.framework.context.ContextEnvironment;
import com.sprite.framework.entity.EntityLoader;
import com.sprite.framework.entity.EntityScriptExecutor;
import com.sprite.framework.entity.config.EntityResourceInfo;
import com.sprite.framework.entity.util.OrmConfig;
import com.sprite.utils.UtilCollection;
import com.sprite.utils.UtilString;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.ApplicationListener;
import org.springframework.context.event.ContextRefreshedEvent;

import java.util.LinkedList;
import java.util.List;

/**
 * @author Jack
 * @since 1.1.0
 */
public class SpriteOrmLoader implements ApplicationListener<ContextRefreshedEvent> {

    private Logger logger = LoggerFactory.getLogger(getClass());

    private volatile boolean inited = false;

    @Autowired
    private EntityScriptExecutor entityScriptExecutor;

    @Autowired
    private OrmConfig ormConfig;

    @Autowired
    private SpriteConfigLoader spriteConfigLoader;

    /**
     * Handle an application event.
     *
     * @param event
     *         the event to respond to
     */
    @Override
    public void onApplicationEvent(ContextRefreshedEvent event) {
        if (inited) {
            return;
        }

        OrmConfig.setConfig(ormConfig);
        initOrm();
        inited = true;
    }

    private void initOrm() {
        if (entityScriptExecutor == null) {
            throw new RuntimeException("not found entityScriptExecutor");
        }

        String sqlType = ormConfig.getSqlType();
        boolean ddlauto = ormConfig.isDdlAuto();

        if (UtilString.isBlank(sqlType)) {
            throw new RuntimeException("not specified sql type ");
        }

        try {
            List<SpriteConfig> componentConfigs = spriteConfigLoader.getAllComponents();

            if (UtilCollection.isEmpty(componentConfigs)) {
                logger.info("[initOrm] not found sprite.xml");
                return;
            }

            List<EntityResourceInfo> resourceInfos = new LinkedList<>();
            for (SpriteConfig spriteConfig : componentConfigs) {
                resourceInfos.addAll(EntityResourceInfo.resolveResourceInfo(spriteConfigLoader.getSpriteConfigElement(spriteConfig), spriteConfig));
            }

            EntityLoader entityLoader = new EntityLoader();
            entityLoader.setEntityResourceInfoList(resourceInfos);
            entityLoader.setEntityScriptExecutor(entityScriptExecutor);
            entityLoader.setSqlType(sqlType);
            entityLoader.loadEntityModel(ddlauto);
            entityLoader.loadEntityData();
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }
}
