package com.sprite.framework.entity.support.spring;

import com.sprite.framework.component.SpriteConfigLoader;
import com.sprite.framework.context.ContextEnvironment;
import com.sprite.framework.entity.*;
import com.sprite.framework.entity.bridge.JdbcTemplateBridge;
import com.sprite.framework.entity.mapper.DefaultStatementDelegator;
import com.sprite.framework.entity.mapper.StatementDelegator;
import com.sprite.framework.entity.util.OrmConfig;
import com.sprite.utils.UtilString;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.datasource.DataSourceTransactionManager;

import javax.sql.DataSource;
import java.util.List;

/**
 * @author Jack
 * @since 1.1.0
 */
@Configuration
public class SpriteOrmConfiguration implements InitializingBean {

    @Autowired(required = false)
    private EntityInterceptor entityInterceptor;

    @Autowired
    private EntityScriptExecutor entityScriptExecutor;

    @Autowired(required = false)
    private List<IdGenerator> generatorList;

    @Value("${sprite.orm.sqltype:}")
    private String ormSqlType;

    @Value("${sprite.orm.ddlauto:}")
    private Boolean ormDdlauto;

    @Value("${sprite.sqllog:}")
    private Boolean sqllog;

    @Value("${sprite.cached_mapper:}")
    private Boolean cachedMapper;


    @Bean
    public Delegator delegator() {
        GenericDelegator delegator = new GenericDelegator();
        delegator.setEntityInterceptor(entityInterceptor);
        delegator.addGenerator(generatorList);
        delegator.setScriptExecutor(entityScriptExecutor);
        return delegator;
    }

    @Bean
    public EntityScriptExecutor scriptExecutor(DataSource dataSource) {
        JdbcTemplateBridge jdbcTemplateBridge = new JdbcTemplateBridge();

        JdbcTemplate jdbcTemplate = new JdbcTemplate();
        jdbcTemplate.setDataSource(dataSource);
        jdbcTemplateBridge.setJdbcTemplate(jdbcTemplate);

        DataSourceTransactionManager dataSourceTransactionManager = new DataSourceTransactionManager();
        dataSourceTransactionManager.setDataSource(dataSource);
        jdbcTemplateBridge.setPlatformTransactionManager(dataSourceTransactionManager);

        return jdbcTemplateBridge;
    }

    @Bean
    public StatementDelegator statementDelegator() {
        return new DefaultStatementDelegator();
    }

    @Bean
    public SpriteOrmLoader spriteOrmLoader() {
        return new SpriteOrmLoader();
    }

    @Bean
    public OrmConfig ormConfig() {
        OrmConfig ormConfig = new OrmConfig();
        ormConfig.setCachedMapper(cachedMapper);
        ormConfig.setDdlAuto(ormDdlauto);
        ormConfig.setSqlType(ormSqlType);
        ormConfig.setOpenLog(sqllog);
        return ormConfig;
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        if (UtilString.isEmpty(ormSqlType)) {
            ormSqlType = ContextEnvironment.getProperty("sprite.orm.sqltype");
        }

        if (ormDdlauto == null) {
            ormDdlauto = "true".equalsIgnoreCase(ContextEnvironment.getProperty("sprite.orm.ddlauto", "true"));
        }

        if (sqllog == null) {
            sqllog = "true".equalsIgnoreCase(ContextEnvironment.getProperty("sprite.sqllog", "false"));
        }

        if (cachedMapper == null) {
            cachedMapper = "true".equalsIgnoreCase(ContextEnvironment.getProperty("sprite.cached_mapper", "true"));
        }

    }
}
