package Main.cases.http;

import Main.LockFiles;
import Main.TestLogger;
import Main.cases.DBHelper;
import Main.cases.DataUtils;
import Main.runner.E2ECase;
import cn.ymatrix.apiclient.MxClient;
import cn.ymatrix.builder.ConnectionListener;
import cn.ymatrix.builder.MxBuilder;
import junit.framework.AssertionFailedError;
import org.slf4j.Logger;

import java.io.IOException;
import java.nio.charset.Charset;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.HashMap;
import java.util.Map;

import static java.nio.charset.StandardCharsets.US_ASCII;
import static java.nio.charset.StandardCharsets.UTF_8;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

public class VariousEncodedStringHTTP implements E2ECase {
    private final Logger l = TestLogger.init(this.getClass());

    private final Main.mxgateHelper mxgateHelper;
    final static DBHelper dbHelper = new DBHelper();
    private final DataUtils dataMaker = new DataUtils();

    private final MxBuilder builder;
    private int grpcPort;

    public VariousEncodedStringHTTP(MxBuilder builder, Main.mxgateHelper mxgateHelper) {
        this.mxgateHelper = mxgateHelper;
        this.builder = builder;
    }

    @Override
    public String getName() {
        return this.getClass().getName();
    }
    @Override
    public void execute() throws AssertionError, IOException, SQLException {
        String schema = "public";
        String tableName = String.format("%s_table_0001", this.getClass().getSimpleName()).toLowerCase();

        // prepare
        dbHelper.createTable(String.format("CREATE TABLE IF NOT EXISTS %1$s.%2$s (time TIMESTAMP, tagid int, c1 text, c2 char(1024)) DISTRIBUTED BY (tagid);TRUNCATE %1$s.%2$s RESTART IDENTITY;", schema, tableName));

        // start mxgate
        String configStr = String.format("[source]\n" +
                "\tsource = \"http\"\n" +
                "\t[source.http]\n" +
                "\t\tmax-request-bytes = 3000000\n" +
                "\t\trequest-timeout = 500\n" +
                "[transform]\n" +
                "\ttransform = \"plain\"\n" +
                "[job]\n" +
                "\tallow-dynamic = true\n" +
                "\tdelimiter = \"|\"\n" +
                "\tformat = \"csv\"\n" +
                "\ttime-format = \"unix-ms\"\n" +
                "\t[[job.target]]\n" +
                "\t\tdelimiter = \"|\"\n" +
                "\t\tformat = \"csv\"\n" +
                "\t\tname = \"%1$s.%2$s\"\n" +
                "\t\tschema = \"%1$s\"\n" +
                "\t\ttable = \"%2$s\"\n" +
                "[writer]\n" +
                "\twriter = \"stream\"\n" +
                "[misc]\n" +
                "\tdebug = true\n", schema, tableName);

        System.out.println(configStr);
        mxgateHelper.startWithConfig(dbHelper.getDBName(), configStr);
        this.grpcPort = new LockFiles().getOneGrpcPort();

        String cfgStr = mxgateHelper.getConfig();
        assertTrue("should get mxgate config",cfgStr != null && cfgStr.length() > 0 );


        decodeAndEncodeToTest(schema, tableName, 0, "\\//1234测试测试\"2234；。……¥#&…*%@$^~`～·");
        decodeAndEncodeToTest(schema, tableName, 10, "CAPAC(“全容量”或者“预测全容量\"或者“简化容量”请选择其中一个)");
    }

    private void decodeAndEncodeToTest(String schema, String tableName, int tagidBase, String originalS) throws AssertionError, IOException, SQLException {
        HashMap<Integer, Charset> decodeCharsetMap = new HashMap<>();
        HashMap<Integer, Charset> encodeCharsetMap = new HashMap<>();
        {
            decodeCharsetMap.put(1, Charset.forName("gbk"));
            encodeCharsetMap.put(1, UTF_8);

            decodeCharsetMap.put(2, UTF_8);
            encodeCharsetMap.put(2, Charset.forName("gbk"));

            decodeCharsetMap.put(3, US_ASCII);
            encodeCharsetMap.put(3, Charset.forName("gbk"));

            decodeCharsetMap.put(4, Charset.forName("gbk"));
            encodeCharsetMap.put(4, US_ASCII);

            decodeCharsetMap.put(5, UTF_8);
            encodeCharsetMap.put(5, US_ASCII);

            decodeCharsetMap.put(6, US_ASCII);
            encodeCharsetMap.put(6, UTF_8);
        }

        int beforeCnt = dbHelper.countTuple(tableName);

        String grpcURL = String.format("localhost:%d", this.grpcPort);
        String httpURL = String.format("http://localhost:%d/", new LockFiles().getOneHttpPort());
        for (Map.Entry<Integer, Charset> entry: decodeCharsetMap.entrySet()) {
            String reEncodedV = new String(originalS.getBytes(entry.getValue()), encodeCharsetMap.get(entry.getKey()));

            this.builder.connect(httpURL, grpcURL, schema, tableName, new ConnectionListener() {
                @Override
                public void onFailure(String failureMsg) {
                    throw new AssertionFailedError(String.format("failed to connect to {}: {}", tableName, failureMsg));
                }

                @Override
                public void onSuccess(MxClient client) {
                    dataMaker.sendString(client, 100, 100, entry.getKey()+tagidBase, reEncodedV);
                }
            });
        }
        dataMaker.waitProcess();

        int afterCnt = dbHelper.countTuple(tableName);
        assertEquals("all re-encoded string value should be inserted", decodeCharsetMap.size(), afterCnt-beforeCnt);

        Statement stmt = dbHelper.getConn().createStatement();
        for (Map.Entry<Integer, Charset> entry: decodeCharsetMap.entrySet()) {
            String expectedV = new String(originalS.getBytes(entry.getValue()), encodeCharsetMap.get(entry.getKey()));

            ResultSet rs = stmt.executeQuery(String.format("SELECT c1 AS c1,c2 AS c2 FROM %s WHERE tagid=%d", tableName, entry.getKey()+tagidBase));
            if (rs.next()) {
                assertEquals("c1 should equal to insert when tagid="+entry.getKey(), expectedV, rs.getString("c1"));
                assertEquals("c2 should equal to insert when tagid="+entry.getKey(), expectedV, rs.getString("c2").trim());
            }
        }
    }
}
