package cn.yerl.android.promise.http;

import com.loopj.android.http.FileAsyncHttpResponseHandler;
import com.loopj.android.http.HttpDelete;
import com.loopj.android.http.HttpGet;
import com.loopj.android.http.RequestParams;
import com.loopj.android.http.ResponseHandlerInterface;
import com.loopj.android.http.TextHttpResponseHandler;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.charset.Charset;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import cn.yerl.android.promise.core.Promise;
import cn.yerl.android.promise.core.PromiseCallbackWithResolver;
import cn.yerl.android.promise.core.PromiseResolver;
import cz.msebera.android.httpclient.Header;
import cz.msebera.android.httpclient.HeaderElement;
import cz.msebera.android.httpclient.NameValuePair;
import cz.msebera.android.httpclient.client.methods.HttpEntityEnclosingRequestBase;
import cz.msebera.android.httpclient.client.methods.HttpHead;
import cz.msebera.android.httpclient.client.methods.HttpPost;
import cz.msebera.android.httpclient.client.methods.HttpPut;
import cz.msebera.android.httpclient.client.methods.HttpUriRequest;
import cz.msebera.android.httpclient.client.utils.URIBuilder;
import cz.msebera.android.httpclient.client.utils.URLEncodedUtils;

/**
 * Promise Http Client
 * Created by Alan Yeh on 16/6/8.
 */
public class PromiseHttp {
    private String baseUrl;
    private File cachePath;

    public String getBaseUrl() {
        return baseUrl;
    }

    public PromiseHttp setBaseUrl(String baseUrl) {
        this.baseUrl = baseUrl;
        return this;
    }

    public File getCachePath() {
        return cachePath;
    }

    public PromiseHttp setCachePath(File cachePath) {
        this.cachePath = cachePath;
        return this;
    }

    private HttpClient httpClient;

    private PromiseHttp(){
        httpClient = new HttpClient();
    }

    private static PromiseHttp instance;
    public static PromiseHttp client(){
        if (instance == null){
            synchronized (PromiseHttp.class){
                if (instance == null){
                    instance = new PromiseHttp();
                }
            }
        }
        return instance;
    }

    private Map<String, String> sharedHeaders = new LinkedHashMap<>();
    private Map<String, String> sharedCookies = new LinkedHashMap<>();

    public Promise<PromiseResponse> execute(final PromiseRequest request){
        return new Promise<>(new PromiseCallbackWithResolver<Object, PromiseResponse>() {
            @Override
            public void call(Object arg, PromiseResolver resolver) {
                _execute(request, getTextHandler(request, resolver));
            }
        });
    }

    public Promise<PromiseResponse> download(final PromiseRequest request){
        return new Promise<>(new PromiseCallbackWithResolver<Object, PromiseResponse>() {
            @Override
            public void call(Object arg, PromiseResolver resolver) {
                _execute(request, getDownloadHandler(request, resolver));
            }
        });
    }

    private void _execute(final PromiseRequest request, final ResponseHandlerInterface handler){
        URI requestURI = _processURI(baseUrl, request);

        // 处理参数
        RequestParams params = _processParams(request);

        HttpUriRequest req = null;

        switch (request.getMethod()){
            case GET:{
                req = new HttpGet(requestURI);

                break;
            }
            case DELETE:{
                req = new HttpDelete(requestURI);
                break;
            }
            case HEAD:{
                req = new HttpHead(requestURI);
                break;
            }
            case POST: {
                HttpEntityEnclosingRequestBase requestBase = new HttpPost(requestURI);
                req = requestBase;
                try {
                    requestBase.setEntity(params.getEntity(handler));
                } catch (IOException ex) {
                    throw new RuntimeException(ex);
                }
                break;
            }
            case PUT:{
                HttpEntityEnclosingRequestBase requestBase = new HttpPut(requestURI);
                req = requestBase;

                try {
                    requestBase.setEntity(params.getEntity(handler));
                } catch (IOException ex) {
                    throw new RuntimeException(ex);
                }
                break;
            }
        }

        // 处理Header
//        Header[] headers = _processHeader(sharedHeaders, request.getHeaders(), sharedCookies, request.getCookies());
//        req.setHeaders(headers);


        httpClient.sendRequest(req, handler);
    }

    private ResponseHandlerInterface getTextHandler(final PromiseRequest request, final PromiseResolver resolver){
        return new TextHttpResponseHandler() {
            @Override
            public void onFailure(int statusCode, Header[] headers, String responseString, Throwable throwable) {
                resolver.resolve(new PromiseHttpException(new PromiseResponse(request, statusCode, headers, responseString), throwable));
            }

            @Override
            public void onSuccess(int statusCode, Header[] headers, String responseString) {
                resolver.resolve(new PromiseResponse(request, statusCode, headers, responseString));
            }
        };
    }

    private ResponseHandlerInterface getDownloadHandler(final PromiseRequest request, final PromiseResolver resolver){
        URI uri = _processURI(baseUrl, request);

        String suggestFileName = uri.getPath().substring(uri.getPath().lastIndexOf("/") + 1);

        File cacheFile = new File(cachePath.getAbsolutePath() + File.separator + suggestFileName);
        if (cacheFile.exists()){
            cacheFile.delete();
        }

        return new FileAsyncHttpResponseHandler(cacheFile) {
            @Override
            public void onFailure(int statusCode, Header[] headers, Throwable throwable, File file) {
                resolver.resolve(new PromiseHttpException(new PromiseResponse(request, statusCode, headers, file), throwable));
            }

            @Override
            public void onSuccess(int statusCode, Header[] headers, File file) {
                String fileName = "";
                for (Header header : headers) {
                    if ("Content-Disposition".equalsIgnoreCase(header.getName())) {
                        HeaderElement[] elements = header.getElements();
                        for (HeaderElement element : elements) {
                            if ("attachment".equalsIgnoreCase(element.getName())) {
                                fileName = element.getParameterByName("filename").getValue();
                            }
                        }
                    }
                }
                File newFile = file;
                try {
                    if (!fileName.isEmpty()) {
                        newFile = new File(file.getParent() + File.separator + fileName);
                        file.renameTo(newFile);
                    }
                } catch (Exception e) {
                    throw new RuntimeException(e);
                }

                resolver.resolve(new PromiseResponse(request, statusCode, headers, newFile));
            }

            @Override
            public void onProgress(long bytesWritten, long totalSize) {
                request.onProgress(bytesWritten, totalSize);
            }
        };
    }


    private URI _processURI(String baseUrl, PromiseRequest request){
        try {
            URIBuilder uriBuilder = new URIBuilder(baseUrl);

            URI requestURI = new URI(request.getUrlString());

            if (requestURI.getHost() != null){
                uriBuilder = new URIBuilder(request.getUrlString()).clearParameters();
            }

            if (requestURI.getPath().startsWith("/")){
                uriBuilder.setPath(requestURI.getPath());
            }else {
                uriBuilder.setPath(uriBuilder.getPath() + "/" + requestURI.getPath());
            }

            List<NameValuePair> queryParams = URLEncodedUtils.parse(requestURI.getQuery(), Charset.forName(request.getEncoding()));
            for (NameValuePair pair : queryParams){
                uriBuilder.setParameter(pair.getName(), pair.getValue());
            }

            for (Map.Entry<String, String> param : request.getQueryParams().entrySet()){
                uriBuilder.setParameter(param.getKey(), param.getValue());
            }

            return uriBuilder.build();
        } catch (URISyntaxException e) {
            throw new RuntimeException(e);
        }
    }

    private Header[] _processHeader(Map<String, String> sharedHeaders, Map<String, String> reuquestHeaders, Map<String, String> sharedCookies, Map<String, String> requestCookies){
//        List<Header> headers = new ArrayList<>();

        Map<String, String> cookies = new HashMap<>();
        cookies.putAll(sharedCookies);
        cookies.putAll(requestCookies);

        Map<String, String> headers = new HashMap<>();
        headers.putAll(sharedHeaders);
        headers.putAll(reuquestHeaders);


//        Header header = new BasicHeader()

        return null;
    }

    private RequestParams _processParams(PromiseRequest request){
        RequestParams params = new RequestParams();
        params.setContentEncoding(request.getEncoding());

        try {
            for (Map.Entry<String, Object> param : request.getBodyParams().entrySet()){
                if (param.getValue() instanceof File){
                    params.put(param.getKey(), (File)param.getValue());
                }else if (param.getValue() instanceof File[]){
                    params.put(param.getKey(), (File[])param.getValue());
                }else if (param.getValue() instanceof InputStream){
                    params.put(param.getKey(), (InputStream)param.getValue());
                }else {
                    params.put(param.getKey(), param.getValue());
                }
            }
        }catch (Exception ex){
            throw new RuntimeException(ex);
        }
        return params;
    }
}
