package cn.xnatural.xnet;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.*;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.util.Date;
import java.util.Iterator;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;

/**
 * 固定长度的汇聚流
 * 可以把多流汇聚到一条流, 但长度固定
 */
public class XioStream extends InputStream {
    protected final static Logger log = LoggerFactory.getLogger(XioStream.class);
    /**
     * 流长
     */
    public final long             length;
    /**
     * 内容已接收了多长的数据
     */
    protected    long             received;
    /**
     * 已读长度
     */
    protected    long             readCnt = 0;
    /**
     * 读取过程中产生的异常
     */
    protected    RuntimeException ex;
    /**
     * 对列流
     */
    protected final Queue<InputStream> streams = new ConcurrentLinkedQueue<>();
    protected final Date createTime = new Date();

    /**
     *
     * @param length 流总长度
     */
    public XioStream(long length) {
        if (length < 1) throw new IllegalArgumentException("Must need a fix length stream");
        this.length = length;
    }

    public XioStream(byte[] data) {
        this.length = data.length;
        addStream(new ByteArrayInputStream(data));
    }

    /**
     *
     * @param length 流总长度
     * @param streams 子流
     */
    public XioStream(long length, InputStream...streams) {
        if (length < 1) throw new IllegalArgumentException("Must need a fix length stream");
        this.length = length;
        if (streams != null) {
            for (InputStream stream : streams) addStream(stream);
        }
    }

    /**
     * 创建一个文件流
     * @param file 文件
     */
    public XioStream(File file) throws IOException {
        this.length = file.length();
        addStream(Files.newInputStream(file.toPath()));
    }

    @Override
    public int read() throws IOException {
        if (isEnd()) return -1;
        byte[] bs = new byte[1];
        read(bs);
        return bs[0];
    }

    @Override
    public int read(final byte[] b, final int off, final int len) throws IOException {
        if (isEnd()) return -1;
        if (ex != null) throw ex;

        int readed = 0;
        int left;
        for (Iterator<InputStream> it = streams.iterator(); it.hasNext() && readed < len && (left = available()) > 0; ) {
            InputStream is = it.next();
            int l = is.read(b, off + readed, Math.min(len - readed, left));
            if (l == -1) { // 这条流没得读了, 删除
                is.close();
                it.remove();
            } else {
                readed += l;
                readCnt += l;
            }
        }
        if (readed < len && available() > 0) { // 此次没读完并且还有可读的数据, 等
            synchronized (this) {
                try {
                    wait();
                } catch (InterruptedException e) {
                    throw new RuntimeException(e);
                }
            }
            readed += read(b, off + readed, len - readed);
        }
        return readed;
    }


    /**
     * 添加新的流
     */
    public XioStream addStream(byte[] data) {
        if (data == null) throw new NullPointerException("Param data required");
        if (isEnd()) throw new RuntimeException("Already end");
        if (leftReceived() < 1) throw new RuntimeException("Already full");
        received += data.length;
        streams.offer(new ByteArrayInputStream(data));
        synchronized (this) { notify(); }
        return this;
    }


    /**
     * 添加新的流
     */
    public XioStream addStream(InputStream stream) {
        return addStream(stream, null);
    }

    /**
     * 添加新的流
     * @param length 流长
     */
    public XioStream addStream(InputStream stream, Long length) {
        if (stream == null) throw new NullPointerException("Param stream required");
        if (isEnd()) throw new RuntimeException("Already end");
        if (leftReceived() < 1) throw new RuntimeException("Already full");
        if (length != null) received += length;
        else if (stream instanceof ByteArrayInputStream) received += ((ByteArrayInputStream) stream).available();
        streams.offer(stream);
        synchronized (this) { notify(); }
        return this;
    }


    /**
     * 添加流
     * @param stream 数据流
     */
    public XioStream addStream(XioStream stream) {
        if (stream == null) throw new NullPointerException("Param stream required");
        if (isEnd()) throw new RuntimeException("Already end");
        if (leftReceived() < 1) throw new RuntimeException("Already full");
        received += stream.length;
        streams.offer(stream);
        synchronized (this) { notify(); }
        return this;
    }


    @Override
    public int available() { return (int) (length - readCnt); }


    @Override
    public void reset() throws IOException {
        for (InputStream stream : streams) {
            stream.reset();
        }
        readCnt = 0;
    }

    @Override
    public void close() {
        while (true) {
            InputStream is = streams.poll();
            if (is == null) break;
            try { is.close(); } catch (IOException e) {
                log.error("", e);
            }
        }
    }

    /**
     * 把流text化
     * @param charset 字符编码
     */
    public String text(Charset charset) throws IOException {
        if (length > 0) {
            byte[] buf = new byte[(int) length];
            read(buf);
            close();
            return new String(buf, charset == null ? Charset.defaultCharset() : charset);
        } else return null;
    }

    public String text() throws IOException { return text(Charset.defaultCharset()); }


    /**
     * 转存到文件
     * @param target 目标文件
     * @param bufSize 复制缓存大小
     */
    public File transferTo(File target, int bufSize) throws IOException {
        if (target == null) throw new IllegalArgumentException("Param target required");
        target.getParentFile().mkdirs();
        if (!target.exists()) target.createNewFile();
        if (length > 0) {
            try (OutputStream fos = Files.newOutputStream(target.toPath())) {
                byte[] bs = new byte[bufSize];
                while (true) {
                    int length = read(bs);
                    if (length == -1) break;
                    fos.write(bs, 0, length);
                }
            }
            close();
        }
        return target;
    }

    public File transferTo(File target) throws IOException {
        return transferTo(target, 1024 * 5);
    }

    /**
     * 内容
     */
    public byte[] content() throws IOException {
        if (length > 0) {
            byte[] buf = new byte[(int) length];
            read(buf);
            close();
            return buf;
        } else return null;
    }


    /**
     * 是否读取完成
     */
    public boolean isEnd() { return readCnt >= length; }


    /**
     * 还剩多少没接收
     */
    public long leftReceived() { return length - received; }



    /**
     * 已读长度
     */
    public long getReadCnt() { return readCnt; }


    /**
     * 当前还剩多少长度没读
     */
    public long getLeftRead() { return received - readCnt; }


    public long getReceived() { return received; }
}
