package cn.xnatural.xnet;

import java.nio.ByteBuffer;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * web socket 连接实例
 */
public class WebSocket implements Protocol {
    // 关联的Http aio 会话
    protected final HttpIOSession   session;
    // 关联的请求
    public final    HttpRequest      request;
    // 消息监听
    protected final List<WsListener> listeners = new LinkedList<>();
    // 数据解析器
    public final WsDecoder        decoder   = new WsDecoder(this);
    // 关闭标识
    protected final AtomicBoolean    closed    = new AtomicBoolean(false);


    public WebSocket(HttpRequest request) {
        this.session = request.ioSession();
        this.request = request;
    }


    /**
     * 发送消息
     * synchronized 避免 WritePendingException
     * @param msg 消息
     */
    public void send(String msg) {
        if (closed.get()) return;
        session.write(encode(msg));
    }


    /**
     * 关闭 当前 websocket
     */
    @Override
    public void close() {
        if (closed.compareAndSet(false, true)) {
            session.close();
            listeners.forEach(l -> l.onClose(this));
        }
    }


    /**
     * 设置消息监听
     * @param listener {@link WsListener}
     * @return {@link WebSocket}
     */
    public WebSocket listen(WsListener listener) { listeners.add(listener); return this; }


    /**
     * {@link #encode(byte[], byte)}
     * @param msg 消息
     * @return {@link ByteBuffer}
     */
    protected ByteBuffer encode(String msg) {
        return encode(msg.getBytes(session.server.getCharset()), (byte) 1);
    }


    /**
     * 编码 响应 文档消息
     * 编码参考: tio WsServerEncoder
     * @param body 消息体
     * @param opCode opCode
     * @return {@link ByteBuffer}
     */
    public static ByteBuffer encode(byte[] body, byte opCode) {
        ByteBuffer buf;
        byte header0 = (byte) (0x8f & (opCode | 0xf0));
        if (body.length < 126) {
            buf = ByteBuffer.allocate(2 + body.length);
            buf.put(header0);
            buf.put((byte) body.length);
        } else if (body.length < (1 << 16) - 1) {
            buf = ByteBuffer.allocate(4 + body.length);
            buf.put(header0);
            buf.put((byte) 126);
            buf.put((byte) (body.length >>> 8));
            buf.put((byte) (body.length & 0xff));
        } else {
            buf = ByteBuffer.allocate(10 + body.length);
            buf.put(header0);
            buf.put((byte) 127);
            // buf.put(new byte[] { 0, 0, 0, 0 });
            buf.position(buf.position() + 4);
            buf.put((byte) (body.length >>> 24));
            buf.put((byte) (body.length >>> 16));
            buf.put((byte) (body.length >>> 8));
            buf.put((byte) (body.length & 0xff));
        }
        buf.put(body);
        buf.flip();
        return buf;
    }


    public HttpIOSession getSession() {return session;}


    @Override
    public ProtocolDecoder decoder(HttpIOSession ioSession) { return decoder; }
}
