package cn.xnatural.xnet;

import java.time.Duration;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Consumer;
import java.util.stream.Collectors;

/**
 * 分片文件上传处理器
 */
public abstract class PieceFileHandler extends HttpHandler {

    public static final String X_PIECE_FILE = "X-PieceFile";
    /**
     * 分片上传映射
     */
    protected final Map<String, FileData> pieceUploadMap = new ConcurrentHashMap<>();


    public PieceFileHandler(String path) {
        super(X_PIECE_FILE, path);
    }

    public PieceFileHandler(String path, Set<String> method, Set<String> consume, Set<String> produce) {
        super(X_PIECE_FILE, path, method, consume, produce);
    }


    @Override
    public boolean handle(HttpContext hCtx) throws Throwable {
        pieceUpload(hCtx);
        return false;
    }


    /**
     * <pre>
     *     x-pieceupload-id: 上传id(必须)
     *     x-pieceupload-length: 文件大小(第一次必须)
     *     x-pieceupload-filename: 文件名(第一次可选)
     * </pre>
     */
    protected void pieceUpload(HttpContext hCtx) throws Exception {
        String uploadId = hCtx.request().getHeader("x-pieceupload-id");
        if (uploadId == null || uploadId.isEmpty()) {
            hCtx.render(R.fail("header x-pieceupload-id required"));
            return;
        }
        String key = hCtx.request().getPath() + "_" + uploadId;
        FileData convergeFd = pieceUploadMap.get(key);
        XioStream xioStream = convergeFd == null ? null : (XioStream) convergeFd.getInputStream();

        List<FileData> pieces = hCtx.request().getFormParams().values().stream()
                .filter(o -> o instanceof FileData).map(o -> (FileData) o)
                .collect(Collectors.toList());
        if (pieces.isEmpty()) {
            if (convergeFd == null) hCtx.render(R.fail("Not found uploadId: " + uploadId));
            else hCtx.render(R.ok()
                    .attr("uploadId", uploadId)
                    .attr("fileId", convergeFd.getFinalName())
                    .attr("end", xioStream.isEnd())
                    .attr("leftRead", xioStream.getLeftRead()));
            return;
        }

        HttpServer srv = hCtx.param(null, HttpServer.class);
        // 2. 第一个分片上传
        if (convergeFd == null) {
            synchronized (pieceUploadMap) {
                convergeFd = pieceUploadMap.get(key);
                if (convergeFd == null) {
                    convergeFd = new FileData();
                    Long length = Optional.ofNullable(hCtx.request().getHeader("x-pieceupload-length")).map(Long::valueOf).orElse(null);
                    if (length == null) {
                        hCtx.render(R.fail("header x-pieceupload-length required"));
                        return;
                    }
                    if (length < 1) {
                        hCtx.render(R.fail("header x-pieceupload-length value must > 0"));
                        return;
                    }
                    if (length > srv._fileMaxLength.get()) {
                        hCtx.render(R.fail("header x-pieceupload-length value must < " + srv._fileMaxLength));
                        return;
                    }
                    pieceUploadMap.put(key, convergeFd);
                    xioStream = new XioStream(length);
                    convergeFd.setInputStream(xioStream);
                    convergeFd.setOriginName(hCtx.request().getHeader("x-pieceupload-filename"));
                    convergeFd.setSize(length);

                    // 清理时间长的 和 已结束的流
                    long expire = Duration.ofMinutes(srv.getAttr("pieceUpload.expire", Long.class, 180L)).toMillis();
                    for (Iterator<Map.Entry<String, FileData>> it = pieceUploadMap.entrySet().iterator(); it.hasNext(); ) {
                        Map.Entry<String, FileData> e = it.next();
                        XioStream stream = (XioStream) e.getValue().getInputStream();
                        if (stream.isEnd() || System.currentTimeMillis() - stream.createTime.getTime() > expire) it.remove();
                    }
                    // 转发一个新的内部请求到Controller层
                    srv.exec(() -> {
                        try {
                            doHandle(new HttpContext(hCtx.path, (c) -> hCtx.id(), (n, t) -> {
                                if (t != null && FileData.class.isAssignableFrom(t)) return pieceUploadMap.get(key);
                                if (t != null && FileData[].class.equals(t)) return new FileData[]{pieceUploadMap.get(key)};
                                return hCtx.param(n, t);
                            }, srv) {
                                @Override
                                public HttpRequest request() { return hCtx.request(); }

                                @Override
                                public void render(Object body, Consumer afterFn) { }

                                @Override
                                public Object render(Object body) { return null; }
                            });
                        } catch (Throwable e) {
                            srv.errHandle(hCtx, e);
                        }
                    });
                }
            }
        }
        for (FileData piece : pieces) {
            xioStream.addStream(piece.getInputStream(), (long) piece.getInputStream().available());
        }
        hCtx.render(
                R.ok().attr("uploadId", uploadId)
                        .attr("fileId", convergeFd.getFinalName())
                        .attr("end", xioStream.isEnd())
                        .attr("leftRead", xioStream.getLeftRead())
        );
    }


    public abstract void doHandle(HttpContext ctx) throws Throwable;
}
