package cn.xnatural.xnet;

import cn.xnatural.xchain.IMvc;
import com.alibaba.fastjson2.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.StandardSocketOptions;
import java.nio.ByteBuffer;
import java.nio.channels.AsynchronousSocketChannel;
import java.nio.charset.Charset;
import java.security.AccessControlException;
import java.time.Duration;
import java.util.*;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.function.BiFunction;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;

public class HttpServer extends XNetBase implements IMvc<HttpContext> {
    protected static final Logger log = LoggerFactory.getLogger(HttpServer.class);
    public static final    String PRIVILEGES = "privileges";
    protected final HChain chain = new HChain(this);
    /**
     * 当前连接
     */
    protected final Queue<HttpIOSession> connections      = new ConcurrentLinkedQueue<>();
    /**
     * 请求/响应 io 字节编码
     */
    protected final Lazier<Charset>       _charset         = new Lazier<>(() -> Charset.forName(getAttr("charset",  String.class,"utf-8")));
    /**
     * 文件最大长度限制(即: 分片上传的最大文件限制). 默认500M
     */
    protected final Lazier<Long>          _fileMaxLength   = new Lazier<>(() -> {
        Long l = getAttr("fileMaxLength", Long.class, null);
        return l == null ? 1024 * 1024 * 500L : l;
    });
    /**
     * 忽略打印的请求路径后缀
     */
    protected final Lazier<Set<String>>   _ignoreLogSuffix = new Lazier<>(() -> {
        Set<String> set = new HashSet<>(Arrays.asList(".js", ".css", ".html", ".vue", ".png", ".jpg", ".jpeg", ".ttf", ".woff", ".woff2", ".ico", ".map"));
        for (String suffix : getAttr("ignoreLogUrlSuffix", String.class,"").split(",")) {
            if (suffix != null && !suffix.trim().isEmpty()) set.add(suffix.trim());
        }
        return set;
    });
    /**
     * 忽略打印的请求路径前缀
     */
    protected final Lazier<Set<String>>   _ignoreLogPrefix = new Lazier<>(() -> {
        Set<String> set = new HashSet<>();
        for (String suffix : getAttr("ignoreLogPrefix", String.class,"").split(",")) {
            if (suffix != null && !suffix.trim().isEmpty()) set.add(suffix.trim());
        }
        return set;
    });
    // 属性委托,一般用于http的session持久化
    protected Function<HttpContext, Map<String, Object>> attrDelegateSupplier;

    public HttpServer(XNet xNet) {
        super(xNet);
    }

    @Override
    public Object getAttr(String key) { return super.getAttr("http." + key); }

    @Override
    public HChain chain() { return chain; }


    @Override
    public void close() {
        // 尽可能等待请求处理结束再关
        for (
                long waitCount = 0, perWait = 100, waitLimit = getAttr("closeWaitLimit",  Long.class,3000L);
                waitCount < waitLimit && connections.stream().anyMatch(conn -> conn.request != null && conn.request.hCtx != null);
                waitCount += perWait
        ) {
            try {
                Thread.sleep(perWait);
            } catch (InterruptedException e) {
                log.error("", e);
            }
        }
    }


    /**
     * 接收新的 http 请求
     * @param request {@link HttpRequest}
     */
    protected void receive(HttpRequest request) {
        HttpContext hCtx = null;
        try {
            WebSocket ws = WSHandler.WEB_SOCKET.equalsIgnoreCase(request.getUpgrade()) ? new WebSocket(request) : null;
            request.hCtx = hCtx = new HttpContext(request.getRowUrl(), ctx -> request.getId(), new BiFunction<String, Class, Object>() {
                @Override
                public Object apply(String k, Class t) {
                    if (t != null && XNet.class.isAssignableFrom(t)) return xNet;
                    if (t != null && WebSocket.class.isAssignableFrom(t)) return ws;
                    Object v = request.getFormParams().get(k);
                    if (v == null) v = request.getJsonParams().get(k);
                    return v;
                }

                // 用于日志打印参数
                @Override
                public String toString() { return Objects.toString(request.getBodyStr(), ""); }
            }, this) {
                @Override
                public String protocol() { return request.getUpgrade() == null ? request.getProtocol() : request.getUpgrade(); }

                @Override
                public String method() { return request.method; }

                @Override
                public String contentType() { return request.getContentType(); }

                Set<String> _accept;
                @Override
                public Set<String> accept() {
                    if (_accept == null) {
                        _accept = Optional.ofNullable(request.getAccept()).map(s -> Arrays.stream(s.split(","))
                                .map(String::trim).collect(Collectors.toSet())).orElse(Collections.emptySet());
                    }
                    return _accept;
                }

                @Override
                public HttpRequest request() { return request; }
            };
            if (connections.size() <= getAttr("maxConnection", Integer.class,1024)) {
                handle(hCtx, request, request.session, attrDelegateSupplier == null ? null : attrDelegateSupplier.apply(hCtx));
            } else {
                hCtx.response.status(503);
                hCtx.render(R.fail("server busy, please wait..."));
            }
        } catch (Exception ex) {
            log.error("("+request.getId()+"), Handle error", ex);
            if (hCtx != null) {
                hCtx.response.status(500); hCtx.render();
            }
        }
    }


    /**
     * 设置属性委托
     */
    public HttpServer attrDelegate(Function<HttpContext, Map<String, Object>> attrDelegateSupplier) {
        this.attrDelegateSupplier = attrDelegateSupplier;
        return this;
    }


    @Override
    public void render(HttpContext ctx, Object body, Consumer<Object> afterFn) {
        if (ctx.closed.get()) throw new RuntimeException("Already closed");
        if (!ctx.commit.compareAndSet(false, true)) {
            throw new RuntimeException("Already submit response");
        }
        long spend = System.currentTimeMillis() - ctx.request().createTime.getTime();
        if (spend > getAttr("logWarnTimeout", Integer.class, 5000)) { // 请求超时警告单位ms
            log.warn("{}, timeout: {}ms", chain().logStr(ctx.protocol(), ctx.id(), ctx.method(), ctx.path), spend);
        }

        try {
            String etag = getETag(ctx, body);
            if (body == null) { //无内容返回
                ctx.response.statusIfNotSet(204);
                ctx.response.contentLengthIfNotSet(0);
                ctx.ioSession().write(ByteBuffer.wrap(ctx.preRespBytes()));
            } else if (etag != null && !etag.isEmpty() && Objects.equals(etag, ctx.request().getIfNoneMatch())) {
                ctx.response.statusIfNotSet(304);
                ctx.response.contentLengthIfNotSet(0);
                ctx.ioSession().write(ByteBuffer.wrap(ctx.preRespBytes()));
            } else {
                ctx.response.statusIfNotSet(200);
                if (etag != null && !etag.isEmpty()) ctx.response.etag(etag);
                // HttpResponseEncoder
                if (body instanceof String) { //回写字符串
                    ctx.response.contentTypeIfNotSet("text/plain;charset=" + getCharset());
                    ctx.renderBytes(((String) body).getBytes(getCharset()));
                } else if (body instanceof R) {
                    ctx.response.contentTypeIfNotSet("application/json;charset=" + getCharset());
                    ((R) body).setMark((String) ctx.param("mark"));
                    ((R) body).setId(ctx.id());
                    ctx.renderBytes(responseToJsonStr(body).getBytes(getCharset()));
                } else if (body instanceof byte[]) {
                    ctx.renderBytes((byte[]) body);
                } else if (body instanceof InputStream) {
                    ctx.renderInputStream((InputStream) body);
                } else if (body instanceof File) {
                    ctx.renderFile((File) body);
                } else if (ctx.response.getContentType() != null) {
                    String ct = ctx.response.getContentType();
                    if (ct.contains("application/json")) {
                        ctx.renderBytes(responseToJsonStr(body).getBytes(getCharset()));
                    } else if (ct.contains("text/plain")) {
                        ctx.renderBytes(body.toString().getBytes(getCharset()));
                    } else throw new Exception("Not support response Content-Type: " + ct);
                } else throw new Exception("Not support response type: " + body.getClass().getName());
            }
            ctx.determineClose();
        } catch (Exception ex) {
            log.error("("+ ctx.id() +"): response error", ex);
            close();
        } finally {
            if (afterFn != null) afterFn.accept(null);
            ctx.request().hCtx = null; // 代表关联的连接可以被关闭
            for (Runnable fn : ctx.finishedFns) {
                fn.run();
            }
        }
    }

    @Override
    public void errHandle(HttpContext hCtx, Throwable ex) {
        if (ex instanceof AccessControlException) {
            log.warn(chain().logStr(hCtx.protocol(), hCtx.id(), hCtx.method(), hCtx.path) + ", " + ex.getMessage() + ", from: " + hCtx.ioSession().getRemoteAddress());
            hCtx.render(R.of(hCtx.response.status + "", ex.getMessage()));
            return;
        }
        log.error(chain().logStr(hCtx.protocol(), hCtx.id(), hCtx.method(), hCtx.path) + ", from: " + hCtx.ioSession().getRemoteAddress(), ex);
        hCtx.render(R.fail((ex.getClass().getSimpleName() + (ex.getMessage() != null ? ": " + ex.getMessage() : ""))));
    }


    @Override
    protected void accept(AsynchronousSocketChannel channel, Consumer<Boolean> confirm) {
        exec(() -> {
            HttpIOSession se = null;
            try {
                // channel.setOption(StandardSocketOptions.SO_REUSEADDR, true);
                // channel.setOption(StandardSocketOptions.SO_RCVBUF, getInteger("so_rcvbuf", 1024 * 1024 * 2));
                // 必须要设置这个值, 不然有的环境(docker) 在chunked时返回不了正确的文件大小
                // channel.setOption(StandardSocketOptions.SO_SNDBUF, getInteger("so_sndbuf", 1024 * 1024 * 10)); // 必须大于 chunk 最小值
                channel.setOption(StandardSocketOptions.SO_KEEPALIVE, true);
                channel.setOption(StandardSocketOptions.TCP_NODELAY, true);

                se = new HttpIOSession(channel, this) {
                    @Override
                    protected void doClose(HttpIOSession session) { // 立即删除,释放内存
                        connections.remove(session);
                    }
                };
                se.start(confirm);
            } catch (Throwable e) { // 用Throwable, 不然有异常, 系统线程就会死掉
                if (se != null) se.close();
                else {
                    try { channel.close(); } catch (IOException ex) {}
                }
                log.error("Create " + HttpIOSession.class.getSimpleName() + " error", e);
            }
        });
    }


    /**
     * 清理链接
     */
    protected void clean() {
        if (connections.isEmpty()) return;
        int size = connections.size();
        long httpExpire = Duration.ofSeconds(getAttr("connection.maxIdle", Integer.class,
                ((Supplier<Integer>) () -> {
                    if (size > 200) return 10;
                    if (size > 128) return 30;
                    if (size > 80) return 60;
                    if (size > 50) return 120;
                    if (size > 30) return 180;
                    if (size > 20) return 240;
                    if (size > 10) return 300;
                    return 360;
                }).get()
        )).toMillis();
        long wsExpire = Duration.ofSeconds(getAttr("wsConnection.maxIdle", Integer.class,
                ((Supplier<Integer>) () -> {
                    if (size > 60) return 300;
                    if (size > 40) return 600;
                    if (size > 20) return 1200;
                    return 1800;
                }).get()
        )).toMillis();

        for (Iterator<HttpIOSession> it = connections.iterator(); it.hasNext(); ) {
            HttpIOSession se = it.next();
            if (se == null) it.remove();
            else if (se.closed.get()) it.remove();
            else if (!se.channel.isOpen()) {
                it.remove(); se.close();
                log.info("Cleaned unavailable {}: {}, connected: {}", se, connections.size(), HttpIOSession.class.getSimpleName());
            } else if (WSHandler.WEB_SOCKET.equals(se.request.getUpgrade()) && System.currentTimeMillis() - se.lastUsed > wsExpire) {
                it.remove(); se.protocol.close();
                log.debug("Closed expired (WS){}: {}, connected: {}", HttpIOSession.class.getSimpleName(), se, connections.size());
            }
            // 不清理正在使用的连接
            else if (se.request.hCtx == null && System.currentTimeMillis() - se.lastUsed > httpExpire) {
                it.remove(); se.close();
                log.debug("Closed expired {}: {}, connected: {}", HttpIOSession.class.getSimpleName(), se, connections.size());
            }
        }
    }


    //>>>>>>>>>>>>>>>>>>>>>>>>json序列化相关>>>>>>>>>>>>>>>>>>>>>>//

    protected String toJsonStr(Object o) {
        JSONWriter.Context context = new JSONWriter.Context();
        context.setDateFormat("millis"); // 所有时间默认都转换成时间戳
        context.config(JSONWriter.Feature.WriteMapNullValue, JSONWriter.Feature.WriteEnumsUsingName);
        return JSON.toJSONString(o, context);
    }


    protected Map<String, Object> jsonStrToMap(String jsonStr) {
        return JSON.parseObject(jsonStr, JSONObject.class, JSONReader.Feature.AllowUnQuotedFieldNames);
    }


    //<<<<<<<<<<<<<<<<<<<<<<<<<json序列化相关<<<<<<<<<<<<<<<<<<<<<//


    /**
     * api 接口 响应 对象 转 json 字符串
     * @param resp 对象
     */
    protected String responseToJsonStr(Object resp) { return toJsonStr(resp); }

    /**
     * 请求的 json body 转换成 map
     * @param jsonBody json 字符串
     * @return 参数 map 映射
     */
    protected Map<String, Object> requestJsonToMap(String jsonBody) {
        if (jsonBody == null || jsonBody.isEmpty()) return Collections.emptyMap();
        try {
            return Collections.unmodifiableMap(jsonStrToMap(jsonBody));
        } catch (JSONException ex) {
            log.error("Request body is not a JSON: " + jsonBody);
        }
        return Collections.emptyMap();
    }



    /**
     * 分段传送, 每段大小
     * NOTE: 限于带宽(带宽必须大于分块的最小值, 否则会导致前端接收数据不全)
     * @param hCtx HttpContext
     * @param body 返回的对象
     * @return 每段大小. >0: 分片
     */
    protected int chunkedSize(HttpContext hCtx, Object body) {
        if (body == null) return -1;
        long size;
        if (File.class.isAssignableFrom(body.getClass())) {
            size = ((File) body).length();
            if (size > 1024 * 1024 * 30) return 1024 * 1024;
            else if (size > 1024 * 1024 * 10) return 1024 * 200;
            else if (size > 1024 * 80) return 1024 * 10;
            // 小文件不需要分段传送
        } else if (byte[].class.equals(body.getClass())) {
            size = ((byte[]) body).length;
            if (size > 1024 * 1024 * 30) return 1024 * 1024;
            else if (size > 1024 * 1024 * 10) return 1024 * 200;
            else if (size > 1024 * 80) return 1024 * 10;
        } else if (InputStream.class.isAssignableFrom(body.getClass())) {
            // available 可读字节数 不可靠
            return getAttr("inputStreamChunkedSize", Integer.class,20 * 1024);
        }
        return -1; // 不分块传输
    }


    /**
     * 根据相应body生成etag,用于缓存
     */
    protected String getETag(HttpContext hCtx, Object body) {
        if (body instanceof File) return ((File) body).lastModified() + "";
        return hCtx.response.header("etag");
    }


    /**
     * 权限验证
     * @param permissions 权限名 验证用户是否有此权限, 没有权限名代表: 验证当前会话是否有有效
     * @return true: 验证通过
     */
    protected boolean auth(HttpContext hCtx, String... permissions) {
        if (!hasAuth(hCtx, permissions)) {
            hCtx.response.statusIfNotSet(403);
            throw new AccessControlException(HttpResponse.statusMsg.get(hCtx.response.status));
        }
        return true;
    }


    /**
     * 是否存在权限
     * @param permissions 权限名 验证用户是否有此权限
     * @return true: 存在
     */
    protected boolean hasAuth(HttpContext hCtx, String... permissions) {
        Set<String> pIds = hCtx.getAttr(PRIVILEGES, Set.class);
        if (pIds == null) {
            hCtx.response.statusIfNotSet(401);
            return false;
        }
        if (permissions != null && permissions.length > 0) {
            for (String permission : permissions) {
                if (pIds.contains(permission)) return true;
            }
            return false;
        }
        return true;
    }


    /**
     * https
     * @return
     */
//    protected Tuple2<PrivateKey, X509Certificate> security() {
//        SecureRandom random = new SecureRandom()
//        def gen = KeyPairGenerator.getInstance("RSA")
//        gen.initialize(2048, random)
//        def pair = gen.genKeyPair()
//
//        X509CertInfo info = new X509CertInfo()
//        X500Name owner = new X500Name("C=x,ST=x,L=x,O=x,OU=x,CN=x")
//        info.set(X509CertInfo.VERSION, new CertificateVersion(CertificateVersion.V3))
//        info.set(X509CertInfo.SERIAL_NUMBER, new CertificateSerialNumber(new BigInteger(64, random)))
//        try {
//            info.set(X509CertInfo.SUBJECT, new CertificateSubjectName(owner))
//        } catch (CertificateException ignore) {
//            info.set(X509CertInfo.SUBJECT, owner)
//        }
//        try {
//            info.set(X509CertInfo.ISSUER, new CertificateIssuerName(owner));
//        } catch (CertificateException ignore) {
//            info.set(X509CertInfo.ISSUER, owner);
//        }
//        info.set(X509CertInfo.VALIDITY, new CertificateValidity(
//                new Date(System.currentTimeMillis() - 86400000L * 365),
//                new Date(253402300799000L))
//        )
//        info.set(X509CertInfo.KEY, new CertificateX509Key(pair.getPublic()));
//        info.set(X509CertInfo.ALGORITHM_ID, new CertificateAlgorithmId(new AlgorithmId(AlgorithmId.sha256WithRSAEncryption_oid)));
//
//        // Sign the cert to identify the algorithm that's used.
//        X509CertImpl cert = new X509CertImpl(info)
//        cert.sign(pair.private, "SHA256withRSA");
//
//        // Update the algorithm and sign again.
//        info.set(CertificateAlgorithmId.NAME + '.' + CertificateAlgorithmId.ALGORITHM, cert.get(X509CertImpl.SIG_ALG))
//        cert = new X509CertImpl(info);
//        cert.sign(pair.private, "SHA256withRSA")
//        cert.verify(pair.public)
//
//        return Tuple.tuple(pair.private, cert)
//    }

    /**
     * 字符集
     */
    public Charset getCharset() { return _charset.get(); }
}
