package cn.xnatural.xnet;

import cn.xnatural.xchain.IMvc;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.net.URLDecoder;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

/**
 * Http 请求数据
 */
public class HttpRequest {
    protected static final Logger              log        = LoggerFactory.getLogger(HttpRequest.class);
    // 请求的创建时间
    final                  Date                createTime = new Date();
    // HTTP/HTTPS
    protected              String              protocol;
    // GET/POST
    protected              String              method;
    // 原始url地址字符串
    protected              String              rowUrl;
    // http协议版本: 1.0/1.1/1.2
    protected              String              version;
    protected final        Map<String, String> headers    = new HashMap<>();
    protected final        HttpDecoder         decoder    = new HttpDecoder(this);
    // 请求关联的io连接
    protected final        HttpIOSession       session;
    // 关联的请求处理上下文
    protected              HttpContext         hCtx;


    HttpRequest(HttpIOSession session) { this.session = session; }


    public HttpIOSession ioSession() { return session; }


    private final Lazier<String> _id = new Lazier<>(() -> {
        String id = getHeader("X-Request-ID");
        return id == null || id.isEmpty() ? XNet.nanoId() : id;
    });
    /**
     * 请求id
     */
    public String getId() { return _id.get(); }


    private final Lazier<Map<String, String>> _cookies = new Lazier<>(() -> {
        String cookieStr = getHeader("Cookie");
        if (cookieStr == null) return null;
        else {
            Map<String, String> cookies = new HashMap<>();
            for (String entry : cookieStr.split(";")) {
                if (entry == null) continue;
                entry = entry.trim();
                if (entry.isEmpty()) continue;
                String[] arr = entry.split("=");
                cookies.put(arr[0], arr.length > 1 ? URLDecoder.decode(arr[1]) : null);
            }
            return Collections.unmodifiableMap(cookies);
        }
    });
    /**
     * cookie 值映射
     */
    public Map<String, String> getCookies() { return _cookies.get(); }

    /**
     * 请求url 的查询字符串 ? 后面那坨
     */
    public String getQueryStr() { return hCtx.getQueryStr(); }


    /**
     * 请求查询参数Map
     */
    public Map<String, Object> getQueryParams() { return hCtx.queryParams(); }


    // 懒计算(只计算一次)例子
    private final Lazier<String> _path = new Lazier<>(() -> {
        int i = rowUrl.indexOf("?");
        return i == -1 ? rowUrl : rowUrl.substring(0, i);
    });
    /**
     * 请求路径
     */
    public String getPath() { return _path.get(); }


    private final Lazier<Map<String, Object>> _formParams = new Lazier<>(() -> {
        String ct = getContentType();
        if (ct != null && ct.contains("application/x-www-form-urlencoded")) {
            return IMvc.formatQueryStr(getBodyStr(), ioSession().server.getCharset().toString());
        }
        if (ct != null && ct.contains("multipart/form-data") && decoder.multiForm != null) {
            return Collections.unmodifiableMap(decoder.multiForm);
        }
        return Collections.emptyMap();
    });
    /**
     * 表单参数
     * Content-Type: application/x-www-form-urlencoded
     */
    public Map<String, Object> getFormParams() { return _formParams.get(); }


    private final Lazier<Map<String, Object>> _jsonParams = new Lazier<>(() -> {
        String ct = getContentType();
        if (ct != null && ct.contains("application/json")) {
            return ioSession().server.requestJsonToMap(getBodyStr());
        }
        return Collections.emptyMap();
    });
    /**
     * json 参数
     */
    public Map<String, Object> getJsonParams() { return _jsonParams.get(); }


    /**
     * 请求头: Content-Type
     */
    public String getContentType() { return getHeader("Content-Type"); }

    /**
     * 请求头: Accept
     */
    public String getAccept() { return getHeader("Accept"); }

    /**
     * 请求头: Accept-Encoding
     */
    public String getAcceptEncoding() { return getHeader("Accept-Encoding"); }

    /**
     * 请求头: Connection
     */
    public String getConnection() { return getHeader("Connection"); }

    /**
     * 请求头: Host
     */
    public String getHost() { return getHeader("Host"); }

    /**
     * 请求头: User-Agent
     */
    public String getUserAgent() { return getHeader("User-Agent"); }

    /**
     * 请求头: Upgrade
     */
    public String getUpgrade() {
        String u = getHeader("Upgrade");
        return u == null || u.isEmpty() ? getXUpgrade() : u;
    }

    /**
     * 请求头: Upgrade 默认不让改变
     */
    public String getXUpgrade() { return getHeader("X-Upgrade"); }

    /**
     * 请求头: if-None-Match
     */
    public String getIfNoneMatch() {
        return getHeader("if-None-Match");
    }


    /**
     * 取Header值
     * @param hName header 名
     * @return header 值
     */
    public String getHeader(String hName) { return headers.get(hName.toLowerCase()); }


    /**
     * 取cookie值
     * @param cName cookie 名
     * @return cookie 值
     */
    public String getCookie(String cName) {
        if (getCookies() == null) return null;
        return getCookies().get(cName);
    }


    /**
     * http 协议: http, https
     * @return http or https
     */
    public String getProtocol() { return protocol; }

    /**
     * 请求方法: get, post...
     * @return get, post...
     */
    public String getMethod() { return method; }

    /**
     * 原始url
     * @return 原始url
     */
    public String getRowUrl() { return rowUrl; }

    /**
     * http 版本
     * @return 1.0 or 1.1
     */
    public String getVersion() { return version; }


    protected final Lazier<String> _bodyStr = new Lazier<>(() -> HttpDecoder.text(decoder.bodyContent, ioSession().server.getCharset()));
    /**
     * 请求body字符串
     * @return str
     */
    public String getBodyStr() { return _bodyStr.get(); }
}
