package cn.xnatural.xnet;

import cn.xnatural.xchain.Context;
import cn.xnatural.xchain.IMvc;
import com.alibaba.fastjson2.JSON;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.lang.reflect.Array;
import java.nio.ByteBuffer;
import java.util.*;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import static cn.xnatural.xchain.IMvc.to;
import static cn.xnatural.xnet.HttpServer.log;

/**
 * HTTP处理上下文
 */
public class HttpContext extends Context implements AutoCloseable {
    protected static final byte[] LAST_CHUNK_BYTES = {(byte) '0', (byte) '\r', (byte) '\n'};
    protected static final byte[] CRLF_BYTES      = {(byte) '\r', (byte) '\n'};
    protected static final String CRLF            = "\r\n";
    protected static final byte[] END_CHUNK_BYTES = {(byte) '0', (byte) '\r', (byte) '\n', (byte) '\r', (byte) '\n'};

    public final HttpResponse response = new HttpResponse();
    /**
     * 是否已经提交，只能提交一次
     */
    protected final AtomicBoolean        commit  = new AtomicBoolean(false);
    /**
     * 是否已关闭
     */
    protected final AtomicBoolean closed = new AtomicBoolean(false);
    /**
     * session 数据操作委托
     */
    protected final Lazier<Map<String, Object>> sessionSupplier = new Lazier<>(() -> attach.length > 2 ? (Map<String, Object>) attach[2] : null);
    /**
     * 结束回调函数
     */
    protected final List<Runnable> finishedFns = new LinkedList<>();


    /**
     * HTTP请求执行上下文
     *
     * @param path   路径路由
     * @param paramProvider  参数提供函数
     * @param server {@link HttpServer}
     */
    public HttpContext(String path, Function<Context, String> idProvider, BiFunction<String, Class, Object> paramProvider, IMvc server) {
        super(path, idProvider, paramProvider, server);
    }

    @Override
    public void close() {
        if (closed.compareAndSet(false, true)) {
            ioSession().close();
        }
    }


    public HttpRequest request() { return (HttpRequest) attach[0]; }


    public HttpIOSession ioSession() { return (HttpIOSession) attach[1]; }


    /**
     * 从cookie中取session 标识
     * @return session id(会话id)
     */
    public String getSessionId() {
        Map<String, Object> data = sessionSupplier.get();
        if (data == null) return null;
        return (String) data.get("id");
    }

    /**
     * 设置 session 属性
     * @param aName 属性名
     * @param value 属性值
     * @return {@link HttpContext}
     */
    public HttpContext setSessionAttr(String aName, Object value) {
        if (sessionSupplier.get() == null) return this;
        if (value == null) sessionSupplier.get().remove(aName);
        else sessionSupplier.get().put(aName, value);
        return this;
    }

    /**
     * 设置 session 懒计算属性
     * @param aName 属性名
     * @param value 属性值
     * @return {@link HttpContext}
     */
    public <T> HttpContext setSessionAttr(String aName, Supplier<T> value) {
        if (sessionSupplier.get() == null) return this;
        if (value == null) sessionSupplier.get().remove(aName);
        else sessionSupplier.get().put(aName, value);
        return this;
    }


    /**
     * 删除 session 属性
     * @param aName 属性名
     * @return 属性值
     */
    public Object removeSessionAttr(String aName) {
        if (sessionSupplier.get() == null) return null;
        return sessionSupplier.get().remove(aName);
    }


    /**
     * 获取 session 属性
     * @param aName 属性名
     * @return 属性值
     */
    public Object getSessionAttr(String aName) { return getSessionAttr(aName, null); }


    /**
     * 获取 session 属性
     * @param aName 属性名
     * @param aType 值类型
     * @return 属性值
     */
    public <T> T getSessionAttr(String aName, Class<T> aType) {
        if (sessionSupplier.get() == null) return null;
        Object v = sessionSupplier.get().get(aName);
        v = v instanceof Supplier ? ((Supplier<?>) v).get() : v;
        return to(v, aType);
    }


    /**
     * 注册结束时回调函数
     */
    public HttpContext regFinishedFn(Runnable fn) {
        finishedFns.add(fn);
        return this;
    }

    /**
     * 权限验证
     * @param permissions 权限名 验证用户是否有此权限
     * @return true: 验证通过
     */
    public boolean auth(String... permissions) { return ((HttpServer) server).auth(this, permissions); }

    /**
     * 是否存在权限
     * @param permissions 多个权限标识
     * @return true: 存在
     */
    public boolean hasAuth(String... permissions) { return ((HttpServer) server).hasAuth(this, permissions); }


    /**
     * 重定向
     */
    public void redirect(String url) {
        response.statusIfNotSet(301);
        response.header("Location", url);
        render();
    }


    /**
     * 响应请求
     */
    public void render() { render(null); }


    /**
     * http 响应的前半部分
     * 包含: 起始行, 公共 header
     * @return byte[]
     */
    protected byte[] preRespBytes() {
        StringBuilder sb = new StringBuilder();
        sb.append("HTTP/").append(request().getVersion()).append(" ").append(response.status).append(" ").append(HttpResponse.statusMsg.get(response.status)).append(CRLF); // 起始行
        response.headers.forEach((key, value) -> {
            key = Arrays.stream(key.split("-")).map(s -> s.length() > 0 ? Character.toUpperCase(s.charAt(0)) + "" + s.subSequence(1, s.length()) : "").collect(Collectors.joining("-"));
            sb.append(key).append(": ").append(value).append(CRLF);
        });
        response.cookies.forEach((key, value) -> sb.append("Set-Cookie: ").append(key).append("=").append(value).append(CRLF));
        return sb.append(CRLF).toString().getBytes(((HttpServer) server).getCharset());
    }


    /**
     * 写流
     */
    protected void renderInputStream(InputStream body) throws Exception {
        response.contentTypeIfNotSet("application/octet-stream");
        try (InputStream is = body) {
            chunked(((HttpServer) server).chunkedSize(this, body), is, -1);
        }
    }


    /**
     * 发送字节
     */
    protected void renderBytes(byte[] bodyBs) throws Exception {
        response.contentTypeIfNotSet("application/octet-stream");
        try (InputStream is = new ByteArrayInputStream(bodyBs)) {
            chunked(((HttpServer) server).chunkedSize(this, bodyBs), is, bodyBs.length);
        }
    }


    /**
     * 渲染文件
     * 分块传送. js 加载不全, 可能是网速限制的原因
     * @param file 文件
     */
    protected void renderFile(File file) throws Exception {
        if (!file.exists()) {
            response.status(404);
            log.warn("Request({}): ({}){}, {}({})", id(), request().method, request().getRowUrl(), HttpResponse.statusMsg.get(response.status), response.status);
            response.contentLengthIfNotSet(0);
            ioSession().write(ByteBuffer.wrap(preRespBytes()));
            return;
        }
        if (file.getName().endsWith(".html")) {
            response.contentTypeIfNotSet("text/html");
        } else if (file.getName().endsWith(".css")) {
            response.contentTypeIfNotSet("text/css");
        } else if (file.getName().endsWith(".js")) {
            response.contentTypeIfNotSet("application/javascript");
        }

        try (InputStream is = new FileInputStream(file)) {
            chunked(((HttpServer) server).chunkedSize(this, file), is, file.length());
        }
    }


    /**
     * 分批发送数据 chunked
     * @param chunkedSize 分批大小
     * @param is 输入数据流
     * @param totalLength body总长度. >0: 长度已知,设置Content-Length; =0: 没有内容; <0: body 长度未知
     */
    protected void chunked(int chunkedSize, InputStream is, long totalLength) throws Exception {
        if (totalLength > 0) {
            chunkedSize = chunkedSize > 0 ? chunkedSize : (int) totalLength;
            response.contentLengthIfNotSet(totalLength);
            ioSession().write(ByteBuffer.wrap(preRespBytes())); // 1. 先写公共header
            while (!ioSession().closed.get()) {
                byte[] buf = new byte[chunkedSize]; // 数据缓存buf
                int len = is.read(buf); // 一批一批的读, 减少IO
                // 是否已结束(不能用: len < chunkedSize当读出来的数据少于chunkedSize)
                if (len == -1) break;
                if (len > 0) {
                    ioSession().write(ByteBuffer.wrap(buf, 0, len)); // 2. 写chunked: body
                }
            }
        } else if (totalLength == 0) {
            response.statusIfNotSet(204);
            response.contentLengthIfNotSet(0);
            ioSession().write(ByteBuffer.wrap(preRespBytes()));
        } else {
            chunkedSize = chunkedSize > 0 ? chunkedSize : 1024 * 10;
            response.transferEncoding("chunked");
            ioSession().write(ByteBuffer.wrap(preRespBytes())); // 1. 先写公共header
            while (!ioSession().closed.get()) {
                byte[] buf = new byte[chunkedSize]; // 数据缓存buf
                int len = is.read(buf); // 一批一批的读, 减少IO
                // 是否已结束(不能用: len < chunkedSize当读出来的数据少于chunkedSize)
                if (len == -1) break;
                if (len > 0) {
                    // tomcat org.apache.coyote.http11.filters.ChunkedOutputFilter()
                    ioSession().write(ByteBuffer.wrap((Integer.toHexString(len) + CRLF).getBytes())); // 1. 写chunked: header
                    ioSession().write(ByteBuffer.wrap(buf, 0, len)); // 2. 写chunked: body
                    ioSession().write(ByteBuffer.wrap(CRLF_BYTES)); // 3. 写chunked: end
                }
            }
            //3. 结束chunk
            ioSession().write(ByteBuffer.wrap(END_CHUNK_BYTES));
        }
    }


    /**
     * 判断是否应该关闭此次Http连接会话
     */
    protected void determineClose() {
        String connection = request().getConnection();
        if (
                (connection != null && connection.toLowerCase().contains("close")) ||
                404 == response.status || 503 == response.status || 500 == response.status
        ) {
            // http/1.1 规定 只有显示 connection:close 才关闭连接
            close();
        }
    }


    @Override
    public <T> T param(String pName, Class<T> type) {
        T t = super.param(pName, type);
        if ((t == null || !t.getClass().equals(type)) && pName != null && type != null && (!type.getName().startsWith("java."))) { // 可能是javabean
            try {
                Map<String, Object> m = new HashMap<>();
                m.putAll(request().getJsonParams());
                m.putAll(request().getFormParams());
                m.putAll(request().getQueryParams());
                return JSON.to(type, m);
            } catch (Exception e) {
                // ignore
            }
        }
        return t;
    }
}
