package cn.xnatural.xnet;

import cn.xnatural.xchain.*;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Parameter;
import java.util.Arrays;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.Set;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * http Chain
 */
public class HChain extends Chain<HttpContext> {

    public HChain(HttpServer server) { super(server); }

    @Override
    protected void resolve(Object ctrl, Method method) {
        super.resolve(ctrl, method);
        resolveWS(ctrl, method);
    }


    /**
     * 解析 {@link WS}
     * @param ctrl 控制层对象
     * @param method 方法
     */
    protected void resolveWS(Object ctrl, Method method) {
        WS aWS = method.getAnnotation(WS.class);
        if (aWS == null) return;
        if (!void.class.isAssignableFrom(method.getReturnType())) {
            log.warn("@WS return type must be void. {}#{}", ctrl.getClass().getName(), method.getName());
        }
        Route aCtrl = ctrl.getClass().getAnnotation(Route.class);
        Set<String> paths = Stream.of(aCtrl == null ? new String[]{"/"} : aCtrl.path()).flatMap(p -> Stream.of(aWS.path())
                .map(pp -> ("/" + p + "/" + pp).replaceAll("[/]+", "/")))
                .collect(Collectors.toCollection(LinkedHashSet::new));

        Parameter[] ps = method.getParameters();
        method.setAccessible(true);
        for (String path : paths) {
            ws(path, hCtx -> { // 实际@WS 方法 调用
                try {
                    method.invoke(ctrl, Arrays.stream(ps).map((p) -> hCtx.param(p.getName(), p.getType())).toArray());
                } catch (InvocationTargetException ex) {
                    throw ex.getCause();
                }
                return false;
            });
        }
    }


    @Override
    protected void resolveRoute_preProcess(Object ctrl, Method method, Set<String> paths, Set<String> protocols, Set<String> methods, Set<String> consumes, Set<String> produces) {
        if (Arrays.stream(method.getParameters()).anyMatch(p -> FileData.class.isAssignableFrom(p.getType()))) {
            if (consumes.stream().noneMatch(s -> s.toLowerCase().contains("multipart/form-data"))) {
                consumes.add("multipart/form-data");
            }
        }
        if (methods.isEmpty()) {
            for (String m : server.getAttr("defaultMethod", String.class, "GET,POST").split(",")) {
                if (m.isEmpty()) continue;
                if (m.trim().isEmpty()) continue;
                methods.add(m.trim().toUpperCase());
            }
        }
        super.resolveRoute_preProcess(ctrl, method, paths, protocols, methods, consumes, produces);
    }


    @Override
    protected RouteHandler<HttpContext> routeHandler(String protocol, String path, Set<String> method, Set<String> consume, Set<String> produce, Handler<HttpContext> handler) {
        if (PieceFileHandler.X_PIECE_FILE.equals(protocol)) {
            return new PieceFileHandler(path, method, consume, produce) {
                @Override
                public void doHandle(HttpContext ctx) throws Throwable {
                    handler.handle(ctx);
                }
            };
        }

        return new HttpHandler(protocol, path, method, consume, produce) {
            @Override
            public boolean handle(HttpContext ctx) throws Throwable {
                return handler.handle(ctx) && !ctx.commit.get();
            }
        };
    }


    @Override
    public Chain<HttpContext> filter(String protocol, String prefix, int order, Handler<HttpContext> handler) {
        return super.filter(protocol, prefix, order, ctx -> handler.handle(ctx) && !ctx.commit.get());
    }


    @Override
    protected boolean handle(HttpContext hCtx) {
        boolean match = super.handle(hCtx);
        if (hCtx.commit.get()) return match;
        if (hCtx.response.status != null) { // 已经设置了status
            log.warn("{}, {}({}), from: {}", logStr(hCtx.protocol(), hCtx.id(), hCtx.method(), hCtx.path),
                    HttpResponse.statusMsg.get(hCtx.response.status), hCtx.response.status, hCtx.ioSession().getRemoteAddress());
            hCtx.render();
            hCtx.close();
        }
        return match;
    }


    @Override
    protected void handleNoMatch(HttpContext ctx) {
        ctx.response.statusIfNotSet(404);
    }


    @Override
    protected boolean handle(PathHandler<HttpContext> h, HttpContext ctx) throws Throwable {
        if (((HttpServer) server)._ignoreLogPrefix.get().stream().noneMatch(ctx.path::startsWith)
                && ((HttpServer) server)._ignoreLogSuffix.get().stream().noneMatch(ctx.path::endsWith)) {
            String sid = ctx.getSessionId();
            log.info("Handle {}, {}, from: {}", logStr(ctx.protocol(), ctx.id() + (sid == null || sid.isEmpty() ? "" : ", " + sid), ctx.method(), ctx.path), ctx.paramProvider, ctx.ioSession().getRemoteAddress());
        }
        return h.handle(ctx);
    }


    /**
     * 添加方法路由执行
     * @param path 路由path
     * @param method 方法
     */
    public HChain method(String path, String method, Consumer<HttpContext> handler) {
        return (HChain) register(new HttpHandler(path, Collections.singleton(method), null, null) {
            @Override
            public boolean handle(HttpContext ctx) throws Throwable {
                handler.accept(ctx);
                return false;
            }
        });
    }

    public HChain get(String path, Consumer<HttpContext> handler) {
        return method(path, "get", handler);
    }

    public HChain post(String path, Consumer<HttpContext> handler) {
        return method(path, "post", handler);
    }

    public HChain put(String path, Consumer<HttpContext> handler) {
        return method(path, "put", handler);
    }

    public HChain delete(String path, Consumer<HttpContext> handler) {
        return method(path, "delete", handler);
    }


    /**
     * websocket监听处理器
     */
    public HChain ws(String path, Handler<HttpContext> handler) {
        register(new WSHandler(path) {
            @Override
            protected void doHandle(HttpContext hCtx) throws Throwable {
                handler.handle(hCtx);
            }
        });
        return this;
    }

}
