package cn.xnatural.xnet;

import cn.xnatural.xchain.Route;
import com.alibaba.fastjson2.JSON;
import com.alibaba.fastjson2.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.time.Duration;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.BiConsumer;
import java.util.function.Predicate;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import static cn.xnatural.xchain.IMvc.to;
import static cn.xnatural.xnet.R.fail;
import static cn.xnatural.xnet.R.ok;

/**
 * 集群处理中心
 */
@Route(path = "/", protocol = Cluster.X_CLUSTER)
public class Cluster extends XNetBase {
    protected static final Logger log = LoggerFactory.getLogger(Cluster.class);
    public static final String X_CLUSTER = "X-Cluster";
    protected static final String NODE_UPDATE = "nodeUpdate";
    protected static final String NODE_UP = "nodeUp";
    protected static final String NODE_DOWN = "nodeDown";
    /**
     * 应用名
     */
    protected final Lazier<String> _nodeName = new Lazier<>(() -> getAttr("name", String.class, null));
    /**
     * 应用实例id
     */
    protected final Lazier<String> _nodeId = new Lazier<>(() -> {
        String id = getAttr("id", String.class, null);
        return id == null || id.isEmpty() ? XNet.nanoId(10) : id;
    });
    /**
     * 保存集群中的应用节点信息
     * name -> Node
     */
    protected final Map<String, CopyOnWriteArrayList<Node>> nodeMap = new ConcurrentHashMap<>();
    /**
     * 请求id序列
     */
    protected final AtomicLong idSeq = new AtomicLong(1);


    public Cluster(XNet xNet) { super(xNet); }


    /**
     * 初始化
     */
    @Override
    protected void start() {
        if (getNodeName() == null || getNodeName().isEmpty()) throw new IllegalArgumentException("Cluster node name required");
        // 注册自己
        Node self = getMe();
        self._uptime = System.currentTimeMillis();
        getOrCreate(getNodeName()).add(self);
        log.info("Register self: {}, my master: {}", self, master().getData());

        xNet.http().chain().resolve(this); // 委托http协议转发请求
        heartbeat();
    }


    @Override
    public void close() {
        // 向master通知自己下线
        for (Iterator<Map.Entry<String, CopyOnWriteArrayList<Node>>> it = nodeMap.entrySet().iterator(); it.hasNext(); ) {
            Map.Entry<String, CopyOnWriteArrayList<Node>> e = it.next();
            for (Node n : e.getValue()) {
                if (!n._master) continue; // 用continue, 有可能有的master节点暂时还没同步到
                if (isMe(n)) continue;
                upgrade(http(n, NODE_DOWN + "/true").param("name", getNodeName()).param("id", getNodeId())
                        .connectTimeout(1000).readTimeout(2000)).debug().post();
                break;
            }
        }
    }


    @Override
    public Object getAttr(String key) { return super.getAttr("cluster." + key); }


    /**
     * 不定时调度同步
     */
    protected void heartbeat() {
        final Supplier<Integer> next = new Supplier<Integer>() {
            final String[] arr = getAttr("heartbeat", String.class, "30~180").split("~");
            final int min = Integer.parseInt(arr[0]);
            final int bound = Integer.parseInt(arr[1]) - min;
            // 启动时执行几次同步
            List<Integer> ls = Arrays.stream(getAttr("initTimes",  String.class, "5,15").split(","))
                    .map(String::trim).filter(s -> !s.isEmpty()).map(Integer::valueOf)
                    .collect(Collectors.toCollection(LinkedList::new));
            @Override
            public Integer get() {
                if (ls != null) {
                    if (ls.isEmpty()) ls = null;
                    else return ls.remove(0);
                }
                return new Random().nextInt(bound) + min;
            }
        };
        new Runnable() {
            @Override
            public void run() {
                try {
                    if (xNet.exec.isShutdown()) return;
                    sync();
                } catch (Throwable ex) {
                    log.error("Sched up error", ex);
                } finally {
                    xNet.sched(Duration.ofSeconds(next.get()), this);
                }
            }
        }.run();
    }


    /**
     * 接收节点的上传同步
     */
    @Route(path = NODE_UP, method = "post")
    R<?> nodeUp(final Node upNode, final Boolean backFeed) {
        if (upNode == null || upNode.name == null || upNode.hp == null || upNode.id == null) { // 数据验证
            return fail("Node up incomplete: " + upNode);
        }
        if (isMe(upNode)) return fail("Not allow receive self " + NODE_UP);
        upNode._uptime = System.currentTimeMillis();
        log.debug("Receive node up: {}", upNode);
        upNode.exposeTo = upNode.exposeTo == null ? Collections.singleton("*") : upNode.exposeTo;
        boolean exposeToMe = upNode.isExposeTo(getNodeName());
        Predicate<Node> isUpNode = n -> upNode.id.equals(n.id) && upNode.name.equals(n.name);

        // 1. 更新/新增
        AtomicBoolean isNew = new AtomicBoolean(false);
        if (exposeToMe) {
            List<Node> nodes = getOrCreate(upNode.name);
            Node exist = nodes.stream().filter(n -> upNode.id.equals(n.id)).findFirst().orElse(null);
            isNew.set(exist == null); // 是否是新来注册的节点
            if (exist == null) { // 新增
                nodes.add(upNode);
                log.info("New node online. {}", upNode);
            } else {
                boolean m = exist._master;
                exist.copyFrom(upNode); //更新
                exist._master = m;
            }
        }

        // 2. 数据修正
        long timeout = Duration.ofMinutes(getAttr("dropNodeTimeout", Long.class, 15L)).toMillis();
        for (Map.Entry<String, CopyOnWriteArrayList<Node>> e: nodeMap.entrySet()) {
            for (Node node : e.getValue()) {
                // 删除空的坏数据
                if (node == null) { e.getValue().remove(null); }
                // 删除和当前up的节点相同的hp的节点(节点重启,但节点上次的信息还没被移除)
                else if (node.hp.equals(upNode.hp) && !node.id.equals(upNode.id) && !node.name.equals(upNode.name)) {
                    e.getValue().remove(node);
                    log.debug("Drop same hp and conflict app node: {}", node);
                }
                // 更新当前节点的信息. 超过1分钟更新一次,不必每次更新
                else if (isMe(node)) {
                    if (upNode._uptime - node._uptime > getAttr("updateSelfTimeout", Integer.class, 1000 * 60)) {
                        node.copyFrom(getMe());
                        node._uptime = upNode._uptime;
                    }
                }
                // 删除一段时间未活动的注册信息, dropAppTimeout 单位: 分钟
                // TODO 万一是节点被隔离超时?
                else if (upNode._uptime - node._uptime > timeout) {
                    e.getValue().remove(node);
                    log.warn("Drop timeout node: {}", node);
                }
            }

            if (e.getValue().isEmpty()) nodeMap.remove(e.getKey());
        }

        BiConsumer<Node, Node> doUpdate = (to, src) -> {
            try {
                upgrade(http(to, NODE_UPDATE).jsonBody(JSONObject.toJSONString(src))).debug(log.isDebugEnabled()).post();
            } catch (Exception ex) {
                log.error(NODE_UPDATE + " error. " + to + "; " + src, ex);
            }
        };

        // 2. 同步(other -> upNode)
        if (upNode.syncToMe) {
            exec(() -> {
                nodeMap.entrySet().stream()
                        .filter(e -> Boolean.TRUE.equals(backFeed) || isNew.get() || e.getKey().equals(getNodeName()))
                        .flatMap(e -> e.getValue().stream())
                        .filter(n -> !isUpNode.test(n) && n.isExposeTo(upNode.name))
                        .forEach(n -> {
                            // 当前节点是所有来注册的节点的master
                            doUpdate.accept(upNode, isMe(n) ? n.copyTo(new Node()).set_master(true) : n);
                        });
            });
        }

        // 3. 同步(upNode -> other)
        exec(() -> {
            nodeMap.entrySet().stream()
                    .filter(e -> upNode.isExposeTo(e.getKey()))
                    .flatMap(e -> e.getValue().stream())
                    .filter(n -> !isMe(n) && !isUpNode.test(n) && n.syncToMe)
                    .forEach(n -> doUpdate.accept(n, upNode));
        });

        return ok();
    }


    /**
     * 接收节点更新
     */
    @Route(path = NODE_UPDATE, method = "post")
    R<?> nodeUpdate(final Node updateNode) {
        // 数据验证
        if (updateNode == null || updateNode.name == null || updateNode.name.isEmpty() || updateNode.hp == null ||
                updateNode.id == null || updateNode.id.isEmpty()) {
            return fail("Node info incomplete");
        }
        if (isMe(updateNode)) return fail("Not allow receive self " + NODE_UPDATE);
        updateNode.exposeTo = updateNode.exposeTo == null ? Collections.singleton("*") : updateNode.exposeTo;
        if (!updateNode.isExposeTo(getNodeName())) return fail("Not expose to me");
        updateNode._uptime = updateNode._uptime == null ? System.currentTimeMillis() : updateNode._uptime;
        Predicate<Node> isUpdateNode = n -> updateNode.id.equals(n.id) && updateNode.name.equals(n.name);

        long timeout = Duration.ofMinutes(getAttr("dropNodeTimeout", Long.class, 15L)).toMillis();
        getOrCreate(updateNode.name);
        for (Map.Entry<String, CopyOnWriteArrayList<Node>> e : nodeMap.entrySet()) {
            Node exist = null;
            for (Node node : e.getValue()) {
                if (isUpdateNode.test(node)) {
                    exist = node;
                    if (updateNode._uptime > node._uptime) {
                        node.copyFrom(updateNode);
                        log.debug("Update node: {}", node);
                    }
                } else if (node.hp.equals(updateNode.hp)) { // hp相同,id不同(可能是重启后,proxyHp可能相同). 同名app的保留，删除其他应用的
                    if (!updateNode.name.equals(e.getKey())) {
                        // appName不一样但hp一样(可能docker环境应用在其他机器上启了)
                        e.getValue().remove(node);
                        log.info("Drop same hp and conflict app node: {}", node);
                    }
                } else if (isMe(node)) {
                    if (updateNode._uptime - node._uptime > getAttr("updateSelfTimeout", Integer.class, 1000 * 60)) {
                        node.copyFrom(getMe());
                        node._uptime = updateNode._uptime;
                    }
                } else if (updateNode._uptime - node._uptime > timeout) {
                    e.getValue().remove(node);
                    log.info("Drop timeout node: {}", node);
                }
            }
            if (e.getKey().equals(updateNode.name) && exist == null) {
                e.getValue().add(updateNode);
                log.info("Add new node: " + updateNode);
            }
            if (e.getValue().isEmpty()) nodeMap.remove(e.getKey());
        }
        return ok();
    }


    /**
     * 接收节点下线
     */
    @Route(path = NODE_DOWN + "/{infect}", method = "post")
    R<?> nodeDown(final Boolean infect, String name, String id) {
        if (name == null || name.isEmpty()) return fail("Param name required");
        if (getNodeName().equals(name) && getNodeId().equals(id)) return fail("Not allow receive self " + NODE_DOWN);
        AtomicReference<Node> downNode = new AtomicReference<>();
        final List<Node> nodes = nodeMap.get(name);
        if (nodes != null) {
            if (id == null || id.isEmpty()) {
                log.info("Removed down nodes: " + nodes);
                nodeMap.remove(name);
            } else {
                for (Node node : nodes) {
                    if (Objects.equals(node.id, id)) {
                        nodes.remove(node);
                        downNode.set(node);
                        log.info("Removed down node: " + node);
                        break;
                    }
                }
                if (nodes.isEmpty()) nodeMap.remove(name);
            }
        }
        if (Boolean.TRUE.equals(infect)) { // 是否向其他节点通知
            exec(() -> {
                nodeMap.entrySet().stream()
                        .sorted((e1, e2) -> { // 先通知同名应用的节点
                            if (getNodeName().equals(e1.getKey())) return -1;
                            if (getNodeName().equals(e2.getKey())) return 1;
                            return 0;
                        })
                        .flatMap(e -> e.getValue().stream())
                        .filter(n -> !isMe(n) && !(n.id.equals(id) && n.name.equals(name)) && (downNode.get() == null || downNode.get().isExposeTo(n.name)) && n.syncToMe)
                        .forEach(n -> {
                            upgrade(http(n, NODE_DOWN + "/false")).param("name", name).param("id", id).debug().post();
                        });
            });
        }
        return ok();
    }


    @Route(path = "sync", method = "post")
    public R<?> sync() {
        return sync(getAttr("backFeed", Boolean.class, false));
    }

    /**
     * 向master同步自己
     * 每个master应用随机选择某个节点进行同步
     * @param backFeed 是否要求master立即同步所有节点给我
     */
    @Route(path = "sync/{backFeed}", method = "post")
    public R<?> sync(boolean backFeed) {
        master().getData().forEach((app, nodes) -> new Runnable() {
            final int connectTimeout = nodes.size() > 1 ? 1000 : 3000;
            final int readTimeout = nodes.size() > 1 ? 1500 : 5000;
            @Override
            public void run() { // 随便向一个节点上传, 直到成功
                if (nodes.isEmpty()) return;
                final Node n = nodes.remove(new Random().nextInt(nodes.size()));
                try {
                    R<?> resp = JSON.parseObject(upgrade(http(n, NODE_UP + "?backFeed=" + backFeed))
                            .jsonBody(JSONObject.toJSONString(getMe()))
                            .connectTimeout(connectTimeout).readTimeout(readTimeout).debug(log.isDebugEnabled()).post(), R.class);
                    if (R.OK_CODE.equals(resp.getCode())) return;
                } catch (Exception ex) {
                    String s = "Sync error. " + (app.isEmpty() ? "" : "(" + app + ") ") + n;
                    if (nodes.isEmpty()) log.error(s, ex);
                    else log.warn(s, ex);
                }
                run();
            }
        }.run());
        return ok();
    }


    /**
     * 获取一个appName下边的所有节点信息
     * @param appName 应用名
     */
    @Route(path = "nodes/{appName}", method = "get")
    public R<List<Node>> nodes(String appName) {
        return appName == null || appName.isEmpty() ? ok() : ok(nodeMap.get(appName));
    }


    /**
     * 获取所有应用名
     */
    @Route(path = "apps", method = "get")
    public R<Set<String>> apps() {
        return ok(nodeMap.keySet());
    }


    /**
     * 获取当前节点信息
     */
    @Route(path = "me", method = "get")
    public R<Node> me() {
        return ok(nodeMap.entrySet().stream()
                .filter(e -> e.getKey().equals(getNodeName()))
                .flatMap(e -> e.getValue().stream())
                .filter(this::isMe).findFirst().orElse(null)
        );
    }


    /**
     * master应用name -> 节点
     */
    @Route(path = "master", method = "get")
    public R<Map<String, List<Node>>> master() {
        String master = getAttr("master", String.class, null);
        if (master == null || master.isEmpty()) return ok(Collections.emptyMap());

        // 选出master组
        Map<String, List<Node>> masterApp = nodeMap.entrySet().stream()
                .filter(e -> e.getValue().stream().anyMatch(n -> n._master))
                .collect(Collectors.toMap(
                        Map.Entry::getKey,
                        e -> e.getValue().stream().filter(n -> !isMe(n)).collect(Collectors.toCollection(LinkedList::new))
                ));
        Arrays.stream(master.split(",")).map(hp -> {
            if (hp == null) return null;
            try {
                String[] arr = hp.trim().split(":");
                String host = arr[0].trim();
                return new Hp(host.isEmpty() ? "127.0.0.1" : host.trim(), Integer.valueOf(arr[1].trim()));
            } catch (Exception ex) {
                log.error("'master' config error. " + hp, ex);
            }
            return null;
        }).filter(Objects::nonNull).flatMap(hp -> {
            try {
                return Arrays.stream(InetAddress.getAllByName(hp.host)) // 如果是域名则取所有ip
                        .map(addr -> new Hp(addr.getHostAddress(), hp.port));
            } catch (UnknownHostException e) {
                log.error("", e);
            }
            return null;
        }).filter(Objects::nonNull).forEach(hp -> {
            if (masterApp.entrySet().stream().noneMatch(e -> e.getValue().stream().noneMatch(n -> n.hp.equals(hp)))) {
                List<Node> ls = masterApp.computeIfAbsent("", k -> new LinkedList<>());
                if (ls.stream().noneMatch(n -> n.hp.equals(hp))) ls.add(new Node().setHp(hp));
            }
        });
        if (log.isTraceEnabled()) log.trace("available master: {}", masterApp);
        return ok(masterApp);
    }


    protected CopyOnWriteArrayList<Node> getOrCreate(String name) {
        CopyOnWriteArrayList<Node> nodes = nodeMap.get(name);
        if (nodes == null) {
            synchronized (nodeMap) {
                nodes = nodeMap.get(name);
                if (nodes == null) {
                    nodes = new CopyOnWriteArrayList<>(); nodeMap.put(name, nodes);
                }
            }
        }
        return nodes;
    }


    /**
     * 应用自己的信息
     */
    public Node getMe() {
        Node me = new Node();
        me.id = getNodeId();
        me.name = getNodeName();
        me.syncToMe = getAttr("syncToMe", Boolean.class, true);
        String proxyHp = getAttr("proxyHp", String.class, null);
        if (proxyHp == null || proxyHp.isEmpty()) {
            me.hp = xNet.getHp().resolved();
        } else {
            me.hp = Hp.parse(proxyHp);
        }
        String g = getAttr("exposeTo", String.class, "*"); // 默认暴露给所有应用
        if (g != null && !g.isEmpty()) {
            me.exposeTo = Arrays.stream(g.split(",")).map(String::trim).filter(s -> !s.isEmpty()).collect(Collectors.toSet());
        } else {
            me.exposeTo = new HashSet<>(1);
        }
        return me;
    }


    /**
     * 是否是当前节点
     */
    public boolean isMe(Node n) {
        return getNodeName().equals(n.name) && getNodeId().equals(n.id);
    }


    /**
     * 节点名
     */
    public String getNodeName() { return _nodeName.get(); }

    /**
     * 节点id
     */
    public String getNodeId() { return _nodeId.get(); }



    // >>>>>>>>>>>>>>>>>>>>> 客户端功能 >>>>>>>>>>>>>>>>>>>>>

    /**
     * 向某个节点发送路由请求
     * @param node 节点
     * @param path 路由
     * @return {@link Httper}
     */
    public Httper http(Node node, String path) {
        return new Httper(url(node.getHp(), path)).id(requestId());
    }

    /**
     * 向当前同名其他节点发送路由请求
     */
    public Httper http(String path) {
        return http(getNodeName(), path);
    }

    /**
     * 向某个应用发送路由请求
     * <pre>
     * 例: {@code http("a", "test/cus").get()}
     * </pre>
     * @param appName 应用名
     * @param path 路由
     */
    public Httper http(String appName, String path) {
        List<Node> nodes = nodeMap.containsKey(appName)
                ? nodeMap.get(appName).stream().filter(n -> !isMe(n)).collect(Collectors.toList())
                : Collections.emptyList();
        if (nodes.isEmpty()) throw new RuntimeException("Not found available node for " + appName);
        AtomicReference<Node> _node = new AtomicReference<>(nodes.get(new Random().nextInt(nodes.size())));
        return new Httper(url(_node.get().getHp(), path)).id(requestId())
                .exHandler((ex, h) -> {
                    CopyOnWriteArrayList<Node> ns = nodeMap.get(appName);
                    if (ns != null && ns.size() > 1) ns.remove(_node.get());
                    if (XNet.isConnectError(ex)) {
                        nodes.remove(_node.get());
                        if (!nodes.isEmpty()) { // 继续尝试其他节点
                            _node.set(nodes.get(new Random().nextInt(nodes.size())));
                            h.urlStr = url(_node.get().getHp(), path);
                            h.execute();
                            return;
                        }
                    }
                    if (ex instanceof RuntimeException) throw (RuntimeException) ex;
                    else throw new RuntimeException(ex);
                });
    }


    protected String url(Hp hp, String path) {
        hp = hp.resolved();
        return  (hp.secure ? "https://" : "http://") + hp + (path.startsWith("/") ? path : "/" + path);
    }


    /**
     * 生成集群节点之间的请求id
     */
    protected String requestId() {
        return getNodeName() + ":" + getNodeId() + ":" + idSeq.getAndIncrement();
    }


    /**
     * 协议升级成 {@link #X_CLUSTER}
     */
    public Httper upgrade(Httper httper) {
        return httper.header("X-Upgrade", X_CLUSTER);
    }


    // <<<<<<<<<<<<<<<<<<<<< 客户端功能 <<<<<<<<<<<<<<<<<<<<<


    public <T> T getAttr(String key, Class<T> type, T defaultValue) {
        Object v = getAttr(key);
        if (v == null) return defaultValue;
        return to(v, type);
    }
}
