package cn.xnatural.xnet;

import cn.xnatural.xchain.Route;
import com.alibaba.fastjson2.JSON;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedDeque;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;

import static cn.xnatural.xchain.IMvc.to;
import static cn.xnatural.xnet.R.*;

/**
 * 分布式CP实现
 * <pre>
 *  1. 同一客户端的依次多次修改，保证查询到的是最后一次的修改
 *      1.1 第一次用节点a提交修改
 *      1.2 第二次用节点b提交修改
 *  2. 不同客户端并发提交修改，保证查询结果是在所有客户端最后一次修改之中的一个
 *  3. 数据同步(两阶段提交)
 *      3.1. 格式: key->value
 *          流程:
 *              任意节点收到修改请求, 转发给leader
 *              leader先持久化, 再同步给所有其他follower节点
 *              follower持久化, 返回给leader
 *              leader收到follower的结果后，再向所有follower发送commit, 都成功后，leader再commit
 *          流程中所有动作: leader 向所有节点同步数据的时候带上leaderId, follower与之匹配，才算成功
 * 3. 持久化
 * </pre>
 */
@Route(path = "/", protocol = CP.X_CP)
public class CP extends XNetBase {
    protected static final Logger log = LoggerFactory.getLogger(CP.class);
    public static final String X_CP = "X-cp";

    protected Node leader;
    protected final Lazier<File> _dir = new Lazier<>(() -> {
        File dir = new File(getAttr("dataDir", String.class, "./cp"));
        dir.mkdirs();
        return dir;
    });
    protected final AtomicReference<Status> _lock = new AtomicReference<>();
    protected final Map<String, Item> data = new ConcurrentHashMap<>();
    /**
     * 更新事务id
     */
    protected final AtomicLong _txId = new AtomicLong();


    enum Status {
        Vote, Update
    }


    public CP(XNet xNet) {
        super(xNet);
    }


    @Override
    public Object getAttr(String key) { return super.getAttr("cp." + key); }


    @Route(path = "vote", method = "post")
    R<String> vote(String leaderId, Optional<String> exclude, Optional<String> nodeIds) {
        Node n = voteLeader(leaderId, exclude.map(s -> Arrays.stream(s.split(",")).collect(Collectors.toSet())).orElse(null),
                nodeIds.map(s -> Arrays.stream(s.split(",")).collect(Collectors.toSet())).orElse(null));
        return ok(n == null ? null : n.id);
    }


    /**
     * 在集群中投票出一个leader
     * @param leaderId leader节点id
     * @param exclude 排除的leader节点id
     * @param nodeIds 已投过的节点id
     */
    public Node voteLeader(String leaderId, Set<String> exclude, Set<String> nodeIds) {
        if (_lock.compareAndSet(null, Status.Vote)) {
            try {
                Node oldLeader = this.leader; this.leader = null;
                Cluster c = xNet.cluster();
                Integer total = totalNode();
                if (total == null || total < 1) return null;
                List<Node> nodes = c.nodeMap.get(c.getNodeName());
                if (nodes.size() < total) {
                    exec(c::sync); return null;
                }
                Set<String> finalExclude = exclude = exclude == null ? new HashSet<>() : exclude;
                nodeIds = nodeIds == null ? new HashSet<>() : nodeIds;

                if (exclude.size() >= total) {
                    exec(c::sync); return null;
                }
                if (exclude.contains(leaderId)) return voteLeader(null, exclude, null);
                List<Node> sortedNodes = nodes.stream().sorted(Comparator.comparing(n -> n.id)).collect(Collectors.toList());
                if (sortedNodes.size() - exclude.size() < total) {
                    exec(c::sync); return null;
                }

                if (leaderId == null || leaderId.isEmpty()) {
                    // 依次找一个可用的leader节点开始发散vote, 直到所有节点都投了相同leader
                    for (Node n : sortedNodes) {
                        if (exclude.contains(n.id)) continue;
                        // 当前节点选出自己为leader去投票
                        if (c.getNodeId().equals(n.id)) {
                            leaderId = n.id; break;
                        }

                        c.http(n, "/vote").param("leaderId", n.id).param("exclude", String.join(",", exclude))
                                .exHandler((ex, h) -> finalExclude.add(n.id))
                                .debug().post();
                        if (!exclude.contains(n.id)) return this.leader;
                    }
                    if (leaderId == null || leaderId.isEmpty()) {
                        exec(c::sync);
                        return null;
                    }
                }

                if (sortedNodes.get(0).id.equals(leaderId)) { // 这轮投票的leader和当前的leader选择一样
                    if (nodeIds.size() >= total) { // 已投完，回知投票的节点
                        this.leader = sortedNodes.get(0);
                    }
                    else if (nodeIds.size() + 1 >= total) { // 已投完(当前已是最后一个节点)，回知所有节点结果
                        this.leader = sortedNodes.get(0);
                        nodeIds.add(c.getNodeId());
                        for (Node n : sortedNodes) {
                            if (n.id.equals(c.getNodeId())) continue;
                            c.http(n, "/vote").param("leaderId", leaderId).param("nodeIds", String.join(",", nodeIds)).debug().post();
                        }
                    }
                    else { // 没投完传递给其他节点继续投
                        nodeIds.add(c.getNodeId());
                        for (Node n : sortedNodes) {
                            if (nodeIds.contains(n.id) || exclude.contains(n.id)) continue;
                            c.http(n, "/vote").param("leaderId", leaderId).param("exclude", String.join(",", exclude))
                                    .param("nodeIds", String.join(",", nodeIds))
                                    .exHandler((ex, h) -> finalExclude.add(n.id))
                                    .debug().post();
                            if (!exclude.contains(n.id)) break;
                        }
                    }
                } else exec(c::sync);
            } finally {
                _lock.set(null);
            }
            return this.leader;
        }
        return null;
    }


    @Route(path = "set", method = "post")
    R<?> set(String key, String value, Integer _limit) throws Exception {
        if (key == null || key.isEmpty()) return fail("Param key required");
        if (value == null || value.isEmpty()) return fail("Param value required");
        return update(key, value, new AtomicInteger(), _limit == null ? getAttr("updateTryLimit", Integer.class, 3) : _limit);
    }


    /**
     * follower: 第一阶段提交
     */
    @Route(path = "commit1", method = "post")
    R<?> commit1(String leaderId, String txId, String key) {
        if (leaderId == null || leaderId.isEmpty()) return fail("Param leaderId required");
        if (txId == null || txId.isEmpty()) return fail("Param txId required");
        Node leader = this.leader;
        if (leader == null) return of("10", "No leader");
        if (!leaderId.equals(leader.id)) return of("11", "Leader not same. " + leaderId + ", " + leader.id);
        data.computeIfAbsent(key, Item::new).records.addFirst(new Record(leaderId, txId));
        return ok();
    }

    /**
     * follower: 第二阶段提交
     */
    @Route(path = "commit2", method = "post")
    R<?> commit2(String leaderId, String txId, String key, String value) {
        if (leaderId == null || leaderId.isEmpty()) return fail("Param leaderId required");
        if (txId == null || txId.isEmpty()) return fail("Param txId required");
        if (value == null || value.isEmpty()) return fail("Param value required");
        Item item = data.get(key);
        if (item == null) return fail("Not exist");
        for (Record r : item.records) {
            if (r.leaderId.equals(leaderId) && r.txId.equals(txId)) {
                r.value = value;
            }
        }
        if (item.records.size() > getAttr("recordMaxKeep", Integer.class, 10)) {
            item.records.pollLast();
        }
        return ok();
    }



    /**
     * 1. client
     * 2. leader
     * 3. follower
     */
    R<?> update(String key, String value, AtomicInteger _count, int limit) throws Exception {
        _count = _count == null ? new AtomicInteger() : _count;
        if (leader == null) { // leader不存在，则投出一个leader
            voteLeader(null, null, null);
            if (_count.getAndIncrement() > limit) return of("11", "No leader, after try vote leader");
            return update(key, value, _count, limit);
        }

        R resp = ok();
        Cluster c = xNet.cluster();
        if (c.getNodeId().equals(leader.id)) { // 当前节点是leader, 分发update
            if (_lock.compareAndSet(null, Status.Update)) {
                try {
                    // 1. 第一阶段: 分发update给所有follower
                    List<Node> nodes = c.nodeMap.get(c.getNodeName()).stream().filter(n -> !n.id.equals(c.getNodeId())).collect(Collectors.toList());
                    List<R> resps = nodes.stream().map(n -> xNet.exec.submit(() ->
                            JSON.parseObject(upgrade(c.http(n, "/commit1").param("key", key)
                                    .param("leaderId", c.getNodeId()).readTimeout(2000).debug()).post(), R.class)
                    )).map(f -> {
                        try {
                            return f.get();
                        } catch (Exception e) {
                            log.error("", e);
                        }
                        return null;
                    }).filter(Objects::nonNull).collect(Collectors.toList());
                    // 验证
                    for (R<?> res : resps) {
                        if (R.OK_CODE.equals(res.getCode())) continue;
                        if ("10".equals(res.getCode())) this.leader = null;
                        _count.incrementAndGet();
                        return update(key, value, _count, limit);
                    }

                    // 2. 第二阶段: 通知所有follower持久化update
                    nodes.forEach(node -> {
                        exec(() -> {
                            upgrade(c.http(node, "/commit2").param("key", key).param("value", value)
                                    .param("leaderId", c.getNodeId()).readTimeout(2000).debug()).post();
                        });
                    });
                    data.computeIfAbsent(key, Item::new).records.addFirst(new Record(value, c.getNodeId()));
                } finally {
                    _lock.set(null);
                }
            }
        }
        else { // 转给leader去控制update
            upgrade(c.http(leader, "/update").param("key", key).param("value", value).param("limit", limit)
                    .exHandler((ex, h) -> resp.setCode("14").setMsg("Delegate to leader error: " + ex.getMessage()))
                    .debug()).post();
        }
        return resp;
    }


    /**
     * 协议升级成 {@link #X_CP}
     */
    protected Httper upgrade(Httper httper) {
        return httper.header("X-Upgrade", X_CP);
    }


    protected Integer totalNode() {
        return getAttr("totalNode", Integer.class, null);
    }


    protected <T> T getAttr(String key, Class<T> type, T defaultValue) {
        Object v = getAttr(key);
        if (v == null) return defaultValue;
        return to(v, type);
    }


    protected class Item {
        final String key;
        final Deque<Record> records = new ConcurrentLinkedDeque<>();
        final Object lock = new Object();

        public Item(String key) {
            this.key = key;
        }
    }


    protected class Record {
        final String leaderId;
        final String txId;
        final long ct = System.currentTimeMillis();
        String value;

        public Record(String leaderId, String txId) {
            this.leaderId = leaderId;
            this.txId = txId;
        }
    }
}
