package cn.xnatural.xnet;


import cn.xnatural.xchain.Route;
import com.alibaba.fastjson2.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.time.Duration;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;

import static cn.xnatural.xchain.IMvc.to;
import static java.util.Collections.emptyList;

/**
 * 数据最终一致性服务
 */
@Route(path = "/", protocol = AP.X_AP)
public class AP extends XNetBase {
    protected static final Logger log = LoggerFactory.getLogger(AP.class);
    public static final String DATA_VERSION = "dataVersion";
    public static final String X_AP = "X-ap";
    protected final Map<String, DataVersion> dataVersion = new ConcurrentHashMap<>();


    public AP(XNet xNet) { super(xNet); }

    @Override
    protected void start() {
        super.start();
        xNet.http().chain().resolve(this); // 委托http协议转发请求
    }

    @Override
    public Object getAttr(String key) { return super.getAttr("ap." + key); }

    public <T> T getAttr(String key, Class<T> type, T defaultValue) {
        Object v = getAttr(key);
        if (v == null) return defaultValue;
        return to(v, type);
    }


    /**
     * 接收其他节点过来的数据更新
     * @param key 数据集key
     * @param dataKey 数据key
     * @param version 数据版本
     * @param data 数据本身
     */
    @Route(path = DATA_VERSION, method = "post")
    R<?> receive(String key, String dataKey, Long version, Object data) {
        if (version == null) return R.fail("Param version required");
        DataVersion dv = dataVersion.computeIfAbsent(key, s -> new DataVersion(key));
        Record r = dv.data.computeIfAbsent(dataKey, s -> new Record(dataKey));

        if (r.version == null || r.version < version) { // 比较新,需要通知
            r.version = version;
            r.data = data;
            xNet.http().handle(key, DATA_VERSION, ctx -> key + ":" + dataKey + ":" + version, (k, t) -> {
                if ("dataKey".equals(k) && String.class.isAssignableFrom(t)) return dataKey;
                if ("version".equals(k) && (Long.class.isAssignableFrom(t) || long.class.isAssignableFrom(t))) return version;
                if ("data".equals(k)) return t == null ? data : to(data, t);
                return null;
            });
        } else {
            // 当前节点比较新, 通知其他节点有更新的数据
            update(key, dataKey, r.version, r.data);
        }
        return R.ok();
    }


    /**
     * 获取一个key下的所以数据
     */
    @Route(path = "key/{key}", method = "get")
    public R<Map<String, Record>> key(String key) {
        DataVersion dv = dataVersion.get(key);
        return R.ok(dv == null ? null : dv.data);
    }


    /**
     * 获取一个key/dataKey下的记录
     */
    @Route(path = "key/{key}/{dataKey}", method = "get")
    public R<Record> dataKey(String key, String dataKey) {
        DataVersion dv = dataVersion.get(key);
        return R.ok(dv == null ? null : dv.data.get(dataKey));
    }


    /**
     * 更新同名应用的数据
     * @param key 数据集key
     * @param dataKey 数据key
     * @param version 数据版本
     */
    public void update(String key, String dataKey, long version) {
       update(key, dataKey, version, null);
    }

    /**
     * 更新同名应用的数据
     * @param key 数据集key
     * @param dataKey 数据key
     * @param version 数据版本
     * @param data 数据本身
     */
    public void update(String key, String dataKey, long version, Object data) {
        Cluster c = xNet.cluster();
        for (Node node : c.nodes(c.getNodeName()).getData()) {
            if (c.getNodeId().equals(node.id)) continue;
            new Runnable() {
                int count; // 执行次数统计
                @Override
                public void run() {
                    if (count++ > getAttr("updateMaxTry", Integer.class, 10)) {
                        log.error("Ap update up to max try. {}, {}, {}", key, dataKey, version);
                        return;
                    }
                    DataVersion dv = dataVersion.get(key);
                    Record r = dv == null ? null : dv.data.get(dataKey);
                    if (r != null && r.version > version) return; // 已经有其他节点吧数据更新了
                    // 节点已经不在了
                    if (Optional.ofNullable(c.nodes(c.getNodeName()).getData()).orElse(emptyList()).stream().noneMatch(n -> n.id.equals(node.id))) return;
                    try {
                        R<?> resp = JSONObject.parseObject(
                                upgrade(xNet.cluster().http(node, DATA_VERSION)
                                        .param("key", key)
                                        .param("dataKey", dataKey)
                                        .param("version", version)
                                        .param("data", data)
                                ).debug().post(),
                                R.class
                        );
                        if (!R.OK_CODE.equals(resp.getCode())) {
                            log.warn("Ap update result: " + resp);
                            xNet.sched(Duration.ofSeconds(10), this);
                        }
                    } catch (Exception ex) {
                        log.warn("Ap update error", ex);
                        xNet.sched(Duration.ofSeconds(20), this);
                    }
                }
            }.run();
        }
    }


    /**
     * 协议升级成 {@link #X_AP}
     */
    protected Httper upgrade(Httper httper) {
        return httper.header("X-Upgrade", X_AP);
    }

    protected class Record {
        public final String dataKey;
        public Long version;
        public Object data;

        public Record(String dataKey) {
            this.dataKey = dataKey;
        }

        public Record(String dataKey, Long version, Object data) {
            this.dataKey = dataKey;
            this.version = version;
            this.data = data;
        }
    }

    protected class DataVersion {
        final Map<String, Record> data = new ConcurrentHashMap<>();
        final String key;

        public DataVersion(String key) {
            this.key = key;
        }
    }
}
