package cn.xnatural.xchain;

import java.util.Collections;
import java.util.Set;

/**
 * <pre>
 * 路由处理器
 * {@link Route}
 * </pre>
 */
public abstract class RouteHandler<T extends Context> extends PathHandler<T> {
    /**
     * 支持的方法
     */
    public final Set<String> methods;
    /**
     * 支持接收的Content-Type
     */
    public final Set<String> consumes;
    /**
     * 支持响应的Content-Type
     */
    public final Set<String> produces;


    /**
     * 创建 {@link RouteHandler}
     * @param path {@link #path}
     */
    public RouteHandler(String path) {
        this(null, path);
    }

    /**
     * 创建 {@link RouteHandler}
     * @param protocol {@link #protocol}
     * @param path {@link #path}
     */
    public RouteHandler(String protocol, String path) {
        this(protocol, path, null, null, null);
    }

    /**
     * 创建 {@link RouteHandler}
     * @param protocol {@link #protocol}
     * @param path {@link #path}
     * @param methods {@link #methods}
     * @param consumes {@link #consumes}
     * @param produces {@link #produces}
     */
    public RouteHandler(String protocol, String path, Set<String> methods, Set<String> consumes, Set<String> produces) {
        super(protocol, path);
        this.methods = methods == null ? Collections.emptySet() : Collections.unmodifiableSet(methods);
        this.consumes = consumes == null ? Collections.emptySet() : Collections.unmodifiableSet(consumes);
        this.produces = produces == null ? Collections.emptySet() : Collections.unmodifiableSet(produces);
    }


    @Override
    public int compareTo(PathHandler<T> other) {
        int c = super.compareTo(other);
        if (c == 0 && other instanceof RouteHandler) {
            RouteHandler<T> r = (RouteHandler<T>) other;
            // 条件越多放前面
            return (methods.size() + consumes.size() + produces.size()) - (r.methods.size() + r.consumes.size() + r.produces.size());
        }
        return c;
    }

    /**
     *
     * @param ctx {@link Context}
     * @return
     * <li>0: 匹配 </li>
     * <li>-1: 不匹配: 请求路径片 少于 当前handler路径片 </li>
     * <li>-2: 不匹配: method </li>
     * <li>-3: 不匹配: consume </li>
     * <li>-4: 不匹配: produce </li>
     * <li>>0: 不匹配: 是路由的第几个分片, 不匹配 </li>
     */
    @Override
    public int match(T ctx) {
        int m = super.match(ctx);
        if (m == 0) m = methodMatch(ctx) ? 0 : -2;
        if (m == 0) m = consumeMatch(ctx) ? 0 : -3;
        if (m == 0) m = produceMatch(ctx) ? 0 : -4;
        return m;
    }


    /**
     * 方法匹配
     */
    protected boolean methodMatch(T ctx) {
        return methods.isEmpty() || methods.stream().anyMatch(m -> m.equalsIgnoreCase(ctx.method()));
    }


    /**
     * 支持接收的Content-Type匹配
     */
    protected boolean consumeMatch(T ctx) {
        String tType = ctx.contentType() == null ? null : ctx.contentType().split(";")[0];
        return tType == null || tType.isEmpty() || consumes.isEmpty() || consumes.stream().anyMatch(consume -> {
            String cType = consume.split(";")[0]; // application/json;charset=utf-8
            if (tType.equalsIgnoreCase(cType)) return true;
            if (cType.equals("*/*")) return true;
            // 匹配 image/* -> image/png
            if (cType.endsWith("/*")) {
                return cType.split("/")[0].equalsIgnoreCase(tType.split("/")[0]);
            }
            // 匹配 */json -> application/json
            if (cType.startsWith("*/")) {
                return cType.split("/")[1].equalsIgnoreCase(tType.split("/")[1]);
            }
            return false;
        });
    }


    /**
     * 支持响应的Content-Type匹配
     */
    protected boolean produceMatch(T ctx) {
        return produces.isEmpty() || produces.stream().anyMatch(produce -> {
            String cType = produce.split(",")[0];
            if (cType.equals("*/*")) return true;

            return ctx.accept() == null || ctx.accept().stream().anyMatch(accept -> { // */*;q=0.8
                accept = accept.split(";")[0];
                if (cType.equalsIgnoreCase(accept)) return true;
                if (accept.equals("*/*")) return true;
                // 匹配 image/* -> image/png
                if (cType.endsWith("/*")) {
                    return cType.split("/")[0].equalsIgnoreCase(accept.split("/")[0]);
                }
                // 匹配 */json -> application/json
                if (cType.startsWith("*/")) {
                    return cType.split("/")[1].equalsIgnoreCase(accept.split("/")[1]);
                }
                return false;
            });
        });
    }


    @Override
    public String toString() {
        return Route.class.getSimpleName() + "[" +
                (methods.isEmpty() ? "" : "(" + String.join(",", methods) + ")") +
                (consumes.isEmpty() ? "" : "(" + String.join(",", consumes) + ")") +
                (produces.isEmpty() ? "" : "(" + String.join(",", produces) + ")") +
                path + "]";
    }
}
