package cn.xnatural.xchain;

import java.util.*;
import java.util.stream.Collectors;

/**
 * 路径处理器
 */
public abstract class PathHandler<T extends Context> implements Handler<T>, Matchable<T>, Comparable<PathHandler<T>> {
    /**
     * {@link Route#protocol()}
     */
    public final String protocol;

    /**
     * {@link Route#path()}
     */
    public final String path;

    /**
     * {@link #path} 的分片
     */
    public final List<Piece> pieces;


    /**
     * 创建 {@link PathHandler}
     * @param path {@link #path}
     */
    public PathHandler(String path) {
        this("", path);
    }

    /**
     * 创建 {@link PathHandler}
     * @param protocol {@link #protocol}
     * @param path {@link #path}
     */
    public PathHandler(String protocol, String path) {
        if (path == null || path.isEmpty()) throw new IllegalArgumentException("path required");
        this.path = path.trim();
        String[] pieces = "/".equals(path) ? new String[]{"/"} : Handler.extract(path).split("/");
        ArrayList<Piece> result = new ArrayList<>(pieces.length);
        for (String piece : pieces) {
            result.add(new Piece(piece));
        }
        result.trimToSize();
        this.pieces = Collections.unmodifiableList(result);
        this.protocol = protocol == null ? "" : protocol.trim();
    }


    @Override
    public int compareTo(PathHandler<T> other) {
        if (other instanceof FilterHandler) return -1;
        return doCompare(other);
    }


    /**
     * <pre>
     * 当前分片 所计算的 优先级值 用于 Handler 排序匹配
     *  匹配规则:
     *    1. 分片个数 越大越先匹配
     *    2. 分片位置 越小越先匹配
     *    3. 单个分片匹配规则:
     *      3.1. 单 / 比较
     *      3.2. 分片是固定字符串 优先匹配
     *      3.3. 分片中包含模式匹配规则
     *          3.3.1 分片固定值部分个数 越大越先匹配. 例: a{var1}b 这个分片中 有 a, b 两个固定值
     *          3.3.2 分片变量个数 越大越先匹配. 例: a{var1}b{var2} 这个分片中有 var1, var2 两个变量
     *      3.4. 按字符串自然顺序比较(1.在preMismatch时有优势  2.每次的顺序都一样方便调试)
     * 分片位置(必须)
     *  比如这两个路径: /test/a{var1}/a{var2}, /test/{var1}/a{var2}b 应该先匹配前一个的中间piece, 再匹配后一个的最后那个piece
     *
     *  例: /test/{var1}/a{var2}b/c 优先于 /test/a{var1}/a{var2}
     *  </pre>
     */
    protected int doCompare(PathHandler<T> other) {
        // 规则: 1
        int c = pieces.size() - other.pieces.size();
        if (c > 0) return 1;
        if (c < 0) return -1;
        for (Iterator<Piece> it1 = pieces.iterator(), it2 = other.pieces.iterator(); it1.hasNext(); ) {
            Piece p1 = it1.next();
            Piece p2 = it2.next();
            // 规则: 3.1
            if ("/".equals(p1.piece) && "/".equals(p2.piece)) return 0;
            else if ("/".equals(p1.piece)) return -1;
            else if ("/".equals(p2.piece)) return 1;
            // 规则: 3.2
            if (p1.pattern == null && p2.pattern != null) return 1;
            else if (p1.pattern != null && p2.pattern == null) return -1;
            // 规则: 3.3
            else if (p1.pattern != null) {
                // 规则: 3.3.1
                int constCount1 = p1.parts.size() - p1.varNames.size();
                int constCount2 = p2.parts.size() - p2.varNames.size();
                if (constCount1 > constCount2) return 1;
                if (constCount1 < constCount2) return -1;
                // 规则: 3.3.2 例: a{var1}b{var2} 优先于 a{var1}b
                if (p1.varNames.size() > p2.varNames.size()) return 1;
                if (p1.varNames.size() < p2.varNames.size()) return -1;
            }
        }
        // 规则: 3.4 (1.在preMismatch时有优势  2.每次的顺序都一样方便调试)
        for (Iterator<Piece> it1 = pieces.iterator(), it2 = other.pieces.iterator(); it1.hasNext(); ) {
            Piece p1 = it1.next();
            Piece p2 = it2.next();
            int m = p1.piece.compareTo(p2.piece);
            if (m != 0) return -m;
        }
        return 0;
    }


    @Override
    public int match(T ctx) {
        return pathMatch(ctx.pieces, ctx.pathToken);
    }


    /**
     * 路径匹配
     * 路径变量 {name}
     * @return
     *  0: 匹配;
     *  -1: 不匹配: 请求路径片 少于 当前handler路径片;
     *  >0: 不匹配: 是第几个piece;
     */
    protected int pathMatch(List<String> pieces, Map<String, String> pathToken) {
        // 不匹配: 请求路径片 少于 当前handler路径片
        if (pieces.size() < this.pieces.size()) return -1;

        int i = 0;
        for (Iterator<Piece> it = this.pieces.iterator(); it.hasNext(); i++) {
            Piece p = it.next();
            boolean match;
            if (!it.hasNext() && pieces.size() > this.pieces.size()) {
                // 最后一个 并且 请求 路径片多于 匹配路径片, 则匹配后边的所有
                // 例: /js/{fName}  /js/lib/vue.js fName 为 lib/vue.js
                match = p.match(pieces.stream().skip(i).collect(Collectors.joining("/")), pathToken);
            }
            else {
                match = p.match(pieces.get(i), pathToken);
            }
            if (!match) {
                pathToken.clear();
                return i+1;
            }
        }
        return 0;
    }


    @Override
    public String toString() {
        return path;
    }
}
