package cn.xnatural.xchain;

import java.lang.reflect.Array;
import java.nio.charset.Charset;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.BiFunction;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Supplier;

import static cn.xnatural.xchain.IMvc.formatQueryStr;
import static cn.xnatural.xchain.IMvc.to;


/**
 * 路由处理上下文
 */
public class Context {
    protected final IMvc server;
    /**
     * 路径变量值映射
     */
    protected final Map<String, String> pathToken = new LinkedHashMap<>(7);
    /**
     * 路径查询参数 ? 后面那坨
     */
    protected Map<String, Object> _queryParams;
    /**
     * 消息的参数供应函数
     */
    public final BiFunction<String, Class, Object> paramProvider;
    /**
     * 处理id: 唯一
     */
    protected final Function<Context, String> idProvider;
    /**
     * 路由路径
     */
    public final String path;
    /**
     * 动态变化的 路径块, 用于路径匹配 /test/p1/p2 -> test,p1,p2
     * 去除 {@link Route#path()}} 剩下的分片
     */
    final LinkedList<String> pieces = new LinkedList<>();
    /**
     * 当前执行上下文属性集
     */
    protected final Map<String, Object> attrs = new ConcurrentHashMap<>();
    /**
     * 消息处理的开始时间
     */
    public final Date createTime = new Date();
    /**
     * 额外附加对象
     */
    protected Object[] attach;


    /**
     * 请求执行上下文
     * @param path 路径路由 例: /a/b?p1=1
     * @param paramProvider 参数
     * @param server {@link IMvc}
     */
    protected Context(String path, Function<Context, String> idProvider, BiFunction<String, Class, Object> paramProvider, IMvc server) {
        this.server = server;
        this.paramProvider = paramProvider;
        this.path = path;

        if ("/".equals(path)) this.pieces.add("/");
        else {
            for (String piece : Handler.extract(path).split("/")) {
                pieces.add(piece);
            }
        }
        this.idProvider = idProvider;
    }


    /**
     * 添加附加对象
     */
    protected Context attach(Object... attach) {
        this.attach = attach;
        return this;
    }


    /**
     * 附加对象
     */
    public Object[] attach() { return attach; }


    /**
     * 处理id 唯一
     */
    public String id() { return idProvider == null ? null : idProvider.apply(this); }


    /**
     * 协议
     * {@link Route#protocol()}
     */
    public String protocol() { return null; }

    /**
     * 请求url 的查询字符串 ? 后面那坨
     */
    public String getQueryStr() {
        int i = path.indexOf("?");
        return i == -1 ? null : path.substring(i + 1);
    }

    /**
     * 协议
     * {@link Route#method()}}
     */
    public String method() { return null; }

    /**
     * 请求内容类型
     * {@link Route#consume()}}
     */
    public String contentType() { return null; }

    /**
     * 响应内容类型
     * {@link Route#produce()}}}
     */
    public Set<String> accept() { return null; }


    /**
     * 路径后边的查询参数
     */
    public Map<String, Object> queryParams() {
        if (_queryParams == null) {
            synchronized (this) {
                if (_queryParams == null) {
                    int i = Context.this.path.indexOf("?"); // 取第一个问号的位置
                    if (i >= 0) {
                        if (!getQueryStr().isEmpty()) {
                            _queryParams = formatQueryStr(getQueryStr(), Charset.defaultCharset().toString());
                        }
                    }
                    if (_queryParams == null) _queryParams = Collections.emptyMap();
                }
            }
        }
        return _queryParams;
    }


    /**
     * 设置请求属性
     * @param key 属性key
     * @param value 属性值
     * @return {@link Context}
     */
    public Context setAttr(String key, Object value) { attrs.put(key, value); return this; }

    /**
     * 设置请求 懒计算 属性
     * @param key 属性key
     * @param supplier 属性值计算器
     * @return {@link Context}
     */
    public <T> Context setAttr(String key, Supplier<T> supplier) { attrs.put(key, supplier); return this; }


    /**
     * 获取请求属性
     * @param aName 属性名
     * @return 属性值
     */
    public <T> T getAttr(String aName) { return getAttr(aName, null); }

    /**
     * 获取请求属性
     * @param aName 属性名
     * @param aType 属性类型
     * @return 属性值
     */
    public <T> T getAttr(String aName, Class<T> aType) {
        Object v = attrs.get(aName);
        if (v instanceof Supplier) v = ((Supplier<?>) v).get();
        return to(v, aType);
    }


    /**
     * 同步响应
     * @param body 响应的对象体
     * @return 响应结果
     */
    public Object render(Object body) {
         return server.render(this, body);
    }

    /**
     * 异步响应
     * @param body 响应的对象体
     * @param afterFn 响应结束后回调
     */
    public void render(Object body, Consumer afterFn) {
        server.render(this, body, afterFn);
    }


    /**
     * 根据参数名查找
     * {@link #param(String, Class)}
     * @param pName 参数名
     * @return 参数值
     */
    public Object param(String pName) { return param(pName, null); }

    /**
     * 根据参数类型查找
     * {@link #param(String, Class)}
     * @param type 类型
     */
    public <T> T param(Class<T> type) { return param(null, type); }

    /**
     * 取请求参数值
     * @param pName 参数名
     * @param type 结果类型
     * @return 参数值
     */
    public <T> T param(String pName, Class<T> type) {
        if (type != null && Context.class.isAssignableFrom(type)) return (T) this;
        if (type != null && IMvc.class.isAssignableFrom(type)) return (T) server;
        Object v = pathToken.get(pName);
        if (v == null) v = queryParams().get(pName);
        if (v == null) v = paramProvider == null ? null : paramProvider.apply(pName, type);
        if (type == null) return (T) v;
        else if (v == null) return to(null, type);
        else if (type.isArray()) {
            List ls = new LinkedList();
            if (v instanceof List) {
                for (Object o : ((List) v)) { ls.add(to(o, type.getComponentType())); }
                return (T) ls.toArray((T[]) Array.newInstance(type.getComponentType(), ((List) v).size()));
            } else {
                ls.add(to(v, type.getComponentType()));
                return (T) ls.toArray((T[]) Array.newInstance(type.getComponentType(), 1));
            }
        } else return to(v, type);
    }
}
