package cn.xnatural.xchain;

import java.util.Collections;
import java.util.Set;

/**
 * <pre>
 * 路由处理器
 * {@link Route}
 * </pre>
 */
public abstract class RouteHandler<T extends Context> extends PathHandler<T> {
    /**
     * 支持的方法
     */
    public final Set<String> method;
    /**
     * 支持接收的Content-Type
     */
    public final Set<String> consume;
    /**
     * 支持响应的Content-Type
     */
    public final Set<String> produce;


    /**
     * 创建 {@link RouteHandler}
     * @param path {@link #path}
     */
    public RouteHandler(String path) {
        this(null, path);
    }

    /**
     * 创建 {@link RouteHandler}
     * @param protocol {@link #protocol}
     * @param path {@link #path}
     */
    public RouteHandler(String protocol, String path) {
        this(protocol, path, null, null, null);
    }

    /**
     * 创建 {@link RouteHandler}
     * @param protocol {@link #protocol}
     * @param path {@link #path}
     * @param method {@link #method}
     * @param consume {@link #consume}
     * @param produce {@link #produce}
     */
    public RouteHandler(String protocol, String path, Set<String> method, Set<String> consume, Set<String> produce) {
        super(protocol, path);
        this.method = method == null ? Collections.emptySet() : Collections.unmodifiableSet(method);
        this.consume = consume == null ? Collections.emptySet() : Collections.unmodifiableSet(consume);
        this.produce = produce == null ? Collections.emptySet() : Collections.unmodifiableSet(produce);
    }


    @Override
    public int compareTo(PathHandler<T> other) {
        int c = super.compareTo(other);
        if (c == 0 && other instanceof RouteHandler) {
            RouteHandler<T> r = (RouteHandler<T>) other;
            // 条件越多放前面
            return (method.size() + consume.size() + produce.size()) - (r.method.size() + r.consume.size() + r.produce.size());
        }
        return c;
    }

    /**
     *
     * @param ctx {@link Context}
     * @return
     * <li>0: 匹配 </li>
     * <li>-1: 不匹配: 请求路径片 少于 当前handler路径片 </li>
     * <li>-2: 不匹配: method </li>
     * <li>-3: 不匹配: consume </li>
     * <li>-4: 不匹配: produce </li>
     * <li>>0: 不匹配: 是路由的第几个分片, 不匹配 </li>
     */
    @Override
    public int match(T ctx) {
        int m = super.match(ctx);
        if (m == 0) m = methodMatch(ctx) ? 0 : -2;
        if (m == 0) m = consumeMatch(ctx) ? 0 : -3;
        if (m == 0) m = produceMatch(ctx) ? 0 : -4;
        return m;
    }


    /**
     * 方法匹配
     */
    protected boolean methodMatch(T ctx) {
        if (!method.isEmpty() && method.stream().noneMatch(m -> m.equalsIgnoreCase(ctx.method()))) {
            return false;
        }
        return true;
    }


    /**
     * 支持接收的Content-Type匹配
     */
    protected boolean consumeMatch(T ctx) {
        if (!consume.isEmpty() && consume.stream().noneMatch(consume -> {
            String cType = consume.split(";")[0];
            String tType = ctx.contentType() == null ? null : ctx.contentType().split(";")[0];

            if (cType.equalsIgnoreCase(tType)) return true;
            if (cType.equals("*/*")) return true;
            if (cType.endsWith("/*")) {
                if (tType == null) return true;
                // 匹配 image/* -> image/png
                return cType.split("/")[0].equalsIgnoreCase(tType.split("/")[0]);
            }
            if (cType.startsWith("*/")) {
                if (tType == null) return true;
                // 匹配 */json -> application/json
                return cType.split("/")[1].equalsIgnoreCase(tType.split("/")[1]);
            }

            return false;
        })) {
            return false;
        }
        return true;
    }


    /**
     * 支持响应的Content-Type匹配
     */
    protected boolean produceMatch(T ctx) {
        if (!produce.isEmpty() && produce.stream().noneMatch(produce -> {
            String cType = produce.split(",")[0];
            if (cType.equals("*/*")) return true;
            if (ctx.accept() != null && ctx.accept().stream().anyMatch(accept -> { // */*;q=0.8
                accept = accept.split(";")[0];
                if (cType.equalsIgnoreCase(accept)) return true;
                if (accept.equals("*/*")) return true;
                if (cType.endsWith("/*")) {
                    // 匹配 image/* -> image/png
                    return cType.split("/")[0].equalsIgnoreCase(accept.split("/")[0]);
                }
                if (cType.startsWith("*/")) {
                    // 匹配 */json -> application/json
                    return cType.split("/")[1].equalsIgnoreCase(accept.split("/")[1]);
                }
                return false;
            })) return true;

            return false;
        })) {
            return false;
        }
        return true;
    }


    @Override
    public String toString() {
        return Route.class.getSimpleName() + "[" +
                (method.isEmpty() ? "" : "(" + String.join(",", method) + ")") +
                (consume.isEmpty() ? "" : "(" + String.join(",", consume) + ")") +
                (produce.isEmpty() ? "" : "(" + String.join(",", produce) + ")") +
                path + "]";
    }
}
