package cn.xnatural.xchain;

import java.net.URLDecoder;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * 每个Path路径 中的分片
 */
class Piece {
    /**
     * {@link Route#path()} 以 / 分割的每个片段
     */
    final String piece;
    /**
     * piece 所对应的 正则 匹配模式 用于匹配 请求路径中的 分片
     */
    Pattern pattern;
    /**
     * 例: a{var}.html <- abcd.html : var=bcd
     * part: a, var, .html
     */
    final List<String> parts = new LinkedList<>();
    /**
     * 当前 piece 中对应的 变量
     */
    final List<String> varNames = new LinkedList<>();


    public Piece(String piece) {
        if (piece == null || piece.isEmpty()) throw new IllegalArgumentException("Param piece required");
        this.piece = piece;
        resolve();
    }


    /**
     * 解析分片 piece
     */
    protected void resolve() {
        if (piece.contains("{") && piece.contains("}")) {
            StringBuilder part = new StringBuilder();
            for (int j = 0; j < piece.length(); j++) {
                char c = piece.charAt(j);
                if ('{' == c) {
                    if (j > 0) {
                        parts.add(part.toString());
                        part = new StringBuilder();
                    }
                    continue;
                } else if ('}' == piece.charAt(j)) {
                    parts.add(null); varNames.add(part.toString());
                    part = new StringBuilder();
                    continue;
                }
                part.append(c);
            }
            if (part.length() > 0) parts.add(part.toString());
        }
        // TODO 验证piece字符串
        StringBuilder reg = new StringBuilder();
        for (int i = 0; i < parts.size(); i++) {
            String s = parts.get(i);
            if (s == null) { // 把 {name} 替换 成一个正则 分组
                reg.append("(.*)");
            } else {
                if (i == 0) reg.append("^").append(s);
                else if (i + 1 == parts.size()) reg.append(s).append("$");
                else reg.append(s);
            }
        }
        if (reg.length() > 0) {
            pattern = Pattern.compile(reg.toString());
        }
    }


    // TODO 匹配缓存
    // final Map<String, Map<String, String>> cache = new ConcurrentHashMap<>();
    /**
     * 匹配 并把 路径变量 设值
     * @param str 要匹配的字符
     * @param pathToken 路径变量容器
     * @return 是否匹配
     */
    boolean match(String str, Map<String, String> pathToken) {
        if (pattern == null) {
            if (Objects.equals(piece, str)) return true;
        } else {
            Matcher m = pattern.matcher(str);
            if (m.find()) {
                if (pathToken != null) {
                    for (int j = 0; j < m.groupCount(); j++) {
                        pathToken.put(varNames.get(j), URLDecoder.decode(m.group(j + 1)));
                    }
                }
                return true;
            }
        }
        return false;
    }


    /**
     * 路径是模糊的，还是明确的
     */
    boolean isFuzzy() { return pattern != null; }


    @Override
    public boolean equals(Object other) {
        return other instanceof Piece && Objects.equals(piece, ((Piece) other).piece);
    }
}
