package cn.xnatural.xchain;

import java.io.UnsupportedEncodingException;
import java.lang.annotation.Annotation;
import java.lang.reflect.Array;
import java.lang.reflect.Method;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.net.URI;
import java.net.URLDecoder;
import java.util.*;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.atomic.LongAdder;
import java.util.function.BiFunction;
import java.util.function.Consumer;
import java.util.function.Function;

/**
 * MVC执行服务接口
 */
public interface IMvc<T extends Context> {

    /**
     * 执行一个路由请求
     * @param path 路由路径
     * @param protocol 协议
     * @param idProvider 处理id提供函数
     * @param paramProvider 参数提供函数
     * @param attach 额外附加对象
     * @return false: 未找到可用路由处理器
     */
    default boolean handle(String path, String protocol, Function<T, String> idProvider, BiFunction<String, Class, Object> paramProvider, Object... attach) {
        return handle((T) new Context(path, (Function<Context, String>) idProvider, paramProvider, this) {
            @Override
            public String protocol() { return protocol; }
        }, attach);
    }


    /**
     * 执行一个路由请求
     * @param path 路由路径
     * @param idProvider 处理id提供函数
     * @param paramProvider 参数提供函数
     * @param attach 额外附加对象
     * @return false: 未找到可用路由处理器
     */
    default boolean handle(String path, Function<T, String> idProvider, BiFunction<String, Class, Object> paramProvider, Object... attach) {
        return handle(path, null, idProvider, paramProvider, attach);
    }


    /**
     * 执行一个路由请求
     * @param path 路由路径
     * @param paramProvider 参数提供函数
     * @param attach 额外附加对象
     * @return false: 未找到可用路由处理器
     */
    default boolean handle(String path, BiFunction<String, Class, Object> paramProvider, Object... attach) {
        return handle(path, null, null, paramProvider, attach);
    }


    /**
     * 执行一个路由请求
     * @param path 路由路径
     * @param protocol 协议
     * @param idProvider 处理id提供函数
     * @param param 请求参数
     * @param attach 额外附加对象
     * @return false: 未找到可用路由处理器
     */
    default boolean handle(String path, String protocol, Function<T, String> idProvider, Map<String, Object> param, Object... attach) {
        return handle(path, protocol, idProvider, new BiFunction<String, Class, Object>() {
            @Override
            public Object apply(String k, Class t) {
                return param == null ? null : to(param.get(k), t);
            }

            @Override
            public String toString() {
                return param == null ? null : param.toString();
            }
        }, attach);
    }

    /**
     * 执行一个路由请求
     * @param path 路由路径
     * @param idProvider 处理id提供函数
     * @param param 请求参数
     * @param attach 额外附加对象
     * @return false: 未找到可用路由处理器
     */
    default boolean handle(String path, Function<T, String> idProvider, Map<String, Object> param, Object... attach) {
        return handle(path, null, idProvider, param, attach);
    }


    /**
     * 执行一个路由请求
     * @param path 路由路径
     * @param protocol 协议
     * @param attach 额外附加对象
     * @return false: 未找到可用路由处理器
     */
    default boolean handle(String path, String protocol, Object... attach) {
        return handle(path, protocol, ctx -> null, (Map<String, Object>) null, attach);
    }


    /**
     * 执行一个路由请求
     * @param path 路由路径
     * @param param 请求参数
     * @param attach 额外附加对象
     * @return false: 未找到可用路由处理器
     */
    default boolean handle(String path, Map<String, Object> param, Object... attach) {
        return handle(path, (String) null, param, attach);
    }


    /**
     * 执行一个路由请求
     * @param path 路由路径
     * @param protocol 协议
     * @param attach 额外附加对象
     * @return false: 未找到可用路由处理器
     */
    default boolean handle(String path, String protocol, Map<String, Object> param, Object... attach) {
        return handle(path, protocol, ctx -> null, param, attach);
    }


    /**
     * 执行一个路由请求
     * @param path 路由路径
     * @param attach 额外附加对象
     * @return false: 未找到可用路由处理器
     */
    default boolean handle(String path, Object... attach) {
        return handle(path, (k, t) -> null, attach);
    }


    /**
     * 执行一个路由请求
     */
    default boolean handle(T ctx, Object... attach) {
        ctx.attach(attach);
        return chain().handle(ctx);
    }


    /**
     * 当前服务处理执行链
     */
    Chain<T> chain();


    /**
     * 当前mvc的名字标识
     * 用于多个相同类型服务打印日志区别
     */
    default String getName() { return ""; }


    /**
     * 渲染响应
     * @param ctx 处理上下文
     * @param body 响应结果对象
     * @return 返回
     */
    default Object render(T ctx, Object body) {
        CountDownLatch latch = new CountDownLatch(1);
        AtomicReference<Object> v = new AtomicReference<>();
        render(ctx, body, r -> {
            v.set(r);
            latch.countDown();
        });
        try {
            latch.await(getAttr("render.sync.wait", Long.class, 60L), TimeUnit.SECONDS);
        } catch (InterruptedException e) {
            throw new RuntimeException(e);
        }
        return v.get();
    }

    /**
     * 异步主动渲染响应
     * @param body 响应结果对象
     * @param afterFn 渲染结束后执行
     */
    default void render(Object body, Consumer<Object> afterFn) { render(null, body, afterFn); }

    /**
     * 异步渲染响应
     * @param ctx 处理上下文
     * @param body 响应结果对象
     * @param afterFn 渲染结束后执行
     */
    void render(T ctx, Object body, Consumer<Object> afterFn);


    /**
     * 异常处理
     * @param ctx 处理上下文
     * @param ex 异常
     */
    void errHandle(T ctx, Throwable ex);


    /**
     * 服务属性获取
     * @param key 属性key
     * @return 属性值
     */
    default Object getAttr(String key) { return null; }

    /**
     * 服务属性获取
     * @param key 属性key
     * @param type 值类型
     * @param defaultValue 默认值
     * @return 属性值
     * @param <T> 类型
     */
    default <T> T getAttr(String key, Class<T> type, T defaultValue) {
        Object v = getAttr(key);
        if (v == null) return defaultValue;
        return to(v, type);
    }


    /**
     * 每时间段处理个数
     * @param time 时间 例: 2023-01-02 13:22
     * @param count 个数
     */
    default void countByPattern(String time, LongAdder count) {
        Chain.log.info((getName() == null || getName().isEmpty() ? "" : getName() + " ") + "{} handle count {}", time, count);
    }


    // 和tiny的ServerTpl方法重复
//    default String getStr(String key, String defaultValue) {
//        return getAttr(key, String.class, defaultValue);
//    }
//
//    default Integer getInteger(String key, Integer defaultValue) {
//        return getAttr(key, Integer.class, defaultValue);
//    }
//
//    default Long getLong(String key, Long defaultValue) {
//        return getAttr(key, Long.class, defaultValue);
//    }
//
//    default Boolean getBoolean(String key, Boolean defaultValue) {
//        return getAttr(key, Boolean.class, defaultValue);
//    }
//
//    default Double getDouble(String key, Double defaultValue) {
//        return getAttr(key, Double.class, defaultValue);
//    }


    /**
     * <pre>
     * 格式化
     *  普通参数: a=1&b=2
     *  数组参数: ids[]=1&ids[]=2
     *  Map参数: data[a]=1&data[b]=2
     * </pre>
     */
    static Map<String, Object> formatQueryStr(String queryStr, String charset) {
        if (queryStr == null) return Collections.emptyMap();
        Map<String, Object> data = new LinkedHashMap<>();
        for (String s : queryStr.split("&")) {
            String[] arr = s.split("=");
            if (arr.length < 1) continue;
            String name = null;
            try {
                name = URLDecoder.decode(arr[0], charset);
                String subName = null;
                // 数组参数 ?a[]=1&a[]=2  或者 a=1&a=2
                if (name.endsWith("[]")) name = name.substring(0, name.length() - 2);
                // map参数 ?map[a]=1&map[b]=2
                else if (name.endsWith("]") && name.contains("[") && !name.startsWith("[")) {
                    subName = name.substring(name.indexOf("[") + 1, name.length() - 1);
                    name = name.substring(0, name.indexOf("["));
                }
                String value = arr.length > 1 ? URLDecoder.decode(arr[1], charset) : null;
                if (data.containsKey(name)) { // 证明name 有多个值
                    Object v = data.get(name);
                    if (v instanceof List) ((List) v).add(value);
                    else if (v instanceof Map) {
                        ((Map<String, Object>) v).put(subName, value);
                    }
                    else {
                        data.put(name, new LinkedList<>(Arrays.asList(v, value)));
                    }
                } else {
                    if (subName == null) data.put(name, value);
                    else {
                        Map<String, Object> d = new LinkedHashMap<>();
                        d.put(subName, value);
                        data.put(name, d);
                    }
                }
            } catch (UnsupportedEncodingException e) {
                throw new RuntimeException(e);
            }
        }
        return Collections.unmodifiableMap(data);
    }


    /**
     * 类型转换
     * @param v 值
     * @param type 转换的类型
     * @return 转换后的结果
     */
    static <R> R to(Object v, Class<R> type) {
        if (type == null) return (R) v;
        if (v == null) {
            if (boolean.class.equals(type)) return (R) Boolean.FALSE;
            if (short.class.equals(type)) return (R) Short.valueOf("0");
            if (byte.class.equals(type)) return (R) Byte.valueOf("0");
            if (int.class.equals(type)) return (R) Integer.valueOf(0);
            if (long.class.equals(type)) return (R) Long.valueOf(0);
            if (double.class.equals(type)) return (R) Double.valueOf(0);
            if (float.class.equals(type)) return (R) Float.valueOf(0);
            if (char.class.equals(type)) return (R) Character.valueOf('\u0000');
            return null;
        }
        else if (type.isAssignableFrom(v.getClass())) return (R) v;
        else if (String.class.equals(type)) return (R) v.toString();
        else if (Boolean.class.equals(type) || boolean.class.equals(type)) return (R) Boolean.valueOf(v.toString());
        else if (Short.class.equals(type) || short.class.equals(type)) return (R) Short.valueOf(v.toString());
        else if (Byte.class.equals(type) || byte.class.equals(type)) return (R) Byte.valueOf(v.toString());
        else if (Integer.class.equals(type) || int.class.equals(type)) return (R) Integer.valueOf(v.toString());
        else if (BigInteger.class.equals(type)) return (R) new BigInteger(v.toString());
        else if (Long.class.equals(type) || long.class.equals(type)) return (R) Long.valueOf(v.toString());
        else if (Double.class.equals(type) || double.class.equals(type)) return (R) Double.valueOf(v.toString());
        else if (Float.class.equals(type) || float.class.equals(type)) return (R) Float.valueOf(v.toString());
        else if (BigDecimal.class.equals(type)) return (R) new BigDecimal(v.toString());
        else if (AtomicInteger.class.equals(type)) return (R) new AtomicInteger(Integer.valueOf(v.toString()));
        else if (AtomicLong.class.equals(type)) return (R) new AtomicLong(Long.valueOf(v.toString()));
        else if (LongAdder.class.equals(type)) {
            LongAdder la = new LongAdder();
            la.add(Long.valueOf(v.toString()));
            return (R) la;
        }
        else if (URI.class.equals(type)) return (R) URI.create(v.toString());
        else if (type.isEnum()) return Arrays.stream(type.getEnumConstants()).filter((o) -> v.equals(((Enum) o).name())).findFirst().orElse(null);
        else if (v instanceof Annotation && type.isAssignableFrom(Map.class)) {
            return (R) new Function<Annotation, Map<String, Object>>() {
                @Override
                public Map<String, Object> apply(Annotation anno) {
                    Map<String, Object> attr = new HashMap<>();
                    Method[] ms = anno.annotationType().getDeclaredMethods();
                    for (Method m : ms) {
                        try {
                            Object v = m.invoke(anno);
                            if (v.getClass().isArray()) {
                                if (v.getClass().getComponentType().isAnnotation()) {
                                    List<Map<String, Object>> ls = new LinkedList<>(); attr.put(m.getName(), ls);
                                    for (Annotation a : ((Annotation[]) v)) {
                                        ls.add(apply(a));
                                    }
                                } else {
                                    List ls = new LinkedList<>(); attr.put(m.getName(), ls);
                                    for (int i = 0, len = Array.getLength(v); i < len; i++) {
                                        ls.add(Array.get(v, i));
                                    }
                                }
                            } else {
                                attr.put(m.getName(), v instanceof Annotation ? apply((Annotation) v) : v);
                            }
                        } catch (Exception e) {
                            Chain.log.warn("", e);
                        }
                    }
                    return attr;
                }
            }.apply((Annotation) v);
        }
        return (R) v;
    }
}
