package cn.xnatural.xchain;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.lang.reflect.*;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;


/**
 * MVC 树结构的执行链
 * <pre>
 * {
 * 	"": {
 * 		Filter[(0)/a], Route[/a/b/d], Route[/a/b/{c}], Route[/map], Route[/arr], Route[/optional], Route[(GET)/getmethod],
 *  }
 * 	"test": {
 * 		Route[/proto],
 *   }
 * }
 * </pre>
 * 线程安全
 */
public class Chain<T extends Context> {
    protected static final Logger log = LoggerFactory.getLogger(Chain.class);
    /**
     * 协议处理器节点链
     */
    protected final Map<String, Node> protocolMap = new ConcurrentHashMap<>();
    protected final IMvc<T> server;
    /**
     * 时间段统计器
     */
    protected final Lazier<Counter> _counter;


    public Chain(IMvc<T> server) {
        this.server = server;
        // 用Lazier是因为属性可延迟获取
        this._counter = new Lazier<>(() -> new Counter(server.getAttr("countByPattern", String.class, "MM-dd HH"), server::countByPattern));
    }


    /**
     * 执行链路节点
     */
    protected class Node implements Handler<T> {
        protected final PathHandler<T> handler;
        protected Node prev;
        protected Node next;
        /**
         * 节点匹配跳转
         * 匹配结果code{@link RouteHandler#match(Context)} -> 节点链
         */
        protected IntMap<Node> matchJump;


        protected Node(Node prev, PathHandler<T> handler, Node next) {
            if (handler == null) throw new NullPointerException();
            this.handler = handler;
            this.prev = prev;
            this.next = next;
        }

        public boolean handle(T ctx) throws Throwable {
            int m; // 匹配结果code
            if (handler instanceof FilterHandler) { // 执行Filter, 可执行多个Filter
                m = handler.match(ctx);
                if (log.isTraceEnabled()) {
                    log.trace("{}, {}: {}({})", logStr(ctx.protocol(), ctx.id(), ctx.method(), ctx.path), "filter " + (m == 0 ? "match" : "unmatch(" + m + ")"), handler.path, ((FilterHandler<T>) handler).order);
                }
                if (m == 0 && !handler.handle(ctx)) return false;
            }
            else if (handler instanceof RouteHandler) { // 只执行一个Path
                m = handler.match(ctx);
                if (log.isTraceEnabled()) {
                    log.trace("{}, {}: {}", logStr(ctx.protocol(), ctx.id(), ctx.method(), ctx.path), "route " + (m == 0 ? "match" : "unmatch(" + m + ")"), handler.path);
                }
                if (m == 0) {
                    if (!Chain.this.handle(handler, ctx)) return true;
                    ctx.pathToken.clear();
                }
            } else throw new RuntimeException(logStr(ctx.protocol(), ctx.id(), ctx.method(), ctx.path) + "unknown Handler type: " + handler.getClass().getName());
            // 优化方案 1. 缓存正则匹配结果 2. 缓存匹配节点链 3. 路由分组 4. 计算每个RouteHandler由于第一个Piece匹配但后面的不匹配，后面可能匹配的节点链, 5. 跳跃匹配
            Node next = matchJump == null ? this.next : matchJump.getOrDefault(m, this.next);
            return next != null && next.handle(ctx);
        }

        @Override
        public String toString() {
            return handler.toString();
        }
    }


    /**
     * ...-2,-1,0,1,2..
     * 针对的int为key的优化map
     * 用下标作key
     */
    protected class IntMap<E> {
        final ArrayList<E> z = new ArrayList<>(); // 正下标
        final ArrayList<E> f = new ArrayList<>(); // 负下标

        void put(int i, E e) {
            if (i >= 0) {
                while (z.size() <= i) z.add(null);
                z.set(i, e);
            } else {
                while (f.size() <= -i) f.add(null);
                f.set(-i, e);
            }
        }

        E get(int i) {
            if (i >= 0) return z.size() > i ? z.get(i) : null;
            else return f.size() > -i ? f.get(-i) : null;
        }

        E getOrDefault(int i, E ee) {
            E e = get(i);
            return e == null ? ee : e;
        }

        E remove(int i) {
            if (i >= 0) return z.size() > i ? (i + 1 == z.size() ? z.remove(i) : z.set(i, null)) : null;
            else {
                i = -i;
                return f.size() > i ? (i + 1 == f.size() ? f.remove(i) : f.set(i, null)) : null;
            }
        }

        boolean isEmpty() {
            return z.stream().allMatch(Objects::isNull) && f.stream().allMatch(Objects::isNull);
        }

        int min() { return -(f.size() - 1); }

        int max() { return z.size() - 1; }

        IntMap<E> freeze() {
            z.trimToSize(); f.trimToSize();
            return this;
        }

        @Override
        public String toString() {
            Map<Integer, E> m = new HashMap<>();
            for (int i = 0; i < f.size(); i++) {
                if (f.get(i) == null) continue;
                m.put(-i, f.get(i));
            }
            for (int i = 0; i < z.size(); i++) {
                if (z.get(i) == null) continue;
                m.put(i, z.get(i));
            }
            return m.isEmpty() ? "" : m.toString();
        }
    }


    /**
     * 执行此Chain
     * @param ctx {@link Context}
     * @return true: 已处理(匹配路由处理:非Filter处理)，false: 未处理
     */
    protected boolean handle(T ctx) {
        boolean handled = false; // 是否被处理了
        try {
            Node node = protocolMap.get(Objects.toString(ctx.protocol(), "").toLowerCase());
            if (node != null) {
                if (node.matchJump == null) preMatch(node);
                handled = node.handle(ctx);
            }
            // 未找到匹配
            if (!handled) handleNoMatch(ctx);
        } catch (Throwable ex) {
            server.errHandle(ctx, ex);
        }
        return handled;
    }


    /**
     * 执行某个{@link RouteHandler}
     */
    protected boolean handle(PathHandler<T> h, T ctx) throws Throwable {
        log.info("Handle: {}, {}", logStr(ctx.protocol(), ctx.id(), ctx.method(), ctx.path), ctx.paramProvider);
        _counter.get().increment();
        return h.handle(ctx);
    }


    /**
     * 处理未有匹配的处理器
     */
    protected void handleNoMatch(T ctx) {
        log.warn("{}, not found", logStr(ctx.protocol(), ctx.id(), ctx.method(), ctx.path));
    }


    /**
     * 注册 {@link PathHandler}
     * 按优先级{@link PathHandler#compareTo(PathHandler)}}的比较结果添加
     * @param handler {@link Handler}
     * @return {@link Chain}
     */
    public Chain<T> register(PathHandler<T> handler) {
        if (handler instanceof RouteHandler) {
            RouteHandler<T> r = (RouteHandler<T>) handler;
            log.info("Route: " + logStr(r.protocol, null, r.method.isEmpty() ? null : String.join(",", r.method), r.path));
        }
        final String protocol = Objects.toString(handler.protocol, "").toLowerCase();
        synchronized (protocolMap) {
            Node first = protocolMap.get(protocol);
            if (first == null) { // 创建头节点
                first = new Node(null, handler, null);
                protocolMap.put(protocol, first);
            } else {
                // 把新节点按优先级加入到链路中
                for (Node n = first; ;n = n.next) {
                    if (handler.compareTo(n.handler) > 0) {
                        if (n.prev == null) { // 替换头节点
                            first = new Node(null, handler, n);
                            n.prev = first;
                            protocolMap.put(protocol, first);
                        } else {
                            n.prev.next = new Node(n.prev, handler, n);
                            n.prev = n.prev.next;
                        }
                        break;
                    }
                    if (n.next == null) {
                        n.next = new Node(n, handler, null);
                        break;
                    }
                }
                first.matchJump = null; // 代表此链需要重新preMatch
            }
        }
        return this;
    }


    /**
     * <pre>
     * 计算节点的可跳跃节点
     * 节点匹配结果代码 {@link RouteHandler#match(Context)}
     * {@link Node#matchJump} 匹配结果代码 -> 跳跃的目的节点
     * </pre>
     */
    protected void preMatch(final Node first) {
        if (!server.getAttr("chainPreMatch", Boolean.class, true)) return;
        synchronized (protocolMap) {
            if (first.matchJump != null) return;
            Node last = null;
            f1: for (Node n = first; n != null; n = n.next) {
                if (n.next == null) { // 最后一个
                    n.matchJump = null; last = n; break;
                }
                // 1. 计算filter的跳跃节点
                if (n.handler instanceof FilterHandler) {
                    if ("/".equals(n.handler.pieces.get(0).piece)) continue;
                    for (int j = 0; j < n.handler.pieces.size(); j++) {
                        if (n.handler.pieces.get(j).isFuzzy()) continue f1;
                    }
                    n.matchJump = new IntMap<>();
                    // 多个filter路径一样，就只保留最后一个的matchJump
                    if (n.prev != null && n.prev.handler instanceof FilterHandler && Objects.equals(n.handler.path, n.prev.handler.path)) {
                        n.prev.matchJump = null;
                    }
                    // 查找filter匹配后，最近的匹配路由节点
                    int count = 0;
                    int jump = 1;
                    for (Node nn = n.next; nn != null && nn.handler instanceof RouteHandler; nn = nn.next, count++) {
                        if (n.handler.pathMatch(nn.handler.pieces.stream().map(p -> p.piece).collect(Collectors.toList()), null) == 0) {
                            if (count >= jump) n.matchJump.put(0, nn);
                            break;
                        }
                    }
                }
                // 2. 计算Route的跳跃节点
                else if (n.handler instanceof RouteHandler) {
                    n.matchJump = new IntMap<>();
                    // 不匹配原因code: -1
                    int count = 0;
                    int jump = 1; // 必须>=1, 没必要跳到下一个，要多跳点
                    for (Node nn = n.next; nn != null; nn = nn.next, count++) { // 查找下一个可能匹配的节点
                        if (n.handler.pieces.size() > nn.handler.pieces.size()) {
                            if (count >= jump) n.matchJump.put(-1, nn);
                            break;
                        }
                    }

                    // 不匹配原因code: >0
                    f1_2: for (int i = 0; i < n.handler.pieces.size(); i++) {
                        int ii = i + 1;
                        for (int j = 0; j < ii; j++) {
                            if (n.handler.pieces.get(j).isFuzzy()) break f1_2; // 路由: /a/{v1}/c 只计算到分片a
                        }

                        count = 0;
                        f1_2_1: for (Node nn = n.next; nn != null; nn = nn.next, count++) { // 查找下一个可能匹配的节点
                            if (nn.handler.pieces.size() < ii) break f1_2;
                            for (int j = 0; j < i; j++) {
                                if (!nn.handler.pieces.get(j).match(n.handler.pieces.get(j).piece, null)) {
                                    continue f1_2_1;
                                }
                            }
                            if ((!nn.handler.pieces.get(i).isFuzzy() && !nn.handler.pieces.get(i).match(n.handler.pieces.get(i).piece, null)) ||
                                    (nn.handler.pieces.get(i).isFuzzy() && nn.handler.pieces.get(i).match(n.handler.pieces.get(i).piece, null))) {
                                if (count >= jump) n.matchJump.put(ii, nn);
                                break;
                            }
                        }
                    }
                }
            }

            // 3. 保留必要的跳跃(连续多个相同的原因跳转节点，只保留第一个)
            for (Node n = last; n != null; n = n.prev) { // 从后往前
                if (n.matchJump == null || n.prev == null || n.prev.matchJump == null) continue;

                for (int i = n.matchJump.min(), max = n.matchJump.max(); i <= max; i++) {
                    if (n.matchJump.get(i) != null && n.matchJump.get(i) == n.prev.matchJump.get(i)) {
                        n.matchJump.remove(i);
                    }
                }

                if (n.matchJump.isEmpty()) n.matchJump = null;
                else n.matchJump.freeze();
            }
            if (first.matchJump == null) {
                first.matchJump = new IntMap<Node>().freeze(); // 节点链的第一个节点的matchJump用来判断是否已preMatch
            }
        }
    }


    /**
     * 指定路径和协议处理器
     * @param protocol {@link Route#protocol()}
     * @param path {@link Route#path()}
     * @param matcher 匹配函数, 可由运行时逻辑动态决定是否匹配
     * @param handler 处理器
     * @return {@link Chain}
     */
    public Chain<T> route(String protocol, String path, Matchable<T> matcher, Handler<T> handler) {
        if (path == null || path.isEmpty()) throw new IllegalArgumentException("Param path required");
        register(new RouteHandler<T>(protocol, path) {
            @Override
            public boolean handle(T ctx) throws Throwable {
                return handler.handle(ctx);
            }

            @Override
            public int match(T ctx) {
                int m = super.match(ctx);
                if (m == 0 && matcher != null) m = matcher.match(ctx);
                return m;
            }
        });
        return this;
    }

    /**
     * 指定路径处理器
     * @param path {@link Route#path()}
     * @param handler 处理器
     * @return {@link Chain}
     */
    public Chain<T> route(String path, Handler<T> handler) {
        return route(null, path, null, handler);
    }


    /**
     * 指定路径和协议处理器
     * @param protocol {@link Route#protocol()}
     * @param path {@link Route#path()}
     * @param handler 处理器
     * @return {@link Chain}
     */
    public Chain<T> route(String protocol, String path, Consumer<T> handler) {
        if (path == null || path.isEmpty()) throw new IllegalArgumentException("Param path required");
        register(new RouteHandler<T>(protocol, path) {
            @Override
            public boolean handle(T ctx) throws Throwable {
                handler.accept(ctx);
                return false;
            }
        });
        return this;
    }

    /**
     * 指定路径处理器
     * @param path {@link Route#path()}
     * @param handler 处理器
     * @return {@link Chain}
     */
    public Chain<T> route(String path, Consumer<T> handler) {
        return route(null, path, handler);
    }


    /**
     * 添加Filter
     * @param protocol {@link Filter#protocol()}
     * @param prefix {@link Filter#prefix()}
     * @param handler {@link Handler}
     * @return {@link Chain}
     */
    public Chain<T> filter(String protocol, String prefix, int order, Handler<T> handler) {
        if (prefix == null || prefix.isEmpty()) throw new IllegalArgumentException("Param prefix required");
        return register(new FilterHandler<T>(protocol, prefix, order) {
            @Override
            public boolean handle(T ctx) throws Throwable {
                return handler.handle(ctx);
            }
        });
    }


    /**
     * 从对象中解析 {@link Route} {@link Filter}
     */
    public Chain<T> resolve(Collection ctrls) {
        if (ctrls == null) return this;
        for (Object ctrl : ctrls) {
            resolve(ctrl);
        }
        return this;
    }

    /**
     * 从对象中解析 {@link Route} {@link Filter}
     */
    public Chain<T> resolve(Object... ctrls) {
        if (ctrls == null) return this;
        return resolve(Arrays.asList(ctrls));
    }

    /**
     * 从对象中解析 {@link Route} {@link Filter}
     */
    public Chain<T> resolve(Object ctrl) {
        if (ctrl == null) return this;
        for (Class c = ctrl.getClass(); c != null && !c.getName().startsWith("java."); c = c.getSuperclass()) {
            for (Method m : c.getDeclaredMethods()) {
                resolve(ctrl, m);
            }
        }
        return this;
    }

    /**
     * 解析路由,可扩展自定义注释
     * 子类重写添加其他解析
     */
    protected void resolve(Object ctrl, Method method) {
        resolveFilter(ctrl, method);
        resolveRoute(ctrl, method);
    }


    /**
     * 解析 {@link Filter}
     * @param ctrl 控制层对象
     * @param method 方法
     */
    protected void resolveFilter(Object ctrl, Method method) {
        Filter aFilter = method.getAnnotation(Filter.class);
        if (aFilter == null) return;
        Route aCtrl = ctrl.getClass().getAnnotation(Route.class);
        Set<String> paths = Stream.of(aCtrl == null ? new String[]{"/"} : aCtrl.path()).flatMap(p -> Stream.of(aFilter.prefix())
                .map(pp -> ("/" + p + "/" + pp).replaceAll("[/]+", "/")))
                .collect(Collectors.toCollection(LinkedHashSet::new));
        Set<String> protocols = Stream.of(aFilter.protocol(), aCtrl == null ? null : aCtrl.protocol())
                .filter(Objects::nonNull).flatMap(Arrays::stream)
                .collect(Collectors.toSet());
        Parameter[] ps = method.getParameters();
        method.setAccessible(true);
        boolean boolType = boolean.class.isAssignableFrom(method.getReturnType()) || Boolean.class.isAssignableFrom(method.getReturnType());
        for (String proto : (protocols.isEmpty() ? new HashSet<String>(Collections.singleton(null)) : protocols)) {
            for (String prefix : paths) {
                log.info("Filter: {}, {}#{}", logStr(proto, null, null, prefix), ctrl.getClass().getName(), method.getName());
                filter(proto, prefix, aFilter.order(), ctx -> { // 实际@Filter 方法 调用
                    try {
                        Object r = method.invoke(ctrl, Arrays.stream(ps).map((p) -> ctx.param(p.getName(), p.getType())).toArray());
                        if (r != null && boolType) {
                            return (boolean) r;
                        }
                    } catch (InvocationTargetException ex) {
                        throw ex.getCause();
                    }
                    return true; // 默认继续往下匹配执行
                });
            }
        }
    }

    /**
     * 解析 {@link Route}
     * @param ctrl 控制层对象
     * @param method 方法
     */
    protected void resolveRoute(Object ctrl, Method method) {
        Route aRoute = method.getAnnotation(Route.class);
        if (aRoute == null) return;
        Route aCtrl = ctrl.getClass().getAnnotation(Route.class);
        Set<String> paths = Stream.of(aCtrl == null ? new String[]{"/"} : aCtrl.path()).flatMap(p -> Stream.of(aRoute.path())
                .map(pp -> ("/" + p + "/" + pp).replaceAll("[/]+", "/")))
                .collect(Collectors.toCollection(LinkedHashSet::new));
        Set<String> methods = Stream.of(aRoute.method(), aCtrl == null ? null : aCtrl.method())
                .filter(Objects::nonNull).flatMap(Arrays::stream)
                .filter(Objects::nonNull).map(String::toUpperCase)
                .collect(Collectors.toSet());
        Set<String> protocols = Stream.of(aRoute.protocol(), aCtrl == null ? null : aCtrl.protocol())
                .filter(Objects::nonNull).flatMap(Arrays::stream)
                .collect(Collectors.toSet());
        Set<String> consumes = Stream.of(aRoute.consume(), aCtrl == null ? null : aCtrl.consume())
                .filter(Objects::nonNull).flatMap(Arrays::stream)
                .filter(Objects::nonNull)
                .collect(Collectors.toSet());
        Set<String> produces = Stream.of(aRoute.produce(), aCtrl == null ? null : aCtrl.produce())
                .filter(Objects::nonNull).flatMap(Arrays::stream)
                .filter(Objects::nonNull)
                .collect(Collectors.toSet());
        resolveRoute_preProcess(ctrl, method, paths, protocols, methods, consumes, produces);

        Parameter[] ps = method.getParameters();
        method.setAccessible(true);
        boolean isVoid = void.class.isAssignableFrom(method.getReturnType());

        for (String path : paths) {
            if (path == null || path.isEmpty()) {
                log.error(logStr(String.join(",", protocols), null,
                        String.join(",", methods), null) + "@Route path item required. {}#{}",
                        ctrl.getClass().getName(), method.getName());
                continue;
            }
            for (String protocol : (protocols.isEmpty() ? new HashSet<String>(Collections.singleton(null)) : protocols)) {
                register(routeHandler(protocol, path, methods, consumes, produces, ctx -> {
                    try {
                        Object result = method.invoke(ctrl, Arrays.stream(ps).map((p) -> {
                            if (Optional.class.equals(p.getType())) { // 支持 Optional 参数
                                Type[] ts = ((ParameterizedType) p.getParameterizedType()).getActualTypeArguments();
                                if (ts.length > 0) {
                                    return Optional.ofNullable(ctx.param(p.getName(), (Class) ts[0]));
                                }
                                return Optional.ofNullable(ctx.param(p.getName()));
                            } else {
                                return ctx.param(p.getName(), p.getType());
                            }
                        }).toArray());
                        if (!isVoid) {
                            log.debug("{}, invoke Handler '{}#{}', result: {}", logStr(ctx.protocol(), ctx.id(), ctx.method(), ctx.path),
                                    ctrl.getClass().getName(), method.getName(), result);
                            ctx.render(result, null);
                        }
                    } catch (InvocationTargetException ex) {
                        throw ex.getCause();
                    }
                    return !aRoute.exclusive() || (aCtrl != null && !aCtrl.exclusive());
                }));
            }
        }
    }


    protected void resolveRoute_preProcess(Object ctrl, Method method, Set<String> paths, Set<String> protocols, Set<String> methods, Set<String> consumes, Set<String> produces) {}



//    /**
//     * 生成doc
//     */
//    protected void doc(Method method, Set<String> paths, Set<String> protocols, Set<String> methods, Set<String> consumes, Set<String> produces) {
//        Operation aOp = method.getAnnotation(Operation.class);
//        if (aOp == null) return;
//        Route aRoute = method.getAnnotation(Route.class);
//        if (aRoute == null) return;
//        List<String> expose = Arrays.stream(aRoute.expose()).collect(Collectors.toCollection(LinkedList::new));
//        paths.forEach(path -> {
//            Map<String, Object> op = IMvc.to(aOp, Map.class);
//            new BiConsumer<Object, Runnable>() {
//                @Override
//                public void accept(Object o, Runnable removeFn) {
//                    if (o instanceof String && ((String) o).isEmpty()) removeFn.run();
//                    else if (o instanceof List) {
//                        for (Iterator<?> it = ((List<?>) o).iterator(); it.hasNext(); ) {
//                            accept(it.next(), it::remove);
//                        }
//                        if (((List<?>) o).isEmpty() && removeFn != null) removeFn.run();
//                    } else if (o instanceof Map) {
//                        for (Iterator<? extends Map.Entry<?, ?>> it = ((Map<?, ?>) o).entrySet().iterator(); it.hasNext(); ) {
//                            Map.Entry<?, ?> e = it.next();
//                            accept(e.getValue(), "responses".equals(e.getKey()) ? null : it::remove);
//                        }
//                        if (((Map<?, ?>) o).isEmpty() && removeFn != null) removeFn.run();
//                    }
//                }
//            }.accept(op, null);
//
//            op.put("_expose", expose);
//            List tags = ((List) op.get("tags"));
//            if (tags == null) op.put("tags", Collections.singleton("default"));
//            for (String m : methods) {
//                Map<String, Object> item = new HashMap<>();
//                item.put(m.toLowerCase(), op);
//                ((Map<String, Map<String, Object>>) openApi().get("paths")).put(path, item);
//            }
//        });
//    }


    /**
     * 创建一个 {@link RouteHandler}
     * 用于子类重写
     */
    protected RouteHandler<T> routeHandler(String protocol, String path, Set<String> method, Set<String> consume, Set<String> produce, Handler<T> handler) {
        return new RouteHandler<T>(protocol, path, method, consume, produce) {
            @Override
            public boolean handle(T ctx) throws Throwable {
                return handler.handle(ctx);
            }
        };
    }


    public String logStr(String protocol, String id, String method, String path) {
        StringBuilder sb = new StringBuilder();
        boolean f1 = server.getName() != null && !server.getName().isEmpty();
        boolean f2 = protocol != null && !protocol.isEmpty();
        boolean f3 = id != null && !id.isEmpty();
        boolean f4 = method != null && !method.isEmpty();
        boolean f5 = path != null && !path.isEmpty();
        if (f1) sb.append("(").append(server.getName()).append(")");
        if (f2) sb.append("(").append(protocol).append(")");
        if (f3) sb.append("(").append(id).append(")");
        if (f4 && f5) sb.append("(").append(method).append(") ").append(path);
        else if (f4) sb.append("(").append(method).append(")");
        else if (f5) sb.append(sb.length() > 0 ? " " : "").append(path);
        return sb.toString();
    }


    final Lazier<Map<String, Object>> _openApi = new Lazier<>(() -> {
        Map<String, Object> m = new ConcurrentHashMap<>();
        m.put("openapi", Chain.this.server.getAttr("openApi.version", String.class, "3.0.1"));
        Map<String, Object> info = new HashMap<>(); m.put("info", info);
        info.put("title", Chain.this.server.getAttr("openApi.info.title", String.class, "XChain Open Api"));
        info.put("version", Chain.this.server.getAttr("openApi.info.version", String.class, "1.0.0"));
        List<Map<String, Object>> tags = new LinkedList<>(); m.put("tags", tags);
        // List<Map<String, Object>> servers = new LinkedList<>(); m.put("servers", servers);
        Map<String, Object> components = new HashMap<>(); m.put("components", components);
        Map<String, Map<String, Object>> paths = new LinkedHashMap<>(); m.put("paths", paths);
        return m;
    });

    public Map<String, Object> openApi() {
        return _openApi.get();
    }


    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder("{");
        for (Map.Entry<String, Node> e : protocolMap.entrySet()) {
            sb.append("\n\t").append(e.getKey().isEmpty() ? "\"\"" : "\"" + e.getKey() + "\"").append(": {\n\t\t");
            for (Node n = e.getValue(); n != null; n = n.next) {
                if (n != e.getValue()) sb.append("\n\t\t");
                if (n.prev != null && (
                        (n.prev.handler instanceof FilterHandler && n.handler instanceof RouteHandler) ||
                        (n.handler.pieces.size() < n.prev.handler.pieces.size())
                )) sb.append("\n\t\t");
                sb.append(n).append(n.matchJump == null ? "" : n.matchJump).append(", ");
            }
            sb.append("\n\t}");
        }
        sb.append("\n}");
        return sb.toString();
    }
}
