package cn.xisoil.service;

import cn.xisoil.config.DynamicDataSource;
import cn.xisoil.config.LocalDataSource;
import cn.xisoil.data.DataSourceConfiguration;
import cn.xisoil.exception.RegisterException;
import cn.xisoil.holder.DataSourceHolder;
import cn.xisoil.utils.SpringContextUtils;
import com.zaxxer.hikari.HikariConfig;
import com.zaxxer.hikari.HikariDataSource;
import org.hibernate.cfg.Configuration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import javax.sql.DataSource;
import java.util.HashMap;
import java.util.List;

/**
 * 数据源注册，修改，删除等操作
 */
@Component
public class DataSourceRegistrationComponent {
    private final Logger logger = LoggerFactory.getLogger(this.getClass());
    @Autowired
    private SpringContextUtils springContextUtils;
    @Autowired
    private DynamicDataSource dynamicDataSource;
    @Autowired
    private LocalDataSource localDataSource;
    @Autowired
    private TableDDLService tableDDLService;

    public HikariDataSource registerDataSource(DataSourceConfiguration dataSourceConfiguration){
        if (springContextUtils.containsBean(dataSourceConfiguration.getKey())){
            logger.error("register error=====>数据源已存在");
            throw new RegisterException("数据源已存在");
        }
        HikariConfig config= new HikariConfig();
        BeanUtils.copyProperties(dataSourceConfiguration,config);
        HikariDataSource hikariDataSource=new HikariDataSource(config);
        springContextUtils.registerSingleton(dataSourceConfiguration.getKey(),hikariDataSource);//注册 hikar DataSource
        localDataSource.setDataSourceMap(dataSourceConfiguration.getKey(),hikariDataSource);
        if (dataSourceConfiguration.getDefault()){
            localDataSource.defaultTableBase(dataSourceConfiguration.getKey());
        }
        logger.info(String.format("register success=====>数据源: %s注册成功", dataSourceConfiguration.getKey()));
        this.refreshJpaDataSource();
        DataSourceHolder.setDataSource(dataSourceConfiguration.getKey());
        tableDDLService.updateTable();
        return hikariDataSource;
    }

    public void initDBData(DataSource dataSource, List<String> sqls){
        tableDDLService.initDBData(dataSource,sqls);
    }

    public void removeDataSource(String key){
        if (!springContextUtils.containsBean(key)){
            logger.warn("remove warn=====>已执行删除操作，但是数据源不存在");
        }
        else {
            HikariDataSource hikariDataSource=(HikariDataSource)springContextUtils.getBean(key);
            hikariDataSource.close();
            springContextUtils.destroySingleton(key);
            localDataSource.delDataSourceMap(key);
            logger.info("register success=====>数据源已移除");
            this.refreshJpaDataSource();
        }
    }

    public void clearDataSource(){
        logger.warn("remove warn=====>清除所有数据源将导致程序不可用");
        localDataSource.getDataSourceMap().forEach((k,v)->{
            ((HikariDataSource)v).close();
        });
        localDataSource.setDataSourceMap(new HashMap<>());
        logger.info("register success=====>数据源已清空");
        this.refreshJpaDataSource();
    }

    public void updateDataSource(DataSourceConfiguration dataSourceConfiguration){
        if (!springContextUtils.containsBean(dataSourceConfiguration.getKey())){
            logger.error("register error=====>数据源不存在");
            throw new RegisterException("数据源不存在");
        }
        this.removeDataSource(dataSourceConfiguration.getKey());
        this.registerDataSource(dataSourceConfiguration);
    }

    private void createDDl(){
        Configuration cfg = new Configuration().configure();
    }


    private void refreshJpaDataSource(){
        dynamicDataSource.setTargetDataSources(localDataSource.getDataSourceMap());
        dynamicDataSource.afterPropertiesSet();
    }

}
