package cn.xisoil.asp.filter;

import cn.xisoil.annotation.filter.Idempotent;
import cn.xisoil.data.enums.HTTPCODE;
import cn.xisoil.exception.NormalException;
import cn.xisoil.exception.ResponseException;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson2.JSONObject;
import jakarta.servlet.http.HttpServletRequest;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.RedisCallback;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Component;
import org.springframework.ui.Model;
import org.springframework.validation.BindingResult;

import java.io.Serializable;
import java.lang.reflect.Method;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.concurrent.TimeUnit;

/**
 * @Description 接口幂等验证拦截方法
 * @Author Vien
 * @CreateTime 2023-03-2023/3/23 09:00:47
 **/

@Component
@Aspect
public class IdempotentAspect {

    private static final String KEY_TEMPLATE = "IDEMPOTENT:";

    @Autowired
    private RedisTemplate<String, Serializable> redisTemplate;

    /**
     * 切点(自定义注解)
     */
    @Pointcut("@annotation(cn.xisoil.annotation.filter.Idempotent)")
    public void executeIdempotent(){

    }

    /**
     * 切点业务
     */
    @Around("executeIdempotent()")
    public Object around(ProceedingJoinPoint jPoint) throws Throwable {
        //获取当前方法信息
        Method method = ((MethodSignature)jPoint.getSignature()).getMethod();
        //获取注解
        Idempotent idempotent = method.getAnnotation(Idempotent.class);
        //生成Key
        String key=idempotent.key()+"_"+generate(method, jPoint.getArgs());
        if (Boolean.TRUE.equals(redisTemplate.hasKey(KEY_TEMPLATE + key))){
            throw new ResponseException(HTTPCODE.ACCEPTED,idempotent.message());
        }
        redisTemplate.opsForValue().setIfAbsent(KEY_TEMPLATE+key,idempotent.key(),idempotent.timeout(), TimeUnit.SECONDS);
        return jPoint.proceed();
    }



    public static String  generate(Method method,Object... args) {
        StringBuilder stringBuilder = new StringBuilder(method.toString());
        for (Object  arg : args) {
            stringBuilder.append(toString(arg));
        }
        //进行md5等长加密
        return md5(stringBuilder.toString());
    }

    /**
     * 使用jsonObject对数据进行toString,(保持数据一致性)
     * @param object
     * @return
     */
    public static String toString(Object obj){
        if( obj == null || obj instanceof BindingResult || obj instanceof HttpServletRequest || obj instanceof Model){
            return "-";
        }
        return JSONObject.toJSONString(obj);
    }

    /**
     * 对数据进行MD5等长加密
     * @param str
     * @return
     */
    public static String md5(String str){
        StringBuilder stringBuilder = new StringBuilder();
        try {
            //选择MD5作为加密方式
            MessageDigest mDigest = MessageDigest.getInstance("MD5");
            mDigest.update(str.getBytes());
            byte[] b =  mDigest.digest();
            int j = 0;
            for (int i = 0,max = b.length; i < max; i++) {
                j = b[i];
                if(j < 0 ){
                    i += 256;
                }else if(j < 16){
                    stringBuilder.append(0);
                }
                stringBuilder.append(Integer.toHexString(j));
            }
        } catch (NoSuchAlgorithmException e) {
            e.printStackTrace();
        }
        return stringBuilder.toString();
    }


}
