package cn.ximcloud.homekit.core.starter.core.service.impl;

import cn.ximcloud.homekit.core.starter.constants.HomeKitAccessoryTypeEnum;
import cn.ximcloud.homekit.core.starter.core.bootstrap.HomeKitBootStrapServer;
import cn.ximcloud.homekit.core.starter.core.service.HomeKitControlService;
import cn.ximcloud.homekit.core.starter.core.service.HomeKitService;
import cn.ximcloud.homekit.core.starter.entity.metadata.HomeKitAccessoryEntity;
import cn.ximcloud.homekit.core.starter.entity.metadata.HomeKitAccessoryMethodEntity;
import cn.ximcloud.homekit.core.starter.model.vo.*;
import cn.ximcloud.homekit.core.starter.repository.HomeKitAccessoryMethodRepository;
import cn.ximcloud.homekit.core.starter.repository.HomeKitAccessoryRepository;
import com.google.common.collect.Lists;
import io.github.hapjava.accessories.HomekitAccessory;
import io.github.hapjava.server.HomekitAuthInfo;
import io.github.hapjava.server.impl.crypto.HAPSetupCodeUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * @author usami-mizugi
 */
@Slf4j
@Service
public class HomeKitControlServiceImpl implements HomeKitControlService {

    private final HomeKitBootStrapServer homeKitBootStrapServer;

    private final HomeKitAccessoryRepository homeKitAccessoryRepository;

    private final HomeKitAccessoryMethodRepository homeKitAccessoryMethodRepository;

    private final HomeKitService homeKitService;

    public HomeKitControlServiceImpl(HomeKitAccessoryRepository homeKitAccessoryRepository,
                                     HomeKitBootStrapServer homeKitBootStrapServer,
                                     HomeKitAccessoryMethodRepository homeKitAccessoryMethodRepository,
                                     HomeKitService homeKitService) {
        this.homeKitAccessoryRepository = homeKitAccessoryRepository;
        this.homeKitBootStrapServer = homeKitBootStrapServer;
        this.homeKitAccessoryMethodRepository = homeKitAccessoryMethodRepository;
        this.homeKitService = homeKitService;
    }

    @Override
    public List<HomeKitAccessoryVo> getHomeKitList(HomeKitAccessoryTypeEnum type) {
        return Lists.newArrayList(Optional.ofNullable(type)
                .map(typeEnum -> homeKitAccessoryRepository.findByType(type))
                .orElse(homeKitAccessoryRepository.findAll()))
                .stream()
                .map(item -> {
                    HomeKitAccessoryVo homeKitAccessoryVo = new HomeKitAccessoryVo();
                    BeanUtils.copyProperties(item, homeKitAccessoryVo);
                    homeKitAccessoryVo.setClazz(item.getClazz().getName());
                    homeKitAccessoryVo.setType(item.getType().toString());
                    return homeKitAccessoryVo;
                }).collect(Collectors.toList());
    }

    @Override
    public HomeKitAccessoryVo getHomeKitAccessory(Integer id) {
        HomeKitAccessoryVo homeKitAccessoryVo = new HomeKitAccessoryVo();
        HomeKitAccessoryEntity homeKitAccessoryEntity = homeKitAccessoryRepository.findById(id).orElse(null);
        if (homeKitAccessoryEntity == null) {
            homeKitAccessoryVo.setId(-1);
            return homeKitAccessoryVo;
        }

        BeanUtils.copyProperties(homeKitAccessoryEntity, homeKitAccessoryVo);
        return homeKitAccessoryVo;
    }

    @Override
    public boolean homeKitSwitch(boolean flag) {
        if (homeKitBootStrapServer.isRunning()) {
            if (flag) {
                log.info("homeKit server is running,No need to run again");
                return false;
            } else {
                homeKitBootStrapServer.stopServer();
            }
        } else {
            if (flag) {
                homeKitBootStrapServer.startServer();
            } else {
                log.info("homeKit server is not running,No need to stop again");
            }
        }
        return true;
    }

    @Override
    public List<HomeKitAccessoryDetailVo> getCurrentLoadedHomeKitAccessories() {
        return homeKitBootStrapServer.getCurrentLoadedHomeKitAccessories()
                .stream()
                .map(accessory -> HomeKitAccessoryDetailVo.builder()
                        .id(accessory.getId())
                        .name(accessory.getName().join())
                        .serialNumber(accessory.getSerialNumber().join())
                        .model(accessory.getModel().join())
                        .manufacturer(accessory.getManufacturer().join())
                        .firmwareRevision(accessory.getFirmwareRevision().join()).build())
                .collect(Collectors.toList());
    }

    @Override
    public Object getHomeKitMethod(Integer id, boolean isDefault) {
        return homeKitAccessoryMethodRepository.findAllByHomeKitAccessory_Id(id).stream()
                .map(item -> {
                    HomeKitAccessoryMethodVo homeKitAccessoryMethodVo = new HomeKitAccessoryMethodVo();
                    BeanUtils.copyProperties(item, homeKitAccessoryMethodVo);
                    return homeKitAccessoryMethodVo;
                }).collect(Collectors.toList());
    }

    @Override
    public Object getHomeKitMethod(Integer homeKitId, Integer homeKitMethodId, Boolean isDefault) {
        HomeKitAccessoryMethodEntity homeKitAccessoryMethodEntity = homeKitAccessoryMethodRepository.findOneByHomeKitAccessory_IdAndId(homeKitId, homeKitMethodId).orElse(HomeKitAccessoryMethodEntity.builder().id(-1).build());
        HomeKitAccessoryMethodVo homeKitAccessoryMethodVo = new HomeKitAccessoryMethodVo();
        BeanUtils.copyProperties(homeKitAccessoryMethodEntity, homeKitAccessoryMethodVo);
        return homeKitAccessoryMethodVo;
    }

    @Override
    public HomeKitQrCodeAndPinCodeVo getQrCodeAndPinCode() {
        HomeKitQrCodeAndPinCodeVo homeKitQrCodeAndPinCodeVo = new HomeKitQrCodeAndPinCodeVo();
        HomekitAuthInfo authInfo = homeKitService.getAuthInfo();
        String pin = authInfo.getPin();
        homeKitQrCodeAndPinCodeVo.setQrCode(HAPSetupCodeUtils.getSetupURI(pin.replace("-", ""), authInfo.getSetupId(), 2));
        homeKitQrCodeAndPinCodeVo.setPinCode(pin);
        return homeKitQrCodeAndPinCodeVo;
    }

    @Override
    public HomeKitDashBoardVo getDashBoardData() {
        List<HomeKitAccessoryEntity> homeKitAccessoryMethodEntities = Lists.newArrayList(homeKitAccessoryRepository.findAll());
        return HomeKitDashBoardVo.builder()
                .supportAccessoryCount(homeKitAccessoryMethodEntities.stream().filter(item -> HomeKitAccessoryTypeEnum.ACCESSORY.equals(item.getType())).count())
                .supportOptionalCharacteristicCount(homeKitAccessoryMethodEntities.stream().filter(item -> HomeKitAccessoryTypeEnum.OPTIONAL_CHARACTERISTIC.equals(item.getType())).count())
                .runningAccessoryCount((long) homeKitBootStrapServer.getCurrentRunningHomeKitAccessories().size())
                .accessoryCount((long) homeKitBootStrapServer.getCurrentLoadedHomeKitAccessories().size())
                .userCount((long) homeKitService.getAuthInfo().getUserList().size())
                .build();
    }

    @Override
    public List<HomeKitAccessoryDetailVo> getHomeKitAccessoryList(String type) {
        List<HomekitAccessory> currentRunningHomeKitAccessories = homeKitBootStrapServer.getCurrentRunningHomeKitAccessories();
        List<Integer> runningHomeKitAccessories = currentRunningHomeKitAccessories.stream().map(HomekitAccessory::getId).collect(Collectors.toList());
        return Optional.ofNullable(type)
                .map(statusType -> {
                    if ("running".equals(statusType)) {
                        return currentRunningHomeKitAccessories;
                    } else {
                        return homeKitBootStrapServer.getCurrentLoadedHomeKitAccessories().stream()
                                .filter(item -> !runningHomeKitAccessories.contains(item.getId()))
                                .collect(Collectors.toList());
                    }
                })
                .orElseGet(homeKitBootStrapServer::getCurrentLoadedHomeKitAccessories)
                .stream()
                .map(item -> HomeKitAccessoryDetailVo.builder()
                        .id(item.getId())
                        .name(item.getName().join())
                        .serialNumber(item.getSerialNumber().join())
                        .model(item.getModel().join())
                        .manufacturer(item.getManufacturer().join())
                        .firmwareRevision(item.getFirmwareRevision().join())
                        .status(runningHomeKitAccessories.contains(item.getId()))
                        .build())
                .collect(Collectors.toList());
    }

    @Override
    public boolean accessoryOption(boolean optional, Integer id) {
        if (homeKitBootStrapServer.isRunning()) {
            List<Integer> loadedHomeKitAccessoriesId = homeKitBootStrapServer.getCurrentLoadedHomeKitAccessories().stream().map(HomekitAccessory::getId).collect(Collectors.toList());
            if (loadedHomeKitAccessoriesId.contains(id)) {
                List<Integer> currentRunningHomeKitAccessoriesId = homeKitBootStrapServer.getCurrentRunningHomeKitAccessories().stream().map(HomekitAccessory::getId).collect(Collectors.toList());
                if (optional) {
//                启动该HomeKit Accessory
                    if (currentRunningHomeKitAccessoriesId.contains(id)) {
                        return false;
                    } else {
                        homeKitBootStrapServer.startAccessory(id);
                        return true;
                    }
                } else {
//                停止该HomeKit Accessory
                    if (currentRunningHomeKitAccessoriesId.contains(id)) {
                        homeKitBootStrapServer.stopAccessory(id);
                        return true;
                    } else {
                        return false;
                    }
                }
            } else {
                return false;
            }
        } else {
            return false;
        }
    }

}
