package cn.ximcloud.homekit.core.starter.core.auth;

import cn.ximcloud.homekit.core.starter.entity.HomeKitAuthInfoEntity;
import cn.ximcloud.homekit.core.starter.entity.HomeKitUserEntity;
import cn.ximcloud.homekit.core.starter.model.vo.HomeKitUserVo;
import cn.ximcloud.homekit.core.starter.repository.HomeKitAuthInfoRepository;
import cn.ximcloud.homekit.core.starter.repository.HomeKitUserRepository;
import com.google.common.collect.Lists;
import io.github.hapjava.server.impl.HomekitServer;
import io.github.hapjava.server.impl.HomekitUtils;
import io.github.hapjava.server.impl.crypto.HAPSetupCodeUtils;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import java.math.BigInteger;
import java.security.InvalidAlgorithmParameterException;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;


/**
 * XIMCloudHomeKitAuthInfo
 *
 * @author W9004844
 */
@Slf4j
@AllArgsConstructor
public class BaseOnDataBaseHomeKitAuthInfo implements XIMCloudHomeKitAuthInfo {

    private final HomeKitUserRepository homeKitUserRepository;

    private final HomeKitAuthInfoEntity homeKitAuthInfo;

    public BaseOnDataBaseHomeKitAuthInfo(HomeKitAuthInfoRepository homeKitAuthInfoRepository, HomeKitUserRepository homeKitUserRepository) {
        this.homeKitUserRepository = homeKitUserRepository;
        this.homeKitAuthInfo =  homeKitAuthInfoRepository.findById(1)
                .orElseGet(() -> {
                    try {
                        return new HomeKitAuthInfoEntity(
                                HomekitServer.generatePin(),
                                HomekitServer.generateMac(),
                                HomekitServer.generateSalt(),
                                HomekitUtils.generateKey(),
                                HAPSetupCodeUtils.generateSetupId());
                    } catch (InvalidAlgorithmParameterException e) {
                        log.error("generate HomeKitAuthInfo error");
                        throw new IllegalArgumentException("generate HomeKitAuthInfo error", e);
                    }
                });
        homeKitAuthInfoRepository.update(homeKitAuthInfo);
        log.info("The PIN for pairing is {}", homeKitAuthInfo.getPin());
    }

    @Override
    public String getPin() {
        return homeKitAuthInfo.getPin();
    }

    @Override
    public String getMac() {
        return homeKitAuthInfo.getMac();
    }

    @Override
    public BigInteger getSalt() {
        return homeKitAuthInfo.getSalt();
    }

    @Override
    public byte[] getPrivateKey() {
        return homeKitAuthInfo.getPrivateKey();
    }

    /**
     * A setup Id used for pairing the device using QR Code. It can be any alphanumeric combination of
     * the length of 4.
     *
     * @return setup id
     */
    @Override
    public String getSetupId() {
        return homeKitAuthInfo.getSetupId();
    }

    @Override
    public void createUser(String userName, byte[] publicKey) {
        Optional<HomeKitUserEntity> homeKitUser = homeKitUserRepository.findByUserName(userName);
        if (!homeKitUser.isPresent()) {
            homeKitUserRepository.save(new HomeKitUserEntity(userName, publicKey));
            log.info("Added user for {}", userName);
        } else {
            log.warn("Already have an user for {}", userName);
        }
    }

    @Override
    public void removeUser(String userName) {
        Optional<HomeKitUserEntity> homeKitUser = homeKitUserRepository.findByUserName(userName);
        if (homeKitUser.isPresent()) {
            log.info("Removed user for {}", userName);
            homeKitUserRepository.delete(homeKitUser.get());
        } else {
            log.warn("the user {} not found", userName);
        }
    }

    @Override
    public byte[] getUserPublicKey(String userName) {
        return homeKitUserRepository.findByUserName(userName)
                .map(HomeKitUserEntity::getPublicKey)
                .orElseGet(() -> {
                    log.warn("the user {} not found", userName);
                    return new byte[0];
                });
    }

    /**
     * Called to check if a user has been created. The homekit accessory advertises whether the
     * accessory has already been paired. At this time, it's unclear whether multiple users can be
     * created, however it is known that advertising as unpaired will break in iOS 9. The default
     * value has been provided to maintain API compatibility for implementations targeting iOS 8.
     *
     * @return whether a user has been created and stored
     */
    @Override
    public boolean hasUser() {
        return homeKitUserRepository.count() > 0;
    }

    @Override
    public List<HomeKitUserVo> getUserList() {
        return Lists.newArrayList(homeKitUserRepository.findAll())
                .stream()
                .map(user -> HomeKitUserVo.builder()
                        .id(user.getId())
                        .name(user.getUserName())
                        .publicKey(user.getPublicKey())
                        .build())
                .collect(Collectors.toList());
    }

    @Override
    public boolean hasUser(String userName) {
        return getUserPublicKey(userName).length == 0;
    }
}
