package cn.xiaocuoben.chains.chain;


import cn.xiaocuoben.chains.exception.handler.ExceptionHandlerManager;
import lombok.extern.slf4j.Slf4j;

import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.stream.Collectors;

/**
 * @author Frank
 */
@Slf4j
public class ChainExecute implements Runnable {
    private static final ChainContext CHAIN_CONTEXT = new ChainContext();
    private Chain chain;
    private Lock taskLock = new ReentrantLock();

    public ChainExecute(Chain chain) {
        this.chain = chain;
    }

    @SuppressWarnings("all")
    @Override
    public void run() {
        List<ChainNode> nextChainNodeList = Collections.singletonList(this.chain.getRootChainNode());
        try {
            LocalDateTime startTime = LocalDateTime.now();
            while (nextChainNodeList != null && !nextChainNodeList.isEmpty()) {
                nextChainNodeList = nextChainNodeList.parallelStream()
                        .map(chainNode -> {
                            List<ChainNodeWrapper> cnList = new ArrayList<>();
                            //同任务可互相等待
                            Integer awaitMilliseconds = chainNode.awaitMillis();
                            if (awaitMilliseconds != null && awaitMilliseconds > 0) {
                                //请求前等待
                                try {
                                    TimeUnit.MILLISECONDS.sleep(awaitMilliseconds);
                                } catch (InterruptedException e) {
                                    log.warn(e.getMessage(), e);
                                }
                                this.taskLock.lock();
                            }
                            try {
                                List<ChainNodeWrapper> currentChainNodeList = chainNode.run(CHAIN_CONTEXT);
                                if (currentChainNodeList != null && !currentChainNodeList.isEmpty()) {
                                    currentChainNodeList = currentChainNodeList.parallelStream()
                                            .filter(Objects::nonNull)
                                            .collect(Collectors.toList());
                                    if (currentChainNodeList != null) {
                                        log.trace("after next,size of result:[{}]", currentChainNodeList.size());
                                        cnList.addAll(currentChainNodeList);
                                    } else {
                                        log.trace("after next,result:[{}]", currentChainNodeList);
                                    }
                                } else {
                                    log.trace("after next,result:[{}]", currentChainNodeList);
                                }
                            } catch (Exception e) {
                                ExceptionHandlerManager.get(e.getClass()).handle(e);
                            } finally {
                                if (awaitMilliseconds != null && awaitMilliseconds > 0) {
                                    this.taskLock.unlock();
                                }
                            }
                            return cnList;
                        })
                        .flatMap(chainNodeWrappers -> {
                            return chainNodeWrappers.stream();
                        })
                        .map(ChainNodeWrapper::getChainNode)
                        .collect(Collectors.toList());
            }
            long consumedMillis = ChronoUnit.MILLIS.between(startTime, LocalDateTime.now());
            log.info("[{}] end. Consumed {} milliseconds", this.chain, consumedMillis);
        } catch (Throwable e) {
            ExceptionHandlerManager.get(e.getClass()).handle(e);
        }
    }
}
