package cn.xiaocuoben.chains;

import cn.xiaocuoben.chains.chain.Chain;
import cn.xiaocuoben.chains.chain.ChainExecute;
import cn.xiaocuoben.chains.config.ChainsConfig;
import cn.xiaocuoben.chains.factory.ChainsFactory;
import cn.xiaocuoben.chains.factory.DefaultChainFactory;
import cn.xiaocuoben.chains.fetcher.Fetcher;
import lombok.extern.slf4j.Slf4j;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * @author Frank
 * @version 2017/7/29
 */
@Slf4j
public class Chains {
    private static AtomicBoolean HAS_SHUTDOWN_HOOK = new AtomicBoolean(Boolean.FALSE);
    private static Chains current;
    private Boolean isRunning = Boolean.FALSE;
    private CountDownLatch countDownLatch = new CountDownLatch(1);

    private ChainsFactory chainsFactory;
    private ScheduledExecutorService mainThreadPool;
    private Fetcher fetcher;

    private List<Chain> chainList = new ArrayList<>();

    static {
        current = new Chains(new DefaultChainFactory(new ChainsConfig()));
    }

    private Chains(ChainsFactory chainsFactory) {
        this.chainsFactory = chainsFactory;
        this.init();
        current = this;
    }

    public static Chains create(ChainsConfig chainsConfig) {
        return create(new DefaultChainFactory(chainsConfig));
    }

    public static Chains create(ChainsFactory chainsFactory) {
        return new Chains(chainsFactory);
    }

    public static Chains current() {
        return current;
    }


    private void init() {
        this.mainThreadPool = this.chainsFactory.createMainThreadPool();
        this.fetcher = this.chainsFactory.createFetcher();
        //注册自动关闭
        if (!HAS_SHUTDOWN_HOOK.get()) {
            HAS_SHUTDOWN_HOOK.set(Boolean.TRUE);
            Runtime.getRuntime().addShutdownHook(new Thread(() -> {
                this.countDownLatch.countDown();
                log.debug("Preparing to destroy the thread pool.");
                this.mainThreadPool.shutdown();
                log.debug("Main thread pools have been destroyed.");
                log.info("Bye.");
            }));
        }

    }

    public Chains addChain(Chain... chainArray) {
        this.chainList.addAll(Arrays.asList(chainArray));
        return this;
    }

    public Chains start() {
        this.chainList.forEach(chain -> {
            ChainExecute chainExecute = new ChainExecute(chain);
            if (chain.getPeriod() > 0) {
                this.mainThreadPool.scheduleAtFixedRate(chainExecute, chain.getInitialDelay(), chain.getPeriod(), chain.getUnit());
            } else {
                this.mainThreadPool.submit(chainExecute);
            }
        });
        this.isRunning = Boolean.TRUE;
        log.info("Chains Has Started.");
        return this;
    }

    public Chains await(long timeout, TimeUnit unit) {
        if (!this.isRunning) {
            throw new IllegalStateException("Invoke start() first! ");
        }
        try {
            this.countDownLatch.await(timeout, unit);
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
        return this;
    }

    public Chains await() {
        if (!this.isRunning) {
            throw new IllegalStateException("Invoke start() first! ");
        }
        try {
            this.countDownLatch.await();
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
        return this;
    }

    public Fetcher fetcher() {
        return this.fetcher;
    }
}
