package cn.xiaocuoben.chains;

import cn.xiaocuoben.chains.chain.Chain;
import cn.xiaocuoben.chains.chain.ChainExecute;
import cn.xiaocuoben.chains.chain.ChainNodeExecutor;
import cn.xiaocuoben.chains.chain.ScheduleChain;
import cn.xiaocuoben.chains.config.ChainsConfig;
import cn.xiaocuoben.chains.factory.ChainsFactory;
import cn.xiaocuoben.chains.fetcher.Fetcher;
import cn.xiaocuoben.chains.task.store.ChainNodeStore;
import lombok.extern.slf4j.Slf4j;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

/**
 * @author Frank
 * @version 2017/7/29
 */
@Slf4j
public class Chains {
    private static Chains         current        = new Chains();
    private        Boolean        isRunning      = Boolean.FALSE;
    private        CountDownLatch countDownLatch = new CountDownLatch(1);

    private ChainsFactory            chainsFactory;
    private ScheduledExecutorService mainThreadPool;
    private ChainNodeStore           chainNodeStore;
    private Fetcher                  fetcher;

    private List<Chain> chainList = new ArrayList<>();

    private Chains() {
        this.chainsFactory = new ChainsFactory(new ChainsConfig());
        this.init();
        current = this;
    }

    public static Chains create(ChainsConfig chainsConfig) {
        Chains chains = new Chains();
        chains.chainsFactory = new ChainsFactory(chainsConfig);
        return chains;
    }

    public static Chains current() {
        return current;
    }

    public ChainNodeStore chainNodeStore() {
        return this.chainNodeStore;
    }

    private void init() {
        this.mainThreadPool = this.chainsFactory.createMainThreadPool();
        this.chainNodeStore = this.chainsFactory.createChainNodeStore();
        this.fetcher = this.chainsFactory.createFetcher();
        //注册自动关闭
        Runtime.getRuntime().addShutdownHook(new Thread(() -> {
            this.countDownLatch.countDown();
            log.debug("Preparing to destroy the thread pool.");
            this.mainThreadPool.shutdown();
            log.info("Main thread pools have been destroyed.");
        }));
    }

    public Chains addChain(Chain... chainArray) {
        this.chainList.addAll(Arrays.asList(chainArray));
        return this;
    }

    public Chains start() {
        this.chainList.forEach(chain -> {
            ChainExecute chainExecute = new ChainExecute(chain);
            if (chain instanceof ScheduleChain) {
                ScheduleChain scheduleChain = (ScheduleChain) chain;
                this.mainThreadPool.scheduleAtFixedRate(chainExecute, scheduleChain.getInitialDelay(), scheduleChain.getPeriod(), scheduleChain.getUnit());
            } else {
                this.mainThreadPool.submit(chainExecute);
            }
        });
        this.isRunning = Boolean.TRUE;
        return this;
    }

    public Chains await(long timeout, TimeUnit unit) {
        if (!this.isRunning) {
            throw new IllegalStateException("Invoke start() first! ");
        }
        try {
            this.countDownLatch.await(timeout, unit);
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
        return this;
    }

    public ChainNodeExecutor chainNodeExecutor() {
        return this.chainsFactory.createChainNodeExecutor();
    }

    public Fetcher fetcher() {
        return this.fetcher;
    }
}
