package cn.xiaocuoben.chains.chain;

import cn.xiaocuoben.chains.exception.ChainsException;
import cn.xiaocuoben.chains.exception.handler.ExceptionHandlerManager;
import cn.xiaocuoben.chains.task.store.ChainNodeStore;
import lombok.extern.slf4j.Slf4j;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.RecursiveTask;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Lock;

/**
 * @author Frank
 * @version 2017/7/22
 */
@Slf4j
public class ChainNodeTask extends RecursiveTask<List<ChainNode>> {
    private transient List<ChainNode>   chainNodeList     = new ArrayList<>();
    private transient ChainContext      chainContext      = null;
    private transient ChainNodeExecutor chainNodeExecutor = null;
    private transient ChainNodeStore    chainNodeStore    = null;
    private           int               threshold         = 1;
    private final transient Lock taskLock;

    public ChainNodeTask(List<ChainNode> chainNodeList, ChainContext chainContext, ChainNodeExecutor chainNodeExecutor, Lock taskLock, ChainNodeStore chainNodeStore) {
        this.chainNodeList = chainNodeList;
        this.chainContext = chainContext;
        this.chainNodeExecutor = chainNodeExecutor;
        this.taskLock = taskLock;
        this.chainNodeStore = chainNodeStore;
    }

    @Override
    @SuppressWarnings("all")
    protected List<ChainNode> compute() {
        if (this.chainNodeList.size() > this.threshold) {//继续拆分
            int middleIdx;
            if ((this.chainNodeList.size() - 1) % this.threshold == 0) {
                middleIdx = this.chainNodeList.size() - 1 / this.threshold;
            } else {
                middleIdx = this.chainNodeList.size() - 1 / this.threshold - 1;
            }
            List<ChainNode> leftChainNodeList = this.chainNodeList.subList(0, middleIdx);
            ChainNodeTask   leftTask          = new ChainNodeTask(leftChainNodeList, this.chainContext, this.chainNodeExecutor, this.taskLock, this.chainNodeStore);

            List<ChainNode> rightChainNodeList = this.chainNodeList.subList(middleIdx, this.chainNodeList.size());
            ChainNodeTask   rightTask          = new ChainNodeTask(rightChainNodeList, this.chainContext, this.chainNodeExecutor, this.taskLock, this.chainNodeStore);

            leftTask.fork();
            rightTask.fork();

            if (leftTask.getException() != null) {
                return ExceptionHandlerManager.get(leftTask.getException().getClass()).handle(leftTask.getException());
            }
            List<ChainNode> leftResult = leftTask.join();

            if (rightTask.getException() != null) {
                return ExceptionHandlerManager.get(rightTask.getException().getClass()).handle(rightTask.getException());
            }
            List<ChainNode> rightResult = rightTask.join();

            List<ChainNode> totalResult = new ArrayList<>();
            totalResult.addAll(leftResult);
            totalResult.addAll(rightResult);
            return totalResult;
        } else {//执行
            if (this.chainNodeList.size() != 0) {
                List<ChainNode> cnList = new ArrayList<>();
                for (ChainNode chainNode : this.chainNodeList) {
                    this.chainNodeStore.store(chainNode);
                    //同任务可互相等待
                    Integer awaitMilliseconds = chainNode.awaitMillis();
                    if (awaitMilliseconds != null && awaitMilliseconds > 0) {
                        this.taskLock.lock();
                        try {
                            //保存任务
                            this.chainNodeExecutor.getChainNodeStore().store(chainNode);
                            //请求前等待
                            try {
                                TimeUnit.MILLISECONDS.sleep(awaitMilliseconds);
                            } catch (InterruptedException e) {
                                log.warn(e.getMessage(), e);
                            }
                            List<ChainNode> currentChainNodeList = chainNode.run(this.chainContext);
                            if (currentChainNodeList != null) {
                                log.trace("after execute,size of result:[{}]", currentChainNodeList.size());
                                cnList.addAll(currentChainNodeList);
                            } else {
                                log.trace("after execute,size of result:[{}]", currentChainNodeList);
                            }
                            this.chainNodeStore.remove(chainNode);
                        } finally {
                            this.taskLock.unlock();
                        }
                    }

                }
                return cnList;
            } else {
                throw new ChainsException("Task Split fatal");
            }
        }
    }
}
