package cn.xhteam.boot.http;

import java.io.*;
import java.net.Socket;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

/**
 * 发送响应
 * 响应对象
 * 该类的每一个实例表示HTTP协议规定的一个响应内容
 * 每个响应由三部分构成:
 * 状态行，响应头，响应正文
 */
public class HttpServletResponse {
    private Socket socket;

    //状态行的相关信息
    private int statusCode = 200; //状态代码
    private String statusReason = "ok"; //状态描述

    //响应头的相关信息
    private Map<String, Object> headers = new HashMap<>(); //响应头

    //响应正文的相关信息
    private File contentFile; //正文对应的实体文件

    //字节数组输出流
    private ByteArrayOutputStream baos;


    //构造方法 初始化socket
    public HttpServletResponse(Socket socket) {
        this.socket = socket;
    }

    /**
     * 将当前响应对象内容以标准的HTTP响应格式，发送给客户端(浏览器)
     */
    public void response() throws IOException {
        //发送前的准备工作：向响应头中添加响应正文长度
        sendBefore();

        //3.1发送状态行
        sendStatusLine();
        //3.2发送响应头
        sendHeaders();
        //3.3发送响应正文
        sendContent();
    }

    /**
     * 发送前的准备工作
     */
    private void sendBefore() {
        //判断是否有动态数据存在
        if (baos != null) {
            addHeader("Content-Length", baos.size() + "");
        }
    }

    /**
     * 发送状态行
     */
    private void sendStatusLine() throws IOException {
        println("HTTP/1.1" + " " + statusCode + " " + statusReason);
    }

    /**
     * 发送响应头
     */
    private void sendHeaders() throws IOException {
        //当发送响应头时，所有待发送的都应当都作为键值对存入了headers中
        Set<Map.Entry<String, Object>> entrySet = headers.entrySet();
        for (Map.Entry<String, Object> e : entrySet) {
            String key = e.getKey();
            Object value = e.getValue();
            println(key + ": " + value);
        }
        //单独发送一组回车+换行表示响应头部分发送完了!
        println("");
    }

    /**
     * 发送正文
     */
    private void sendContent() throws IOException {
        if (baos != null) { //存在动态数据
            byte[] data = baos.toByteArray();
            OutputStream out = socket.getOutputStream();
            out.write(data);
        } else if (contentFile != null) { //当正文不为空，我们才进行发送正文
            try (FileInputStream fis = new FileInputStream(contentFile);) {
                OutputStream outputStream = socket.getOutputStream();
                int len;
                byte[] data = new byte[1024 * 10];
                while ((len = fis.read(data)) != -1) {
                    outputStream.write(data, 0, len);
                }
            }
        }
    }

    /**
     * 路由重定向
     */
    public void sendRedirect(String path) {
        statusCode = 302;
        statusReason = "Moved Temporarily";
        addHeader("Location", path);
    }

    /**
     * 向浏览器发送一行字符串
     */
    private void println(String line) throws IOException {
        OutputStream outputStream = socket.getOutputStream();
        outputStream.write(line.getBytes(StandardCharsets.UTF_8));
        outputStream.write(13);//发送回车符
        outputStream.write(10);//发送换行符
    }

    /**
     * 添加响应头的参数
     */
    public void addHeader(String key, Object value) {
        headers.put(key, value);
    }

    public void setContentFile(File contentFile) throws IOException {
        this.contentFile = contentFile;
        //分析tomcat文件类型
        String contentType = Files.probeContentType(contentFile.toPath());
        //如果没有根据文件分析出文件类型，就不添加这个响应头了，由浏览器自己判定这个文件类型
        if (contentType != null) {
            addHeader("Content-Type", contentType);
        }
        addHeader("Content-Length", contentFile.length());
    }

    public OutputStream getOutputStream() {
        if (baos == null) {
            baos = new ByteArrayOutputStream();
        }
        return baos;
    }

    public PrintWriter getWriter() {
        PrintWriter pw = new PrintWriter(new BufferedWriter(new OutputStreamWriter(getOutputStream(), StandardCharsets.UTF_8)), true);
        return pw;
    }

    /**
     * 通过返回的字节输出流写出的字节最终会作为响应正文内容发送给客户端
     */
    public void setContentType(String mime) {
        addHeader("Content-Type", mime);
    }

    //Getter和Setter方法
    public int getStatusCode() {
        return statusCode;
    }

    public void setStatusCode(int statusCode) {
        this.statusCode = statusCode;
    }

    public String getStatusReason() {
        return statusReason;
    }

    public void setStatusReason(String statusReason) {
        this.statusReason = statusReason;
    }

    public File getContentFile() {
        return contentFile;
    }
}