package cn.xhteam.boot.http;


import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.Socket;
import java.net.URLDecoder;
import java.nio.charset.StandardCharsets;
import java.util.*;

/**
 * 解析请求
 * 请求对象
 * 该类的每一个实例用于表示浏览器发送过来的一个HTTP的请求内容
 * 每个请求HTTP协议要求由三部分构成:请求行，消息头，消息正文
 */
public class HttpServletRequest {
    public static final Logger logger = LogManager.getLogger();
    public static String POST = "POST";

    public static String GET = "GET";
    private Socket socket;

    //请求相关的信息
    //请求行的相关信息
    private String method; //请求方法
    private String uri; //抽象路径
    private String protocol; //协议版本

    private String requestURI; // 保存uri中的请求部分（？左侧的内容）
    private String queryString; // 保存uri中的参数部分（？右侧的内容）
    private Map<String, Object> parameters = new HashMap<>();//保存每一对参数\

    private String paramBody;

    //消息头的相关信息
    private Map<String, Object> headers = new HashMap<>();

    public HttpServletRequest(Socket socket) throws IOException, EmptyRequestException {
        this.socket = socket;
        //1.1解析请求行
        parseRequestLine();
        //1.2:解析消息头
        parseHeaders();
        //1.3:解析消息正文
        parseContent();
    }

    /**
     * 1.1解析请求行
     * 请求方法 抽象路径 协议版本
     */
    private void parseRequestLine() throws IOException, EmptyRequestException {
        String line = readLine();

        // 如果请求行为空字符串，则本次请求为空请求
        if (line.isEmpty()) {
            throw new EmptyRequestException();
        }

        String[] data = line.split("\\s");
        method = data[0];
        uri = data[1];
        protocol = data[2];
        logger.error(new StringBuilder("protocol=").append(protocol).append("\tmethod=" + method).append("\turi=" + uri).toString());
        //进一步解析uri
        parseURI();
    }

    //解析uri
    private void parseURI() {
        /**uri有两种情况，带参数和不带参数
         1: 不含参数
         例如：/index.html
         直接将uri的值赋给requestURI
         2.含参数
         例如：http://localhost:8088/regUser.html?username=shusheng&password=123456&nickname=jack&age=18
         将uri中"?"左侧的请求部分赋值给requestURI
         将uri中"?"右侧的参数部分赋值给queryString
         将参数部分首先按照&拆成一对对的参数，在按照=拆成参数名和参数值，参数名作为键，参数值作为值，存入parameters。
         */
        String[] data = uri.split("\\?");
        requestURI = data[0];
        if (data.length > 1) {
            queryString = data[1];
            parseParameters(queryString); //&&
        }
    }

    //解析参数，因为GET请求来自抽象路径的"?"右侧，而POST来自消息正文，所有格式一样，可以重用解析操作
    private void parseParameters(String line) {
        //username=%E5%B0%8F%E7%89%9B%E9%A9%AC&password=root&nickname=%E5%B0%8F%E7%89%9B%E9%A9%AC&age=15
        //先将参数转成中文
        try {
            line = URLDecoder.decode(line, "UTF-8");
        } catch (UnsupportedEncodingException e) {
            throw new RuntimeException(e);
        }
        //将参数按&分隔成一对对的
        String[] data = line.split("&");
        for (String e : data) {
            //将参数按=分隔成键值对
            String[] paras = e.split("=");
            Object val = paras.length > 1 ? paras[1] : "";
            parameters.put(paras[0], val);
        }
    }

    //1.2解析消息头
    private void parseHeaders() throws IOException {
        //1.2解析消息头
        while (true) {
            String line = readLine();
            if (line.isEmpty()) { //如果readLine返回空字符串，则是读取到了换行符和回车符
                break;
            }
            String[] data = line.split(":\\s");
            headers.put(data[0].toLowerCase(Locale.ROOT), data[1]);
        }
    }

    //解析消息正文
    private void parseContent() throws IOException {
        //判断本次请求方式是否为post请求
        if (POST.equalsIgnoreCase(method)) { //忽略大小写
            //根据消息头Content-Length来确定正文的字节数量以便进行读取
            Object contentLength = getHeader("Content-Length");
            if (contentLength != null) {//判断不为null的目的是确保有消息头Content-Length
                int length = Integer.parseInt(contentLength.toString());
                byte[] data = new byte[length];
                InputStream in = socket.getInputStream();
                in.read(data);
                //根据Content-Type来判断正文类型，并进行对应的处理
                Object contentType = getHeader("Content-Type");
                //分支判断不同的类型进行不同的处理
                if ("application/x-www-form-urlencoded".equals(contentType)) {//判断类型是否为form表单不带附件的数据
                    //该类型的正文就是一行字符串，就是原GET请求提交表单是抽象路径中"?"右侧的参数
                    String line = new String(data, StandardCharsets.ISO_8859_1);
                    parseParameters(line);
                }
                if ("application/json".equals(contentType)) {
                    String line = new String(data, StandardCharsets.UTF_8);
                    paramBody = line;
                }
                //扩展其他类型并进行对应的处理
            }
        }
    }

    /**
     * 按照回车符和换行符，获取请求信息
     *
     * @return
     * @throws IOException
     */
    private String readLine() throws IOException {
        //同一个Socket对象无论调用多少次getInputStream获取的都是同一个输入流
        //测试读取一行字符串（CRLF结尾）
        InputStream is = socket.getInputStream();
        int d;
        //cur表示本次读取的字符，pre表示上次读取的字符
        char cur = 'a', pre = 'a';
        StringBuilder builder = new StringBuilder();
        while ((d = is.read()) != -1) {
            cur = (char) d;
            //是否已经连续读取到了回车符+换行符
            if (pre == 13 && cur == 10) {
                break;
            }
            builder.append(cur);
            pre = cur;
        }
        return builder.toString().trim();
    }

    public Object getHeader(String name) {
        return headers.get(name.toLowerCase(Locale.ROOT));
    }

    public Object getParameter(String name) {
        return parameters.get(name);
    }

    public Object getParamBody() {
        return paramBody;
    }

    //Getter和Setter方法
    public String getMethod() {
        return method;
    }

    public void setMethod(String method) {
        this.method = method;
    }

    public String getUri() {
        return uri;
    }

    public void setUri(String url) {
        this.uri = url;
    }

    public String getProtocol() {
        return protocol;
    }

    public void setProtocol(String protocol) {
        this.protocol = protocol;
    }

    public String getRequestURI() {
        return requestURI;
    }

    public String getQueryString() {
        return queryString;
    }
}