package cn.wumoe.hime.module.socket

import cn.wumoe.hime.api.scripting.HimeContext
import cn.wumoe.hime.inter.Function
import cn.wumoe.hime.inter.Module
import cn.wumoe.hime.lexer.Num
import cn.wumoe.hime.lexer.Token
import cn.wumoe.hime.lexer.Word
import cn.wumoe.hime.toNum
import cn.wumoe.hime.toWord
import org.apache.commons.io.IOUtils
import java.net.ServerSocket
import java.net.Socket
import java.nio.charset.StandardCharsets

class SocketModule : Module("hime.socket") {
    override fun init(himeContext: HimeContext) {
        addFunction(HimeServer())               // socket-server
        addFunction(HimeSocket())               // socket
        addFunction(HimeSocketInputString())    // socket-input-string
        addFunction(HimeSocketPrintString())    // socket-writer-string
        addFunction(HimeSocketInputClose())     // socket-input-close
        addFunction(HimeSocketPrintClose())     // socket-writer-close
        addFunction(HimeSocketClose())          // socket-close
        addFunction(HimeShutdownOutput())       // socket-shutdown-writer
        addFunction(HimeShutdownInput())        // socket-shutdown-input
        addFunction(HimeSocketPrintFlush())     // socket-writer-flush
        addFunction(HimeSocketGetIp())          // socket-get-ip
        addFunction(HimeSocketGetPort())        // socket-get-port
    }

    class HimeSocketGetIp : Function("socket-get-ip") {
        override fun call(pars: Array<Token>): Token {
            return if (pars.isNotEmpty() && pars[0] is HimeSocketToken)
                (pars[0] as HimeSocketToken).socket.localAddress.hostAddress.toWord()
            else Word.NIL
        }
    }

    class HimeSocketGetPort : Function("socket-get-port") {
        override fun call(pars: Array<Token>): Token {
            return if (pars.isNotEmpty() && pars[0] is HimeSocketToken)
                (pars[0] as HimeSocketToken).socket.localPort.toNum()
            else Word.NIL
        }
    }

    class HimeSocketPrintFlush : Function("socket-writer-flush") {
        override fun call(pars: Array<Token>): Token {
            if (pars.isNotEmpty() && pars[0] is HimeSocketToken)
                (pars[0] as HimeSocketToken).printWriter.flush()
            return Word.NIL
        }
    }

    class HimeShutdownOutput : Function("socket-shutdown-writer") {
        override fun call(pars: Array<Token>): Token {
            if (pars.isNotEmpty() && pars[0] is HimeSocketToken) {
                val socketToken = pars[0] as HimeSocketToken
                socketToken.socket.shutdownOutput()
            }
            return Word.NIL
        }
    }

    class HimeShutdownInput : Function("socket-shutdown-input") {
        override fun call(pars: Array<Token>): Token {
            if (pars.isNotEmpty() && pars[0] is HimeSocketToken)
                (pars[0] as HimeSocketToken).socket.shutdownInput()
            return Word.NIL
        }
    }

    class HimeSocketPrintClose : Function("socket-writer-close") {
        override fun call(pars: Array<Token>): Token {
            if (pars.isNotEmpty() && pars[0] is HimeSocketToken)
                (pars[0] as HimeSocketToken).printWriter.close()
            return Word.NIL
        }
    }

    class HimeSocketInputClose : Function("socket-input-close") {
        override fun call(pars: Array<Token>): Token {
            if (pars.isNotEmpty() && pars[0] is HimeSocketToken)
                (pars[0] as HimeSocketToken).inputStream.close()
            return Word.NIL
        }
    }

    class HimeSocketClose : Function("socket-close") {
        override fun call(pars: Array<Token>): Token {
            if (pars.isNotEmpty() && pars[0] is HimeSocketToken)
                (pars[0] as HimeSocketToken).socket.close()
            return Word.NIL
        }
    }

    class HimeSocketInputString : Function("socket-writer-string") {
        override fun call(pars: Array<Token>): Token {
            return if (pars.isNotEmpty() && pars[0] is HimeSocketToken) {
                val socketToken = pars[0] as HimeSocketToken
                val inputStream = socketToken.inputStream
                val bytes = IOUtils.toByteArray(inputStream)
                return String(bytes, StandardCharsets.UTF_8).toWord()
            } else Word.NIL
        }
    }

    class HimeSocketPrintString : Function("socket-writer-string") {
        override fun call(pars: Array<Token>): Token {
            if (pars.size >= 2 && pars[0] is HimeSocketToken) {
                val socketToken = pars[0] as HimeSocketToken
                socketToken.printWriter.write(pars[1].toString())
            }
            return Word.NIL
        }
    }

    class HimeServer : Function("socket-server") {
        override fun call(pars: Array<Token>): Token {
            if (pars.size >= 2 && pars[0] is Num) {
                val port = (pars[0] as Num).value.toInt()
                ServerSocket(port).use { serverSocket ->
                    while (!analysis.isEnd) {
                        val socketToken = HimeSocketToken(serverSocket.accept())
                        val func = analysis.getFunction(pars[1].toString())
                        func.analysis = analysis
                        func.call(arrayOf<Token>(socketToken))
                    }
                    analysis.endToFalse()
                    serverSocket.close()
                    return Word.NIL
                }
            } else return Word.NIL
        }
    }

    class HimeSocket : Function("socket") {
        override fun call(pars: Array<Token>): Token {
            return if (pars[0] is Word) {
                val s = (pars[0] as Word).lexeme
                HimeSocketToken(Socket(s.substring(0, s.indexOf(":")), s.substring(s.indexOf(":") + 1).toInt()))
            } else Word.NIL
        }
    }
}